<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

require_once 'config.php';

if (!isLoggedIn() || !isAdmin()) {
    header('Location: index.php?error=' . urlencode('Unauthorized'));
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    // Determine redirect URL based on referer
    $referer = $_SERVER['HTTP_REFERER'] ?? 'index.php';
    $redirect_url = (strpos($referer, '/user/') !== false || strpos($referer, '/admin/') !== false) 
        ? $referer . '?error=' . urlencode('Method not allowed')
        : 'index.php?error=' . urlencode('Method not allowed');
    header('Location: ' . $redirect_url);
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    header('Location: index.php?task_error=' . urlencode('Database connection failed'));
    exit;
}

$user_id = getCurrentUserId();

$name = $_POST['name'] ?? '';
$description = $_POST['description'] ?? '';
$priority = $_POST['priority'] ?? 'medium';
$start_date = $_POST['start_date'] ?? date('Y-m-d');
$end_date = $_POST['end_date'] ?? null;
$start_time = $_POST['start_time'] ?? null;
$end_time = $_POST['end_time'] ?? null;
$task_type = $_POST['task_type'] ?? 'one-time';
$recurring_interval = null; // No longer used, kept for backward compatibility
$recurring_days = $_POST['recurring_days'] ?? [];
$skip_days = null; // Repurposed: now stores the days to repeat on
$task_points = intval($_POST['task_points'] ?? 1);
$quality_points = intval($_POST['quality_points'] ?? 10);
$lost_task_points = intval($_POST['lost_task_points'] ?? -1);
$notes = $_POST['notes'] ?? null;
$assigned_users = $_POST['assigned_users'] ?? [];

if (empty($name) || empty($description)) {
    header('Location: index.php?task_error=' . urlencode('Name and description are required'));
    exit;
}

if (empty($assigned_users)) {
    header('Location: index.php?task_error=' . urlencode('At least one user must be assigned'));
    exit;
}

// Handle recurring days - store in skip_days field (repurposed)
if ($task_type === 'recurring') {
    if (is_array($recurring_days) && !empty($recurring_days)) {
        // Validate that at least one day is selected
        $skip_days = implode(',', array_map('intval', $recurring_days));
    } else {
        header('Location: index.php?task_error=' . urlencode('At least one repeat day must be selected for recurring tasks'));
        exit;
    }
} else {
    $recurring_interval = null;
    $skip_days = null;
}

// Generate unique task number
$task_number = generateTaskNumber();

// Prepare values for bind_param (handle nulls properly)
$end_date_value = !empty($end_date) ? $end_date : null;
$start_time_value = !empty($start_time) ? $start_time : null;
$end_time_value = !empty($end_time) ? $end_time : null;
$recurring_interval_value = $recurring_interval;
$skip_days_value = $skip_days;
$notes_value = !empty($notes) ? $notes : null;

$stmt = $conn->prepare("
    INSERT INTO tasks (task_number, name, description, priority, start_date, end_date, start_time, end_time, task_type, 
                    recurring_interval, skip_days, task_points, quality_points, 
                    lost_task_points, notes, created_by, is_enabled)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1)
");

if (!$stmt) {
    error_log("Failed to prepare insert statement: " . $conn->error);
    header('Location: index.php?task_error=' . urlencode('Database error: ' . htmlspecialchars($conn->error)));
    exit;
}

$bind_result = $stmt->bind_param("sssssssssisiiisi", $task_number, $name, $description, $priority, $start_date, 
                 $end_date_value, $start_time_value, $end_time_value, $task_type, $recurring_interval_value, 
                 $skip_days_value, $task_points, $quality_points, $lost_task_points, $notes_value, $user_id);

if (!$bind_result) {
    error_log("Failed to bind parameters: " . $stmt->error);
    header('Location: index.php?task_error=' . urlencode('Database error: ' . htmlspecialchars($stmt->error)));
    exit;
}

if ($stmt->execute()) {
    $task_id = $conn->insert_id;
    
    // Assign users
    $assign_stmt = $conn->prepare("INSERT INTO task_assignments (task_id, user_id) VALUES (?, ?)");
    if (!$assign_stmt) {
        header('Location: index.php?task_error=' . urlencode('Failed to prepare assignment statement: ' . $conn->error));
        exit;
    }
    
    foreach ($assigned_users as $uid) {
        $uid = intval($uid);
        if ($uid > 0) {
            $assign_stmt->bind_param("ii", $task_id, $uid);
            if (!$assign_stmt->execute()) {
                error_log("Failed to assign task $task_id to user $uid: " . $assign_stmt->error);
                // Continue with other users even if one fails
            } else {
                // Create notification
                $notif_stmt = $conn->prepare("
                    INSERT INTO notifications (user_id, title, message, type)
                    VALUES (?, 'New Task Assigned', ?, 'info')
                ");
                if ($notif_stmt) {
                    $message = "You have been assigned a new task: " . $name;
                    $notif_stmt->bind_param("is", $uid, $message);
                    $notif_stmt->execute();
                    $notif_stmt->close();
                }
            }
        }
    }
    $assign_stmt->close();
    
    // Redirect without hash, let JavaScript handle the toast via URL parameter
    header('Location: index.php?success=' . urlencode('Task created and assigned successfully!') . '#admin-tasks');
    exit;
} else {
    header('Location: index.php?task_error=' . urlencode('Failed to create task: ' . $conn->error));
    exit;
}

$conn->close();
?>
