<?php
require_once '../config.php';

if (!isLoggedIn() || !isAdmin()) {
    header('Location: ../index.php');
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    die("Database connection failed. Please check your configuration.");
}

// Get period parameter - default to 'this_week'
$period = $_GET['period'] ?? 'this_week';
$date_from = null;
$date_to = date('Y-m-d');

// Calculate date ranges based on period
$today = date('Y-m-d');
$yesterday = date('Y-m-d', strtotime('-1 day'));

switch ($period) {
    case 'today':
        $date_from = $today;
        $date_to = $today;
        break;
    case 'yesterday':
        $date_from = $yesterday;
        $date_to = $yesterday;
        break;
    case 'this_week':
        // This week: Monday to Sunday
        $dayOfWeek = date('w', strtotime($today));
        $mondayOffset = ($dayOfWeek == 0) ? -6 : (1 - $dayOfWeek);
        $date_from = date('Y-m-d', strtotime($today . ' ' . $mondayOffset . ' days'));
        $date_to = $today;
        break;
    case 'last_week':
        // Last week: Monday to Sunday
        $dayOfWeek = date('w', strtotime($today));
        $mondayOffset = ($dayOfWeek == 0) ? -6 : (1 - $dayOfWeek);
        $lastMonday = date('Y-m-d', strtotime($today . ' ' . ($mondayOffset - 7) . ' days'));
        $date_from = $lastMonday;
        $date_to = date('Y-m-d', strtotime($lastMonday . ' +6 days'));
        break;
    case 'last_15_days':
        $date_from = date('Y-m-d', strtotime('-15 days'));
        $date_to = $today;
        break;
    case 'last_30_days':
        $date_from = date('Y-m-d', strtotime('-30 days'));
        $date_to = $today;
        break;
    case 'this_month':
        $date_from = date('Y-m-01');
        $date_to = $today;
        break;
    case 'last_month':
        $date_from = date('Y-m-01', strtotime('first day of last month'));
        $date_to = date('Y-m-t', strtotime('last month'));
        break;
    case 'custom':
        $date_from = $_GET['date_from'] ?? $today;
        $date_to = $_GET['date_to'] ?? $today;
        break;
    default:
        // Default to this week
        $dayOfWeek = date('w', strtotime($today));
        $mondayOffset = ($dayOfWeek == 0) ? -6 : (1 - $dayOfWeek);
        $date_from = date('Y-m-d', strtotime($today . ' ' . $mondayOffset . ' days'));
        $date_to = $today;
        $period = 'this_week';
}

// Overall statistics - count ALL assigned tasks in period, not just those with completions
// Get total assigned tasks in period
$total_tasks_stmt = $conn->prepare("
    SELECT COUNT(DISTINCT ta.task_id) as total_tasks
    FROM task_assignments ta
    JOIN tasks t ON ta.task_id = t.id
    WHERE t.start_date <= ? AND (t.end_date IS NULL OR t.end_date >= ?) AND t.is_enabled = 1
");
$total_tasks_stmt->bind_param("ss", $date_to, $date_from);
$total_tasks_stmt->execute();
$total_tasks_result = $total_tasks_stmt->get_result();
$total_tasks_row = $total_tasks_result->fetch_assoc();
$total_tasks = $total_tasks_row['total_tasks'] ?? 0;

// Get completed and failed tasks count from completions in period
$completion_stats_stmt = $conn->prepare("
    SELECT 
        COUNT(DISTINCT CASE WHEN tc.status = 'completed' THEN tc.task_id END) as completed_tasks,
        COUNT(DISTINCT CASE WHEN tc.status = 'failed' THEN tc.task_id END) as failed_tasks
    FROM task_completions tc
    WHERE tc.completion_date BETWEEN ? AND ?
");
$completion_stats_stmt->bind_param("ss", $date_from, $date_to);
$completion_stats_stmt->execute();
$completion_stats_result = $completion_stats_stmt->get_result();
$completion_stats = $completion_stats_result->fetch_assoc();

$completed_tasks = $completion_stats['completed_tasks'] ?? 0;
$failed_tasks = $completion_stats['failed_tasks'] ?? 0;

// Get earned points from completions in period
$earned_points_stmt = $conn->prepare("
    SELECT 
        COALESCE(SUM(CASE WHEN tc.status = 'completed' THEN tc.task_points_earned ELSE 0 END), 0) as total_task_points_earned,
        COALESCE(SUM(CASE WHEN tc.status = 'completed' THEN tc.quality_points_earned ELSE 0 END), 0) as total_quality_points_earned
    FROM task_completions tc
    WHERE tc.completion_date BETWEEN ? AND ?
");
$earned_points_stmt->bind_param("ss", $date_from, $date_to);
$earned_points_stmt->execute();
$earned_points_result = $earned_points_stmt->get_result();
$earned_points = $earned_points_result->fetch_assoc();

$total_task_points_earned = $earned_points['total_task_points_earned'] ?? 0;
$total_quality_points_earned = $earned_points['total_quality_points_earned'] ?? 0;

// Get lost points from failed completions in period
$lost_points_stmt = $conn->prepare("
    SELECT 
        COALESCE(SUM(CASE WHEN tc.status = 'failed' THEN ABS(tc.lost_task_points) ELSE 0 END), 0) as total_lost_task_points
    FROM task_completions tc
    WHERE tc.status = 'failed' AND tc.completion_date BETWEEN ? AND ?
");
$lost_points_stmt->bind_param("ss", $date_from, $date_to);
$lost_points_stmt->execute();
$lost_points_result = $lost_points_stmt->get_result();
$lost_points = $lost_points_result->fetch_assoc();
$total_lost_task_points = $lost_points['total_lost_task_points'] ?? 0;

// Get total possible points from all assigned tasks in period
$total_possible_stmt = $conn->prepare("
    SELECT 
        COALESCE(SUM(t.task_points), 0) as total_possible_task_points,
        COALESCE(SUM(t.quality_points), 0) as total_possible_quality_points
    FROM task_assignments ta
    JOIN tasks t ON ta.task_id = t.id
    WHERE t.start_date <= ? AND (t.end_date IS NULL OR t.end_date >= ?) AND t.is_enabled = 1
");
$total_possible_stmt->bind_param("ss", $date_to, $date_from);
$total_possible_stmt->execute();
$total_possible_result = $total_possible_stmt->get_result();
$total_possible = $total_possible_result->fetch_assoc();

$total_possible_task_points = $total_possible['total_possible_task_points'] ?? 0;
$total_possible_quality_points = $total_possible['total_possible_quality_points'] ?? 0;

// Calculate lost quality points for the period (from failed tasks' quality points)
$lost_quality_stmt = $conn->prepare("
    SELECT COALESCE(SUM(t.quality_points), 0) as total_lost_quality_points
    FROM task_completions tc
    JOIN tasks t ON tc.task_id = t.id
    WHERE tc.status = 'failed' AND tc.completion_date BETWEEN ? AND ? AND t.is_enabled = 1
");
$lost_quality_stmt->bind_param("ss", $date_from, $date_to);
$lost_quality_stmt->execute();
$lost_quality_result = $lost_quality_stmt->get_result();
$lost_quality_row = $lost_quality_result->fetch_assoc();
$total_lost_quality_points = $lost_quality_row['total_lost_quality_points'] ?? 0;

$avg_completion_rate = $total_tasks > 0 ? round(($completed_tasks / $total_tasks) * 100, 1) : 0;

// User performance with detailed points - calculate total possible points from assigned tasks
// First, get all assigned tasks for users in the period, then calculate totals
$performance_stmt = $conn->prepare("
    SELECT 
        u.id,
        u.username,
        u.full_name,
        COUNT(DISTINCT CASE WHEN tc.status = 'completed' THEN tc.id END) as tasks_completed,
        COUNT(DISTINCT CASE WHEN tc.status = 'failed' THEN tc.id END) as tasks_failed,
        COUNT(DISTINCT CASE WHEN ta.task_id IS NOT NULL AND (tc.id IS NULL OR tc.completion_date NOT BETWEEN ? AND ?) THEN ta.task_id END) as tasks_pending,
        COALESCE(SUM(CASE WHEN tc.status = 'completed' THEN tc.task_points_earned ELSE 0 END), 0) as task_points_earned,
        COALESCE(SUM(CASE WHEN tc.status = 'completed' THEN tc.quality_points_earned ELSE 0 END), 0) as quality_points_earned,
        COALESCE(SUM(CASE WHEN tc.status = 'failed' THEN ABS(tc.lost_task_points) ELSE 0 END), 0) as lost_task_points,
        -- Calculate lost quality points for this period (from failed tasks)
        COALESCE(SUM(CASE WHEN tc.status = 'failed' THEN t.quality_points ELSE 0 END), 0) as lost_quality_points,
        -- Total possible points from tasks assigned (tasks that overlap with the date range)
        COALESCE(SUM(CASE WHEN ta.task_id IS NOT NULL AND t.start_date <= ? AND (t.end_date IS NULL OR t.end_date >= ?) THEN t.task_points ELSE 0 END), 0) as total_possible_task_points,
        COALESCE(SUM(CASE WHEN ta.task_id IS NOT NULL AND t.start_date <= ? AND (t.end_date IS NULL OR t.end_date >= ?) THEN t.quality_points ELSE 0 END), 0) as total_possible_quality_points,
        -- Get total points from user_points table (lifetime totals)
        COALESCE(up.total_task_points, 0) as total_task_points,
        COALESCE(up.total_quality_points, 0) as total_quality_points,
        COALESCE(up.total_lost_task_points, 0) as total_lost_task_points,
        COALESCE(up.total_lost_quality_points, 0) as total_lost_quality_points
    FROM users u
    LEFT JOIN task_assignments ta ON u.id = ta.user_id
    LEFT JOIN tasks t ON ta.task_id = t.id AND t.is_enabled = 1
    LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
        AND ta.user_id = tc.user_id 
        AND tc.completion_date BETWEEN ? AND ?
    LEFT JOIN user_points up ON u.id = up.user_id
    WHERE u.role = 'user'
    GROUP BY u.id, u.username, u.full_name, up.total_task_points, up.total_quality_points, up.total_lost_task_points, up.total_lost_quality_points
    ORDER BY task_points_earned DESC, quality_points_earned DESC
");
$performance_stmt->bind_param("ssssssss", 
    $date_from, $date_to,  // pending check
    $date_to, $date_from,  // total possible task_points check (tasks that overlap period)
    $date_to, $date_from,  // total possible quality_points check
    $date_from, $date_to   // completion date range
);
$performance_stmt->execute();
$performance_result = $performance_stmt->get_result();

$performance = [];
while ($row = $performance_result->fetch_assoc()) {
    $total_assigned = $row['tasks_completed'] + $row['tasks_failed'] + $row['tasks_pending'];
    $row['completion_rate'] = $total_assigned > 0 ? round(($row['tasks_completed'] / $total_assigned) * 100, 1) : 0;
    $performance[] = $row;
}

function escapeHtml($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}
?>

<div class="page-header">
    <h1>Progress Reports</h1>
    <div class="reports-filter-container">
        <div class="ios-select-wrapper">
            <select id="reportPeriodSelect" class="ios-select">
                <option value="today" <?php echo $period === 'today' ? 'selected' : ''; ?>>Today</option>
                <option value="yesterday" <?php echo $period === 'yesterday' ? 'selected' : ''; ?>>Yesterday</option>
                <option value="this_week" <?php echo $period === 'this_week' ? 'selected' : ''; ?>>This Week</option>
                <option value="last_week" <?php echo $period === 'last_week' ? 'selected' : ''; ?>>Last Week</option>
                <option value="last_15_days" <?php echo $period === 'last_15_days' ? 'selected' : ''; ?>>Last 15 Days</option>
                <option value="last_30_days" <?php echo $period === 'last_30_days' ? 'selected' : ''; ?>>Last 30 Days</option>
                <option value="this_month" <?php echo $period === 'this_month' ? 'selected' : ''; ?>>This Month</option>
                <option value="last_month" <?php echo $period === 'last_month' ? 'selected' : ''; ?>>Last Month</option>
                <option value="custom">Pick Date</option>
            </select>
        </div>
        <div id="customDateRange" class="custom-date-range" style="display: <?php echo $period === 'custom' ? 'flex' : 'none'; ?>;">
            <input type="date" id="dateFrom" class="ios-date-input" value="<?php echo $date_from; ?>" />
            <span class="date-separator">to</span>
            <input type="date" id="dateTo" class="ios-date-input" value="<?php echo $date_to; ?>" />
        </div>
    </div>
</div>

<div class="reports-content">
    <div class="report-card">
        <h3>Overall Statistics</h3>
        <table class="report-table">
            <thead>
                <tr>
                    <th>Metric</th>
                    <th>Value</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Total Tasks</td>
                    <td><?php echo number_format($total_tasks); ?></td>
                </tr>
                <tr>
                    <td>Completed Tasks</td>
                    <td><?php echo number_format($completed_tasks); ?></td>
                </tr>
                <tr>
                    <td>Failed Tasks</td>
                    <td><?php echo number_format($failed_tasks); ?></td>
                </tr>
                <tr>
                    <td>Average Completion Rate</td>
                    <td><?php echo $avg_completion_rate; ?>%</td>
                </tr>
                <tr>
                    <td>Total Task Points Earned</td>
                    <td><?php echo number_format($total_task_points_earned); ?>/<?php echo number_format($total_possible_task_points); ?></td>
                </tr>
                <tr>
                    <td>Total Quality Points Earned</td>
                    <td><?php echo number_format($total_quality_points_earned); ?>/<?php echo number_format($total_possible_quality_points); ?></td>
                </tr>
                <tr>
                    <td>Total Lost Task Points</td>
                    <td class="negative-value"><?php echo number_format($total_lost_task_points); ?>/<?php echo number_format($total_possible_task_points); ?></td>
                </tr>
                <tr>
                    <td>Total Lost Quality Points</td>
                    <td class="negative-value"><?php echo number_format($total_lost_quality_points); ?>/<?php echo number_format($total_possible_quality_points); ?></td>
                </tr>
            </tbody>
        </table>
    </div>
    
    <div class="report-card">
        <h3>User Performance Report</h3>
        <div class="table-responsive">
            <table class="report-table">
                <thead>
                    <tr>
                        <th>User</th>
                        <th>Completed</th>
                        <th>Failed</th>
                        <th>Pending</th>
                        <th>Completion Rate</th>
                        <th>Task Points</th>
                        <th>Quality Points</th>
                        <th>Lost Task Points</th>
                        <th>Lost Quality Points</th>
                        <th>Total Task Points</th>
                        <th>Total Quality Points</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($performance)): ?>
                        <tr>
                            <td colspan="11" style="text-align: center; padding: 20px;">No performance data available</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($performance as $user): ?>
                            <tr>
                                <td><strong><?php echo escapeHtml($user['full_name'] ?: $user['username']); ?></strong></td>
                                <td><?php echo number_format($user['tasks_completed']); ?></td>
                                <td><?php echo number_format($user['tasks_failed']); ?></td>
                                <td><?php echo number_format($user['tasks_pending']); ?></td>
                                <td><?php echo $user['completion_rate']; ?>%</td>
                                <td><?php echo number_format($user['task_points_earned']); ?>/<?php echo number_format($user['total_possible_task_points']); ?></td>
                                <td><?php echo number_format($user['quality_points_earned']); ?>/<?php echo number_format($user['total_possible_quality_points']); ?></td>
                                <td class="negative-value"><?php echo number_format($user['lost_task_points']); ?>/<?php echo number_format($user['total_possible_task_points']); ?></td>
                                <td class="negative-value"><?php echo number_format($user['lost_quality_points'] ?? 0); ?>/<?php echo number_format($user['total_possible_quality_points']); ?></td>
                                <td><strong><?php echo number_format($user['total_task_points']); ?></strong></td>
                                <td><strong><?php echo number_format($user['total_quality_points']); ?></strong></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
// Minimal JavaScript for date filter - iOS style
window.initReportsFilter = function() {
    const periodSelect = document.getElementById('reportPeriodSelect');
    const customDateRange = document.getElementById('customDateRange');
    const dateFrom = document.getElementById('dateFrom');
    const dateTo = document.getElementById('dateTo');
    
    if (!periodSelect) {
        console.warn('reportPeriodSelect element not found');
        return;
    }
    
    // Remove old event listener if it exists
    if (periodSelect._changeHandler) {
        periodSelect.removeEventListener('change', periodSelect._changeHandler);
    }
    
    async function loadReports(period, dateFromVal = null, dateToVal = null) {
        console.log('Loading reports with period:', period, dateFromVal, dateToVal);
        
        // Build URL with parameters
        let url = 'admin/admin-reports.php?period=' + encodeURIComponent(period);
        if (period === 'custom' && dateFromVal && dateToVal) {
            url += '&date_from=' + encodeURIComponent(dateFromVal) + '&date_to=' + encodeURIComponent(dateToVal);
        }
        
        // Reload the page content via fetch (SPA-friendly)
        const container = document.getElementById('pageContainer');
        if (container) {
            try {
                container.innerHTML = '<div class="empty-state"><div class="empty-state-icon">⏳</div><h3>Loading...</h3></div>';
                const response = await fetch(url);
                if (!response.ok) throw new Error('Failed to load reports');
                const html = await response.text();
                
                // Extract and remove script tags
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = html;
                const scripts = tempDiv.querySelectorAll('script');
                scripts.forEach(script => script.remove());
                
                // Insert HTML without scripts
                container.innerHTML = tempDiv.innerHTML;
                
                // Execute scripts after a short delay to ensure DOM is ready
                setTimeout(() => {
                    scripts.forEach(oldScript => {
                        const newScript = document.createElement('script');
                        Array.from(oldScript.attributes).forEach(attr => {
                            newScript.setAttribute(attr.name, attr.value);
                        });
                        newScript.appendChild(document.createTextNode(oldScript.innerHTML));
                        container.appendChild(newScript);
                    });
                    
                    // Re-initialize the filter after scripts execute
                    if (window.initReportsFilter) {
                        setTimeout(window.initReportsFilter, 150);
                    }
                }, 50);
            } catch (error) {
                console.error('Error loading reports:', error);
                container.innerHTML = '<div class="empty-state"><div class="empty-state-icon">❌</div><h3>Error Loading Reports</h3><p>' + error.message + '</p><button class="btn btn-primary" onclick="loadPage(\'admin-reports\')">Retry</button></div>';
            }
        } else {
            // Fallback: full page reload
            console.log('No pageContainer found, using full page reload');
            window.location.href = url;
        }
    }
    
    // Create and store the change handler
    const handlePeriodChange = function() {
        const selectedPeriod = this.value;
        console.log('Period changed to:', selectedPeriod);
        if (selectedPeriod === 'custom') {
            if (customDateRange) customDateRange.style.display = 'flex';
        } else {
            if (customDateRange) customDateRange.style.display = 'none';
            loadReports(selectedPeriod);
        }
    };
    
    periodSelect.addEventListener('change', handlePeriodChange);
    periodSelect._changeHandler = handlePeriodChange;
    console.log('Report period filter event listener attached');
    
    if (dateFrom && dateTo) {
        // Remove old handlers if they exist
        if (dateFrom._changeHandler) {
            dateFrom.removeEventListener('change', dateFrom._changeHandler);
        }
        if (dateTo._changeHandler) {
            dateTo.removeEventListener('change', dateTo._changeHandler);
        }
        
        let dateTimeout;
        const handleDateChange = function() {
            clearTimeout(dateTimeout);
            dateTimeout = setTimeout(function() {
                if (dateFrom.value && dateTo.value) {
                    console.log('Custom date range changed:', dateFrom.value, 'to', dateTo.value);
                    loadReports('custom', dateFrom.value, dateTo.value);
                }
            }, 500);
        };
        
        dateFrom.addEventListener('change', handleDateChange);
        dateTo.addEventListener('change', handleDateChange);
        dateFrom._changeHandler = handleDateChange;
        dateTo._changeHandler = handleDateChange;
    }
};

// Initialize on load
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', window.initReportsFilter);
} else {
    window.initReportsFilter();
}
</script>

<?php
$total_tasks_stmt->close();
$completion_stats_stmt->close();
$earned_points_stmt->close();
$lost_points_stmt->close();
$total_possible_stmt->close();
$lost_quality_stmt->close();
$performance_stmt->close();
$conn->close();
?>
