<?php
require_once '../config.php';

if (!isLoggedIn() || !isAdmin()) {
    header('Location: ../index.php');
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    die("Database connection failed. Please check your configuration.");
}

// Get only disabled tasks
// Explicitly select disable_reason to ensure it's included
$tasks_result = $conn->query("
    SELECT t.*, 
           t.disable_reason,
           GROUP_CONCAT(DISTINCT u.id) as assigned_user_ids,
           GROUP_CONCAT(DISTINCT u.username) as assigned_usernames,
           GROUP_CONCAT(DISTINCT u.full_name) as assigned_full_names
    FROM tasks t
    LEFT JOIN task_assignments ta ON t.id = ta.task_id
    LEFT JOIN users u ON ta.user_id = u.id
    WHERE t.is_enabled = 0
    GROUP BY t.id
    ORDER BY t.created_at DESC
");

$tasks = [];
while ($row = $tasks_result->fetch_assoc()) {
    $row['assigned_user_ids'] = $row['assigned_user_ids'] ? explode(',', $row['assigned_user_ids']) : [];
    $row['assigned_usernames'] = $row['assigned_usernames'] ? explode(',', $row['assigned_usernames']) : [];
    $row['assigned_full_names'] = $row['assigned_full_names'] ? explode(',', $row['assigned_full_names']) : [];
    $tasks[] = $row;
}

function formatDate($dateString) {
    return date('M d, Y', strtotime($dateString));
}

function escapeHtml($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}
?>

<div class="page-header">
    <h1>Disabled Tasks</h1>
    <button class="btn btn-secondary" onclick="loadPage('admin-tasks')">Back to All Tasks</button>
</div>

<div class="tasks-list" id="tasksListContainer">
    <?php if (empty($tasks)): ?>
        <div class="empty-state">
            <div class="empty-state-icon">📝</div>
            <h3>No disabled tasks</h3>
            <p>All tasks are currently enabled.</p>
        </div>
    <?php else: ?>
        <?php foreach ($tasks as $task): ?>
            <div class="task-card-item disabled-task-item">
            <?php
            $startDate = new DateTime($task['start_date']);
            $endDate = $task['end_date'] ? new DateTime($task['end_date']) : null;
            $typeLabel = $task['task_type'] === 'recurring' ? 'Recurring' : 'One-Time';
            ?>
            <div class="task-card <?php echo $task['priority']; ?>-priority disabled-task">
                <div class="task-header">
                    <div class="task-title-section">
                        <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 4px;">
                            <h3 class="task-title"><?php echo escapeHtml($task['name']); ?></h3>
                            <?php if (isset($task['task_number'])): ?>
                                <span class="task-number-badge" style="font-size: 11px; color: var(--text-secondary); font-weight: 500; background: var(--bg-tertiary); padding: 2px 8px; border-radius: 6px;">
                                    <?php echo escapeHtml($task['task_number']); ?>
                                </span>
                            <?php endif; ?>
                            <span class="task-status-badge" style="font-size: 10px; color: var(--danger); font-weight: 600; background: rgba(255, 59, 48, 0.1); padding: 2px 8px; border-radius: 6px; text-transform: uppercase;">
                                Disabled
                            </span>
                        </div>
                        <div class="task-meta">
                            <span class="task-meta-item">
                                <span class="icon"><?php echo $task['task_type'] === 'recurring' ? '🔄' : '📌'; ?></span>
                                <span><?php echo $typeLabel; ?></span>
                            </span>
                            <span class="task-meta-item">
                                <span class="icon">📅</span>
                                <span>Start: <?php echo formatDate($task['start_date']); ?></span>
                            </span>
                            <?php if ($endDate): ?>
                                <span class="task-meta-item">
                                    <span class="icon">⏰</span>
                                    <span>Due: <?php echo formatDate($task['end_date']); ?></span>
                                </span>
                            <?php endif; ?>
                            <span class="task-priority <?php echo $task['priority']; ?>"><?php echo ucfirst($task['priority']); ?> Priority</span>
                            <?php if (!empty($task['assigned_usernames'])): ?>
                                <span class="task-meta-item">
                                    <span class="icon">👥</span>
                                    <span>Assigned: <?php echo escapeHtml(implode(', ', $task['assigned_usernames'])); ?></span>
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <p class="task-description"><?php echo escapeHtml($task['description']); ?></p>
                <?php 
                // Check if disable_reason exists and has a value
                // Handle both array key existence and NULL values
                $disable_reason = '';
                
                // Debug: Check what we have
                // Uncomment the line below to debug:
                // error_log("Task ID: " . $task['id'] . ", disable_reason key exists: " . (array_key_exists('disable_reason', $task) ? 'yes' : 'no') . ", value: " . var_export($task['disable_reason'] ?? 'NOT SET', true));
                
                if (array_key_exists('disable_reason', $task)) {
                    $disable_reason = ($task['disable_reason'] !== null && $task['disable_reason'] !== '') ? trim($task['disable_reason']) : '';
                }
                
                if (!empty($disable_reason)): 
                ?>
                    <div class="disable-reason-box">
                        <div class="reason-header">
                            <span style="font-size: 14px;">⚠️</span>
                            <strong>Disable Reason:</strong>
                        </div>
                        <p class="reason-text"><?php echo nl2br(escapeHtml($disable_reason)); ?></p>
                    </div>
                <?php else: ?>
                    <div class="disable-reason-box" style="opacity: 0.6;">
                        <div class="reason-header">
                            <span style="font-size: 14px;">ℹ️</span>
                            <strong>Note:</strong>
                        </div>
                        <p class="reason-text" style="font-style: italic;">No reason was provided when this task was disabled.</p>
                    </div>
                <?php endif; ?>
                <div class="task-actions">
                    <button class="btn btn-primary btn-small" onclick="viewTaskDetails(<?php echo $task['id']; ?>)">View Details</button>
                    <button class="btn btn-success btn-small" onclick="openEnableTaskModal(<?php echo $task['id']; ?>, '<?php echo escapeHtml($task['name']); ?>', '<?php echo escapeHtml($task['task_number'] ?? ''); ?>')">Enable</button>
                    <button class="btn btn-danger btn-small" onclick="openDeleteTaskModal(<?php echo $task['id']; ?>, '<?php echo escapeHtml($task['name']); ?>')">Delete</button>
                </div>
            </div>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>
