<?php
require_once '../config.php';

if (!isLoggedIn() || !isAdmin()) {
    header('Location: ../index.php');
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    die("Database connection failed. Please check your configuration.");
}

// Get all enabled tasks only (disabled tasks are shown in admin-tasks-disabled.php)
$tasks_result = $conn->query("
    SELECT t.*, 
           GROUP_CONCAT(DISTINCT u.id) as assigned_user_ids,
           GROUP_CONCAT(DISTINCT u.username) as assigned_usernames,
           GROUP_CONCAT(DISTINCT u.full_name) as assigned_full_names
    FROM tasks t
    LEFT JOIN task_assignments ta ON t.id = ta.task_id
    LEFT JOIN users u ON ta.user_id = u.id
    WHERE (t.is_enabled = 1 OR t.is_enabled IS NULL)
    GROUP BY t.id
    ORDER BY t.created_at DESC, t.id DESC
");

$tasks = [];
while ($row = $tasks_result->fetch_assoc()) {
    $row['assigned_user_ids'] = $row['assigned_user_ids'] ? explode(',', $row['assigned_user_ids']) : [];
    $row['assigned_usernames'] = $row['assigned_usernames'] ? explode(',', $row['assigned_usernames']) : [];
    $row['assigned_full_names'] = $row['assigned_full_names'] ? explode(',', $row['assigned_full_names']) : [];
    $tasks[] = $row;
}

// Get all users for assignment dropdown
$users_result = $conn->query("SELECT id, username, full_name FROM users WHERE role = 'user' ORDER BY username");
$users = [];
while ($row = $users_result->fetch_assoc()) {
    $users[] = $row;
}

function formatDate($dateString) {
    return date('M d, Y', strtotime($dateString));
}

function escapeHtml($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}
?>

<div class="page-header">
    <h1>Manage Tasks</h1>
    <div style="display: flex; gap: 10px; align-items: center;">
        <button class="btn btn-secondary btn-small" onclick="loadPage('admin-tasks-disabled')">View Disabled Tasks</button>
        <button class="btn btn-primary" onclick="openModal('addTaskModal')">+ Add New Task</button>
    </div>
</div>

<!-- Search and Filters -->
<div class="tasks-search-filters">
    <div class="search-group">
        <label for="searchTasks">🔍 Search Tasks:</label>
        <input type="text" id="searchTasks" placeholder="Search by task number, name, description, notes..." oninput="filterTasks()">
    </div>
    <div class="tasks-filters">
        <div class="filter-group">
            <label for="filterUser">Filter by User:</label>
            <select id="filterUser" onchange="filterTasks()">
                <option value="">All Users</option>
                <?php foreach ($users as $user): ?>
                    <option value="<?php echo $user['id']; ?>"><?php echo escapeHtml($user['full_name'] ?: $user['username']); ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="filter-group">
            <label for="filterDate">Filter by Date:</label>
            <input type="date" id="filterDate" onchange="filterTasks()">
        </div>
        <div class="filter-group">
            <label for="filterPriority">Filter by Priority:</label>
            <select id="filterPriority" onchange="filterTasks()">
                <option value="">All Priorities</option>
                <option value="high">High</option>
                <option value="medium">Medium</option>
                <option value="low">Low</option>
            </select>
        </div>
        <button class="btn btn-secondary btn-small" onclick="clearFilters()">Clear Filters</button>
    </div>
</div>

<?php if (isset($_GET['error'])): ?>
    <div class="alert alert-error" style="margin: 15px; padding: 15px; background: #fee; color: #c33; border-radius: 4px;">
        <?php echo htmlspecialchars($_GET['error']); ?>
    </div>
<?php endif; ?>

<div class="tasks-list" id="tasksListContainer">
    <?php if (empty($tasks)): ?>
        <div class="empty-state">
            <div class="empty-state-icon">📝</div>
            <h3>No tasks found</h3>
            <p>Click "Add New Task" to create a task.</p>
        </div>
    <?php else: ?>
        <?php foreach ($tasks as $task): ?>
            <div class="task-card-item" 
                 data-user-ids="<?php echo implode(',', $task['assigned_user_ids']); ?>" 
                 data-date="<?php echo $task['start_date']; ?>" 
                 data-priority="<?php echo $task['priority']; ?>"
                 data-task-number="<?php echo escapeHtml(strtolower($task['task_number'] ?? '')); ?>"
                 data-name="<?php echo escapeHtml(strtolower($task['name'])); ?>"
                 data-description="<?php echo escapeHtml(strtolower($task['description'])); ?>"
                 data-notes="<?php echo escapeHtml(strtolower($task['notes'] ?? '')); ?>"
                 data-assigned-users="<?php echo escapeHtml(strtolower(implode(' ', $task['assigned_usernames']))); ?>">
            <?php
            $startDate = new DateTime($task['start_date']);
            $endDate = $task['end_date'] ? new DateTime($task['end_date']) : null;
            $typeLabel = $task['task_type'] === 'recurring' ? 'Recurring' : 'One-Time';
            ?>
            <div class="task-card <?php echo $task['priority']; ?>-priority">
                <div class="task-header">
                    <div class="task-title-section">
                        <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 4px;">
                            <h3 class="task-title"><?php echo escapeHtml($task['name']); ?></h3>
                            <?php if (isset($task['task_number'])): ?>
                                <span class="task-number-badge" style="font-size: 11px; color: var(--text-secondary); font-weight: 500; background: var(--bg-tertiary); padding: 2px 8px; border-radius: 6px;">
                                    <?php echo escapeHtml($task['task_number']); ?>
                                </span>
                            <?php endif; ?>
                        </div>
                        <div class="task-meta">
                            <span class="task-meta-item">
                                <span class="icon"><?php echo $task['task_type'] === 'recurring' ? '🔄' : '📌'; ?></span>
                                <span><?php echo $typeLabel; ?></span>
                            </span>
                            <span class="task-meta-item">
                                <span class="icon">📅</span>
                                <span>Start: <?php echo formatDate($task['start_date']); ?></span>
                            </span>
                            <?php if ($endDate): ?>
                                <span class="task-meta-item">
                                    <span class="icon">⏰</span>
                                    <span>Due: <?php echo formatDate($task['end_date']); ?></span>
                                </span>
                            <?php endif; ?>
                            <span class="task-priority <?php echo $task['priority']; ?>"><?php echo ucfirst($task['priority']); ?> Priority</span>
                            <?php if (!empty($task['assigned_usernames'])): ?>
                                <span class="task-meta-item">
                                    <span class="icon">👥</span>
                                    <span>Assigned: <?php echo escapeHtml(implode(', ', $task['assigned_usernames'])); ?></span>
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <p class="task-description"><?php echo escapeHtml($task['description']); ?></p>
                <div class="task-actions">
                    <button class="btn btn-primary btn-small" onclick="viewTaskDetails(<?php echo $task['id']; ?>)">View Details</button>
                    <button class="btn btn-secondary btn-small" onclick="openEditTaskModal(<?php echo $task['id']; ?>)">Edit</button>
                    <?php if (($task['is_enabled'] ?? 1) == 1): ?>
                        <button class="btn btn-warning btn-small" onclick="openDisableTaskModal(<?php echo $task['id']; ?>, '<?php echo escapeHtml($task['name']); ?>', '<?php echo escapeHtml($task['task_number'] ?? ''); ?>')">Disable</button>
                    <?php else: ?>
                        <button class="btn btn-success btn-small" onclick="openEnableTaskModal(<?php echo $task['id']; ?>, '<?php echo escapeHtml($task['name']); ?>', '<?php echo escapeHtml($task['task_number'] ?? ''); ?>')">Enable</button>
                    <?php endif; ?>
                    <button class="btn btn-danger btn-small" onclick="openDeleteTaskModal(<?php echo $task['id']; ?>, '<?php echo escapeHtml($task['name']); ?>')">Delete</button>
                </div>
            </div>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>

<script>
// Store users data for edit modal (make globally available)
window.ALL_USERS = <?php echo json_encode($users); ?>;
window.ALL_TASKS = <?php echo json_encode($tasks); ?>;
</script>
