<?php
require_once '../config.php';

if (!isLoggedIn() || !isAdmin()) {
    header('Location: ../index.php');
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    die("Database connection failed. Please check your configuration.");
}

// Get today and tomorrow dates
$today = date('Y-m-d');
$tomorrow = date('Y-m-d', strtotime('+1 day'));

// Get all users with new stats
// Today's Work = tasks that need to be done today (start_date <= today AND end_date >= today, not completed, not on leave)
// Tomorrow's Work = tasks that need to be done tomorrow (start_date <= tomorrow AND end_date >= tomorrow, not completed, not on leave)
$users_result = $conn->query("
    SELECT u.*,
           (SELECT COUNT(*) FROM task_assignments WHERE user_id = u.id) as total_assigned_works,
           (SELECT COUNT(*) FROM task_assignments ta
            JOIN tasks t ON ta.task_id = t.id
            WHERE ta.user_id = u.id 
            AND t.start_date <= CURDATE() 
            AND (t.end_date IS NULL OR t.end_date >= CURDATE())
            AND NOT EXISTS (
                SELECT 1 FROM task_completions tc 
                WHERE tc.task_id = t.id 
                AND tc.user_id = u.id 
                AND tc.status = 'completed'
            )
            AND NOT EXISTS (
                SELECT 1 FROM user_leaves ul 
                WHERE ul.user_id = u.id 
                AND ul.leave_date = CURDATE()
            )) as today_assigned_works,
           (SELECT COUNT(*) FROM task_assignments ta
            JOIN tasks t ON ta.task_id = t.id
            WHERE ta.user_id = u.id 
            AND t.start_date <= DATE_ADD(CURDATE(), INTERVAL 1 DAY)
            AND (t.end_date IS NULL OR t.end_date >= DATE_ADD(CURDATE(), INTERVAL 1 DAY))
            AND NOT EXISTS (
                SELECT 1 FROM task_completions tc 
                WHERE tc.task_id = t.id 
                AND tc.user_id = u.id 
                AND tc.status = 'completed'
            )
            AND NOT EXISTS (
                SELECT 1 FROM user_leaves ul 
                WHERE ul.user_id = u.id 
                AND ul.leave_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
            )) as tomorrow_assigned_works
    FROM users u
    ORDER BY u.role ASC, u.created_at DESC
");

// Count total admins
$admin_count_result = $conn->query("SELECT COUNT(*) as count FROM users WHERE role = 'admin'");
$admin_count = $admin_count_result->fetch_assoc()['count'] ?? 0;

$users = [];
while ($row = $users_result->fetch_assoc()) {
    // Get leave days (today and future dates only)
    $leave_stmt = $conn->prepare("
        SELECT leave_date 
        FROM user_leaves 
        WHERE user_id = ? AND leave_date >= CURDATE()
        ORDER BY leave_date ASC
    ");
    $leave_stmt->bind_param("i", $row['id']);
    $leave_stmt->execute();
    $leave_result = $leave_stmt->get_result();
    $leave_dates = [];
    while ($leave_row = $leave_result->fetch_assoc()) {
        $leave_dates[] = $leave_row['leave_date'];
    }
    $leave_stmt->close();
    
    $row['leave_dates'] = $leave_dates;
    $row['is_admin'] = ($row['role'] === 'admin');
    $row['can_delete'] = !($row['role'] === 'admin' && $admin_count <= 1);
    $users[] = $row;
}

function escapeHtml($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}
?>

<div class="page-header">
    <h1>Manage Users</h1>
    <button class="btn btn-primary" onclick="openModal('addUserModal')">+ Add New User</button>
</div>

<?php if (isset($_GET['error'])): ?>
    <div class="alert alert-error" style="margin: 15px; padding: 15px; background: #fee; color: #c33; border-radius: 4px;">
        <?php echo htmlspecialchars($_GET['error']); ?>
    </div>
<?php endif; ?>

<div class="users-list">
    <?php if (empty($users)): ?>
        <div class="empty-state">
            <div class="empty-state-icon">👥</div>
            <h3>No users found</h3>
            <p>Click "Add New User" to create a new user.</p>
        </div>
    <?php else: ?>
        <?php foreach ($users as $user): ?>
            <div class="user-card <?php echo $user['role'] === 'admin' ? 'admin-card' : ''; ?>">
                <div class="user-card-header">
                    <span class="user-name"><?php echo escapeHtml($user['full_name'] ?: $user['username']); ?></span>
                    <span class="user-role <?php echo $user['role']; ?>"><?php echo $user['role']; ?></span>
                </div>
                <div class="user-info-section">
                    <div class="user-info-item">
                        <span class="user-info-label">Username:</span>
                        <span class="user-info-value"><?php echo escapeHtml($user['username']); ?></span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Password:</span>
                        <span class="user-info-value password-field" data-password="<?php echo escapeHtml($user['password']); ?>">
                            <span class="password-display">••••••••</span>
                            <button type="button" class="password-toggle" onclick="togglePassword(this)" title="Show/Hide Password">
                                <span class="eye-icon">👁️</span>
                            </button>
                        </span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Created Date:</span>
                        <span class="user-info-value"><?php echo date('M d, Y', strtotime($user['created_at'])); ?></span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Total Assigned Works:</span>
                        <span class="user-info-value clickable-count" onclick="showUserTasks(<?php echo $user['id']; ?>, 'all', '<?php echo escapeHtml($user['username']); ?>')" title="Click to view all tasks">
                            <?php echo $user['total_assigned_works'] ?? 0; ?>
                        </span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Today's Work:</span>
                        <span class="user-info-value clickable-count" onclick="showUserTasks(<?php echo $user['id']; ?>, 'today', '<?php echo escapeHtml($user['username']); ?>')" title="Click to view today's tasks">
                            <?php echo $user['today_assigned_works'] ?? 0; ?>
                        </span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Tomorrow's Work:</span>
                        <span class="user-info-value clickable-count" onclick="showUserTasks(<?php echo $user['id']; ?>, 'tomorrow', '<?php echo escapeHtml($user['username']); ?>')" title="Click to view tomorrow's tasks">
                            <?php echo $user['tomorrow_assigned_works'] ?? 0; ?>
                        </span>
                    </div>
                    <?php if (!empty($user['leave_dates'])): ?>
                        <div class="user-info-item">
                            <span class="user-info-label">Leave Days:</span>
                            <span class="user-info-value leave-dates">
                                <?php 
                                $leave_display = [];
                                foreach ($user['leave_dates'] as $leave_date) {
                                    $date_obj = new DateTime($leave_date);
                                    $today_obj = new DateTime();
                                    if ($date_obj->format('Y-m-d') == $today_obj->format('Y-m-d')) {
                                        $leave_display[] = '<span class="leave-badge today">Today</span>';
                                    } else {
                                        $leave_display[] = '<span class="leave-badge">' . $date_obj->format('M d, Y') . '</span>';
                                    }
                                }
                                echo implode(' ', $leave_display);
                                ?>
                            </span>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="task-actions" style="margin-top: 15px;">
                    <button class="btn btn-secondary btn-small" onclick="openEditUserModal(<?php echo $user['id']; ?>)">Edit</button>
                    <?php if ($user['can_delete']): ?>
                        <button class="btn btn-danger btn-small" onclick="openDeleteUserModal(<?php echo $user['id']; ?>, '<?php echo escapeHtml($user['username']); ?>')">Delete</button>
                    <?php endif; ?>
                    <button class="btn btn-info btn-small" onclick="openLeaveModal(<?php echo $user['id']; ?>, '<?php echo escapeHtml($user['username']); ?>')">Leave</button>
                </div>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>

<script>
// Store users data for edit modal
window.ALL_USERS_DATA = <?php echo json_encode($users); ?>;

function openEditUserModal(userId) {
    const user = window.ALL_USERS_DATA.find(u => u.id == userId);
    if (!user) {
        if (typeof showToast === 'function') {
            showToast('User not found', 'error');
        } else {
            showToast('User not found', 'error');
        }
        return;
    }
    
    // Populate form
    document.getElementById('newUsername').value = user.username;
    document.getElementById('newPassword').value = '';
    document.getElementById('newUserRole').value = user.role;
    document.getElementById('newUserEmail').value = user.email || '';
    document.getElementById('newUserFullName').value = user.full_name || '';
    
    // Set form action and user ID
    const form = document.getElementById('addUserForm');
    form.action = 'edit_user.php';
    document.getElementById('hiddenUserId').value = userId;
    document.getElementById('addUserModalTitle').textContent = 'Edit User';
    document.getElementById('submitUserBtn').textContent = 'Update User';
    document.getElementById('newPasswordLabel').textContent = 'Password (leave blank to keep current)';
    document.getElementById('newPassword').removeAttribute('required');
    document.getElementById('passwordHint').style.display = 'block';
    
    openModal('addUserModal');
}

function openDeleteUserModal(userId, username) {
    document.getElementById('hiddenDeleteUserId').value = userId;
    document.getElementById('deleteUserMessage').textContent = `Are you sure you want to delete user "${username}"? This action cannot be undone.`;
    openModal('deleteUserModal');
}

function openLeaveModal(userId, username) {
    // Try to find elements - start with the form since it's more likely to be unique
    const addLeaveForm = document.getElementById('addLeaveForm');
    let leaveModal = null;
    
    // If form exists, find the modal via the form
    if (addLeaveForm) {
        leaveModal = addLeaveForm.closest('.modal');
        if (leaveModal && !leaveModal.id) {
            // If modal doesn't have ID, set it
            leaveModal.id = 'leaveModal';
        }
    }
    
    // If still not found, try direct ID lookup
    if (!leaveModal) {
        leaveModal = document.getElementById('leaveModal');
    }
    
    // Get form elements
    if (!addLeaveForm) {
        console.error('addLeaveForm not found');
        if (typeof showToast === 'function') {
            showToast('Leave form not found. Please refresh the page.', 'error');
        } else {
            showToast('Leave form not found. Please refresh the page.', 'error');
        }
        return;
    }
    
    const hiddenLeaveUserId = document.getElementById('hiddenLeaveUserId');
    const leaveUserName = document.getElementById('leaveUserName');
    
    if (!hiddenLeaveUserId) {
        console.error('hiddenLeaveUserId element not found, creating it');
        // The element should exist, but if it doesn't, create it
        const input = document.createElement('input');
        input.type = 'hidden';
        input.id = 'hiddenLeaveUserId';
        input.name = 'user_id';
        addLeaveForm.insertBefore(input, addLeaveForm.firstChild);
        input.value = userId;
    } else {
        hiddenLeaveUserId.value = userId;
    }
    
    if (!leaveUserName) {
        console.error('leaveUserName element not found');
        if (typeof showToast === 'function') {
            showToast('Leave modal elements not found. Please refresh the page.', 'error');
        } else {
            showToast('Leave modal elements not found. Please refresh the page.', 'error');
        }
        return;
    }
    
    try {
        leaveUserName.textContent = username;
        
        // Load existing leaves for this user
        loadUserLeaves(userId);
        
        // Open modal - use the modal we found or try by ID
        if (leaveModal) {
            leaveModal.classList.add('active');
            document.body.style.overflow = 'hidden';
        } else {
            // Fallback: try to open by ID anyway
            if (typeof openModal === 'function') {
                openModal('leaveModal');
            }
        }
    } catch (e) {
        console.error('Error opening leave modal:', e);
        if (typeof showToast === 'function') {
            showToast('Error opening leave modal: ' + e.message, 'error');
        } else {
            showToast('Error opening leave modal: ' + e.message, 'error');
        }
    }
}

async function loadUserLeaves(userId) {
    try {
        const response = await fetch(`api/leaves.php?user_id=${userId}`);
        const result = await response.json();
        
        const leavesList = document.getElementById('leavesList');
        if (result.success && result.leaves && result.leaves.length > 0) {
            leavesList.innerHTML = result.leaves.map(leave => {
                const date = new Date(leave.leave_date);
                return `
                    <div class="leave-item" style="display: flex; justify-content: space-between; align-items: center; padding: 8px; margin: 5px 0; background: #f5f5f5; border-radius: 4px;">
                        <span>${date.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' })}</span>
                        <button class="btn btn-small btn-danger" onclick="deleteLeave(${leave.id}, ${userId})" style="padding: 4px 8px; font-size: 12px;">Remove</button>
                    </div>
                `;
            }).join('');
        } else {
            leavesList.innerHTML = '<p style="color: #999; padding: 10px;">No leave dates set</p>';
        }
    } catch (error) {
        console.error('Error loading leaves:', error);
        document.getElementById('leavesList').innerHTML = '<p style="color: #c33; padding: 10px;">Error loading leave dates</p>';
    }
}

async function deleteLeave(leaveId, userId) {
    const confirmed = await showIOSConfirm('Are you sure you want to remove this leave date?', 'Remove Leave Date');
    if (!confirmed) {
        return;
    }
    
    try {
        const response = await fetch('api/leaves.php', {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ id: leaveId })
        });
        
        const result = await response.json();
        
        if (result.success) {
            loadUserLeaves(userId);
            if (typeof showToast === 'function') {
                showToast('Leave date removed successfully', 'success');
            } else {
                showToast('Leave date removed successfully', 'success');
            }
        } else {
            if (typeof showToast === 'function') {
                showToast(result.message || 'Failed to remove leave date', 'error');
            } else {
                showToast(result.message || 'Failed to remove leave date', 'error');
            }
        }
    } catch (error) {
        console.error('Error deleting leave:', error);
        if (typeof showToast === 'function') {
            showToast('Error: ' + error.message, 'error');
        } else {
            showToast('Error: ' + error.message, 'error');
        }
    }
}

function togglePassword(button) {
    const passwordField = button.closest('.password-field');
    const passwordDisplay = passwordField.querySelector('.password-display');
    const actualPassword = passwordField.getAttribute('data-password');
    
    if (passwordDisplay.textContent === '••••••••') {
        passwordDisplay.textContent = actualPassword;
        button.querySelector('.eye-icon').textContent = '🙈';
    } else {
        passwordDisplay.textContent = '••••••••';
        button.querySelector('.eye-icon').textContent = '👁️';
    }
}
</script>
