<?php
require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$conn = getDBConnection();
$method = $_SERVER['REQUEST_METHOD'];
$user_id = getCurrentUserId();

switch ($method) {
    case 'POST':
        // Mark task as complete
        $data = json_decode(file_get_contents('php://input'), true);
        
        $task_id = $data['task_id'] ?? null;
        $completion_date = $data['completion_date'] ?? date('Y-m-d');
        $instance_date = $data['instance_date'] ?? null; // For recurring tasks
        $completion_url = $data['completion_url'] ?? null;
        $completion_text = $data['completion_text'] ?? null;
        $completion_image = $data['completion_image'] ?? null;
        $status = $data['status'] ?? 'completed';
        
        // For recurring tasks, use instance_date if provided, otherwise use completion_date
        $effective_date = $instance_date ? $instance_date : $completion_date;
        
        if (!$task_id) {
            echo json_encode(['success' => false, 'message' => 'Task ID is required']);
            exit;
        }
        
        // Check if user is assigned to this task
        $check_stmt = $conn->prepare("SELECT * FROM task_assignments WHERE task_id = ? AND user_id = ?");
        $check_stmt->bind_param("ii", $task_id, $user_id);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if ($check_result->num_rows === 0) {
            echo json_encode(['success' => false, 'message' => 'You are not assigned to this task']);
            exit;
        }
        
        // Get task details (check if enabled)
        $task_stmt = $conn->prepare("SELECT * FROM tasks WHERE id = ? AND is_enabled = 1");
        $task_stmt->bind_param("i", $task_id);
        $task_stmt->execute();
        $task_result = $task_stmt->get_result();
        $task = $task_result->fetch_assoc();
        
        if (!$task) {
            echo json_encode(['success' => false, 'message' => 'Task not found or disabled']);
            exit;
        }
        
        // Handle image upload
        if (isset($_FILES['completion_image']) && $_FILES['completion_image']['error'] === UPLOAD_ERR_OK) {
            $upload_dir = '../uploads/completions/';
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            $file_ext = pathinfo($_FILES['completion_image']['name'], PATHINFO_EXTENSION);
            $file_name = 'completion_' . $task_id . '_' . $user_id . '_' . time() . '.' . $file_ext;
            $file_path = $upload_dir . $file_name;
            
            if (move_uploaded_file($_FILES['completion_image']['tmp_name'], $file_path)) {
                $completion_image = 'uploads/completions/' . $file_name;
            }
        }
        
        // Calculate points
        $task_points_earned = ($status === 'completed') ? $task['task_points'] : 0;
        $quality_points_earned = ($status === 'completed') ? $task['quality_points'] : 0;
        $lost_task_points = ($status === 'failed') ? abs($task['lost_task_points']) : 0;
        
        $stmt = $conn->prepare("
            INSERT INTO task_completions 
            (task_id, user_id, completion_date, completion_url, completion_text, 
             completion_image, status, task_points_earned, quality_points_earned, lost_task_points)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->bind_param("iissssiiii", $task_id, $user_id, $completion_date, $completion_url, 
                         $completion_text, $completion_image, $status, $task_points_earned, 
                         $quality_points_earned, $lost_task_points);
        
        if ($stmt->execute()) {
            // If this is a recurring task, update the instance status
            $task_type_check = $conn->prepare("SELECT task_type FROM tasks WHERE id = ?");
            $task_type_check->bind_param("i", $task_id);
            $task_type_check->execute();
            $task_type_result = $task_type_check->get_result();
            $task_type_row = $task_type_result->fetch_assoc();
            
            if ($task_type_row && $task_type_row['task_type'] === 'recurring') {
                // Update recurring task instance status
                // Use instance_date if provided, otherwise use completion_date
                $instance_date_to_use = $instance_date ? $instance_date : $completion_date;
                
                $update_instance = $conn->prepare("
                    UPDATE recurring_task_instances 
                    SET status = ? 
                    WHERE parent_task_id = ? AND instance_date = ?
                ");
                $update_instance->bind_param("sis", $status, $task_id, $instance_date_to_use);
                $update_instance->execute();
                
                // If task failed, create next instance immediately
                if ($status === 'failed') {
                    createNextRecurringInstance($task_id, $instance_date_to_use);
                }
            }
            
            // Update user points
            $points_stmt = $conn->prepare("
                INSERT INTO user_points (user_id, total_task_points, total_quality_points, 
                                       total_lost_task_points)
                VALUES (?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                total_task_points = total_task_points + VALUES(total_task_points),
                total_quality_points = total_quality_points + VALUES(total_quality_points),
                total_lost_task_points = total_lost_task_points + VALUES(total_lost_task_points)
            ");
            $points_stmt->bind_param("iiii", $user_id, $task_points_earned, $quality_points_earned, $lost_task_points);
            $points_stmt->execute();
            
            echo json_encode(['success' => true, 'message' => 'Task marked as ' . $status]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to mark task as complete']);
        }
        break;
        
    case 'GET':
        // Get completion history
        $date_from = $_GET['date_from'] ?? null;
        $date_to = $_GET['date_to'] ?? null;
        
        $query = "
            SELECT tc.*, t.name as task_name, t.description as task_description, t.task_number
            FROM task_completions tc
            JOIN tasks t ON tc.task_id = t.id
            WHERE tc.user_id = ? AND t.is_enabled = 1
        ";
        
        $params = [$user_id];
        $types = 'i';
        
        if ($date_from) {
            $query .= " AND tc.completion_date >= ?";
            $params[] = $date_from;
            $types .= 's';
        }
        
        if ($date_to) {
            $query .= " AND tc.completion_date <= ?";
            $params[] = $date_to;
            $types .= 's';
        }
        
        $query .= " ORDER BY tc.completed_at DESC";
        
        $stmt = $conn->prepare($query);
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $completions = [];
        while ($row = $result->fetch_assoc()) {
            $completions[] = $row;
        }
        
        echo json_encode(['success' => true, 'completions' => $completions]);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
}

$conn->close();
?>
