<?php
require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$conn = getDBConnection();
$user_id = getCurrentUserId();
$is_admin = isAdmin();
$date_from = $_GET['date_from'] ?? date('Y-m-d');
$date_to = $_GET['date_to'] ?? date('Y-m-d');

if ($is_admin) {
    // Admin dashboard stats
    $stats = [];
    
    // Total users
    $result = $conn->query("SELECT COUNT(*) as count FROM users WHERE role = 'user'");
    $stats['total_users'] = $result->fetch_assoc()['count'];
    
    // Total tasks
    $result = $conn->query("SELECT COUNT(*) as count FROM tasks");
    $stats['total_tasks'] = $result->fetch_assoc()['count'];
    
    // Active tasks (not completed by all assigned users)
    $result = $conn->query("
        SELECT COUNT(DISTINCT t.id) as count
        FROM tasks t
        WHERE t.id NOT IN (
            SELECT DISTINCT task_id 
            FROM task_completions 
            WHERE status = 'completed'
        )
    ");
    $stats['active_tasks'] = $result->fetch_assoc()['count'];
    
    // Completed tasks
    $result = $conn->query("SELECT COUNT(*) as count FROM task_completions WHERE status = 'completed'");
    $stats['completed_tasks'] = $result->fetch_assoc()['count'];
    
    echo json_encode(['success' => true, 'stats' => $stats]);
} else {
    // User dashboard stats
    $stats = [];
    
    // Get user points
    $points_stmt = $conn->prepare("
        SELECT COALESCE(total_task_points, 0) as task_points,
               COALESCE(total_quality_points, 0) as quality_points,
               COALESCE(total_lost_task_points, 0) as lost_task_points,
               COALESCE(total_lost_quality_points, 0) as lost_quality_points
        FROM user_points
        WHERE user_id = ?
    ");
    $points_stmt->bind_param("i", $user_id);
    $points_stmt->execute();
    $points_result = $points_stmt->get_result();
    $points = $points_result->fetch_assoc() ?: [
        'task_points' => 0,
        'quality_points' => 0,
        'lost_task_points' => 0,
        'lost_quality_points' => 0
    ];
    
    $stats['task_points'] = $points['task_points'];
    $stats['quality_points'] = $points['quality_points'];
    $stats['lost_task_points'] = $points['lost_task_points'];
    $stats['lost_quality_points'] = $points['lost_quality_points'];
    
    // Today's tasks
    $today = date('Y-m-d');
    $today_stmt = $conn->prepare("
        SELECT COUNT(*) as total,
               SUM(CASE WHEN tc.id IS NOT NULL THEN 1 ELSE 0 END) as completed,
               SUM(CASE WHEN tc.id IS NULL THEN 1 ELSE 0 END) as pending
        FROM task_assignments ta
        JOIN tasks t ON ta.task_id = t.id
        LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
            AND ta.user_id = tc.user_id 
            AND tc.completion_date = ?
            AND tc.status = 'completed'
        WHERE ta.user_id = ? AND t.start_date = ? AND t.is_enabled = 1
    ");
    $today_stmt->bind_param("sis", $today, $user_id, $today);
    $today_stmt->execute();
    $today_result = $today_stmt->get_result();
    $today_stats = $today_result->fetch_assoc();
    
    $stats['today_total_tasks'] = $today_stats['total'] ?? 0;
    $stats['today_completed_tasks'] = $today_stats['completed'] ?? 0;
    $stats['today_pending_tasks'] = $today_stats['pending'] ?? 0;
    
    // Progress percentage
    $stats['progress_percentage'] = $stats['today_total_tasks'] > 0 
        ? round(($stats['today_completed_tasks'] / $stats['today_total_tasks']) * 100, 1)
        : 0;
    
    // Get today's pending tasks
    $tasks_stmt = $conn->prepare("
        SELECT t.*, 
               CASE WHEN tc.id IS NOT NULL THEN tc.status ELSE 'pending' END as status
        FROM task_assignments ta
        JOIN tasks t ON ta.task_id = t.id
        LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
            AND ta.user_id = tc.user_id 
            AND tc.completion_date = ?
        WHERE ta.user_id = ? AND t.start_date = ? AND t.is_enabled = 1
        ORDER BY t.priority DESC, t.start_date ASC
    ");
    $tasks_stmt->bind_param("sis", $today, $user_id, $today);
    $tasks_stmt->execute();
    $tasks_result = $tasks_stmt->get_result();
    
    $tasks = [];
    while ($row = $tasks_result->fetch_assoc()) {
        // Ensure all required fields are present
        $row['assigned_user_ids'] = [];
        $row['assigned_usernames'] = [];
        $tasks[] = $row;
    }
    
    $stats['today_tasks'] = $tasks;
    
    echo json_encode(['success' => true, 'stats' => $stats]);
}

$conn->close();
?>
