<?php
require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$conn = getDBConnection();
$user_id = getCurrentUserId();

// Get current date and time in Kolkata timezone for accurate comparisons
$now_kolkata = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
$current_date_kolkata = $now_kolkata->format('Y-m-d');
$current_time_kolkata = $now_kolkata->format('H:i:s');

$date_from = $_GET['date_from'] ?? null;
$date_to = $_GET['date_to'] ?? null;

// Set default date range if not provided (last 30 days)
if (!$date_from) {
    $date_from = date('Y-m-d', strtotime('-30 days'));
}
if (!$date_to) {
    $date_to = date('Y-m-d');
}

// Build query to get all tasks (completed, failed, and pending)
// Use UNION to get completed/failed tasks from task_completions and pending tasks from assignments
$query = "
    (
        -- Get completed and failed tasks from task_completions
        SELECT DISTINCT
               t.id as task_id,
               t.name as task_name,
               t.description as task_description,
               t.priority,
               t.task_number,
               t.task_type,
               COALESCE(rti.instance_date, tc.completion_date, t.start_date) as effective_date,
               COALESCE(rti.end_time, t.end_time) as effective_end_time,
               tc.status,
               tc.completion_date,
               tc.completion_url,
               tc.completion_text,
               tc.completion_image,
               tc.task_points_earned,
               tc.quality_points_earned,
               tc.completed_at,
               COALESCE(rti.instance_date, tc.completion_date, t.start_date) as task_date
        FROM task_completions tc
        JOIN tasks t ON tc.task_id = t.id
        JOIN task_assignments ta ON ta.task_id = t.id AND ta.user_id = tc.user_id
        LEFT JOIN recurring_task_instances rti ON t.id = rti.parent_task_id 
            AND t.task_type = 'recurring'
            AND rti.instance_date = tc.completion_date
        WHERE tc.user_id = ? 
          AND t.is_enabled = 1
          AND tc.completion_date >= ?
          AND tc.completion_date <= ?
    )
    UNION
    (
        -- Get pending tasks from task_assignments (not in task_completions or not completed)
        SELECT DISTINCT
               t.id as task_id,
               t.name as task_name,
               t.description as task_description,
               t.priority,
               t.task_number,
               t.task_type,
               COALESCE(rti.instance_date, t.start_date) as effective_date,
               COALESCE(rti.end_time, t.end_time) as effective_end_time,
               CASE 
                   WHEN (COALESCE(rti.instance_date, t.end_date) IS NOT NULL AND COALESCE(rti.instance_date, t.end_date) < ?) 
                        OR (COALESCE(rti.instance_date, t.end_date) = ? AND COALESCE(rti.end_time, t.end_time) IS NOT NULL AND COALESCE(rti.end_time, t.end_time) < ?)
                        THEN 'failed'
                   ELSE 'pending' 
               END as status,
               NULL as completion_date,
               NULL as completion_url,
               NULL as completion_text,
               NULL as completion_image,
               NULL as task_points_earned,
               NULL as quality_points_earned,
               NULL as completed_at,
               COALESCE(rti.instance_date, t.start_date) as task_date
        FROM task_assignments ta
        JOIN tasks t ON ta.task_id = t.id
        LEFT JOIN recurring_task_instances rti ON t.id = rti.parent_task_id 
            AND t.task_type = 'recurring'
            AND rti.instance_date >= ?
            AND rti.instance_date <= ?
        WHERE ta.user_id = ? 
          AND t.is_enabled = 1
          AND (
              (t.task_type = 'one-time' 
               AND t.start_date >= ? 
               AND t.start_date <= ?
               AND NOT EXISTS (
                   SELECT 1 FROM task_completions tc2 
                   WHERE tc2.task_id = t.id 
                   AND tc2.user_id = ta.user_id 
                   AND tc2.status = 'completed'
               ))
              OR (t.task_type = 'recurring' 
                  AND rti.instance_date >= ?
                  AND rti.instance_date <= ?
                  AND (rti.status IS NULL OR rti.status = 'pending')
                  AND NOT EXISTS (
                      SELECT 1 FROM task_completions tc3 
                      WHERE tc3.task_id = t.id 
                      AND tc3.user_id = ta.user_id 
                      AND tc3.completion_date = rti.instance_date
                      AND tc3.status = 'completed'
                  ))
          )
    )
    ORDER BY task_date DESC, 
             CASE WHEN status = 'completed' THEN 1 
                  WHEN status = 'failed' THEN 2 
                  ELSE 3 END
    LIMIT 100
";

// Bind parameters for the query
// First UNION: user_id, date_from, date_to (3 params: i, s, s)
// Second UNION: current_date, current_date, current_time, date_from, date_to, user_id, date_from, date_to, date_from, date_to (10 params: s, s, s, s, s, i, s, s, s, s)
// Total: 13 parameters
$params = [$user_id, $date_from, $date_to, $current_date_kolkata, $current_date_kolkata, $current_time_kolkata, $date_from, $date_to, $user_id, $date_from, $date_to, $date_from, $date_to];
$types = 'isssssssississs'; // 13 parameters: i-s-s-s-s-s-s-s-i-s-s-s-s

$stmt = $conn->prepare($query);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();

$history = [];
while ($row = $result->fetch_assoc()) {
    // Determine final status
    if ($row['status'] === 'completed' || $row['status'] === 'failed') {
        // Use status from task_completions
    } else {
        // Check if task is actually failed (past due date)
        $task_date = $row['effective_date'] ?? $row['task_date'];
        $end_time = $row['effective_end_time'];
        
        if ($task_date && $task_date < date('Y-m-d')) {
            $row['status'] = 'failed';
        } elseif ($task_date === date('Y-m-d') && $end_time) {
            $now = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
            $end_datetime = new DateTime($task_date . ' ' . $end_time, new DateTimeZone('Asia/Kolkata'));
            if ($end_datetime < $now) {
                $row['status'] = 'failed';
            } else {
                $row['status'] = 'pending';
            }
        } else {
            $row['status'] = 'pending';
        }
    }
    
    $history[] = $row;
}

echo json_encode(['success' => true, 'history' => $history]);

$conn->close();
?>
