<?php
require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$conn = getDBConnection();
$method = $_SERVER['REQUEST_METHOD'];
$is_admin = isAdmin();

switch ($method) {
    case 'GET':
        // Get leaves for a user (admin only, or user can see their own)
        $user_id = isset($_GET['user_id']) ? intval($_GET['user_id']) : null;
        
        if (!$user_id) {
            echo json_encode(['success' => false, 'message' => 'User ID is required']);
            exit;
        }
        
        // Check permission: admin can see any user's leaves, users can only see their own
        $current_user_id = getCurrentUserId();
        if (!$is_admin && $user_id != $current_user_id) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Forbidden']);
            exit;
        }
        
        $stmt = $conn->prepare("
            SELECT ul.*, u.username, u.full_name
            FROM user_leaves ul
            JOIN users u ON ul.user_id = u.id
            WHERE ul.user_id = ?
            ORDER BY ul.leave_date ASC
        ");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $leaves = [];
        while ($row = $result->fetch_assoc()) {
            $leaves[] = $row;
        }
        
        echo json_encode(['success' => true, 'leaves' => $leaves]);
        break;
        
    case 'POST':
        // Add leave date (admin only)
        if (!$is_admin) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Forbidden']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        $user_id = isset($data['user_id']) ? intval($data['user_id']) : null;
        $leave_date = isset($data['leave_date']) ? $data['leave_date'] : null;
        
        if (!$user_id || !$leave_date) {
            echo json_encode(['success' => false, 'message' => 'User ID and leave date are required']);
            exit;
        }
        
        // Validate date format
        $date_obj = DateTime::createFromFormat('Y-m-d', $leave_date);
        if (!$date_obj || $date_obj->format('Y-m-d') !== $leave_date) {
            echo json_encode(['success' => false, 'message' => 'Invalid date format']);
            exit;
        }
        
        // Check if leave already exists
        $check_stmt = $conn->prepare("SELECT id FROM user_leaves WHERE user_id = ? AND leave_date = ?");
        $check_stmt->bind_param("is", $user_id, $leave_date);
        $check_stmt->execute();
        if ($check_stmt->get_result()->num_rows > 0) {
            echo json_encode(['success' => false, 'message' => 'Leave date already exists for this user']);
            exit;
        }
        
        $current_user_id = getCurrentUserId();
        $stmt = $conn->prepare("
            INSERT INTO user_leaves (user_id, leave_date, created_by)
            VALUES (?, ?, ?)
        ");
        $stmt->bind_param("isi", $user_id, $leave_date, $current_user_id);
        
        if ($stmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'Leave date added successfully']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to add leave date']);
        }
        break;
        
    case 'DELETE':
        // Delete leave date (admin only)
        if (!$is_admin) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Forbidden']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $leave_id = isset($data['id']) ? intval($data['id']) : null;
        
        if (!$leave_id) {
            echo json_encode(['success' => false, 'message' => 'Leave ID is required']);
            exit;
        }
        
        $stmt = $conn->prepare("DELETE FROM user_leaves WHERE id = ?");
        $stmt->bind_param("i", $leave_id);
        
        if ($stmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'Leave date removed successfully']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to remove leave date']);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
}

$conn->close();
?>
