<?php
require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn() || !isAdmin()) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Forbidden']);
    exit;
}

$conn = getDBConnection();
$filter = $_GET['filter'] ?? 'today';

// Calculate date range based on filter
$date_from = '';
$date_to = '';

switch ($filter) {
    case 'today':
        $date_from = date('Y-m-d');
        $date_to = date('Y-m-d');
        break;
    case 'yesterday':
        $date_from = date('Y-m-d', strtotime('-1 day'));
        $date_to = date('Y-m-d', strtotime('-1 day'));
        break;
    case 'yesterday-before':
        $date_from = date('Y-m-d', strtotime('-2 days'));
        $date_to = date('Y-m-d', strtotime('-2 days'));
        break;
    case 'last7days':
        $date_from = date('Y-m-d', strtotime('-7 days'));
        $date_to = date('Y-m-d');
        break;
    case 'thisweek':
        // Start of week (Monday)
        $dayOfWeek = date('w');
        $daysToMonday = ($dayOfWeek == 0) ? 6 : $dayOfWeek - 1;
        $date_from = date('Y-m-d', strtotime("-$daysToMonday days"));
        $date_to = date('Y-m-d');
        break;
    default:
        $date_from = date('Y-m-d');
        $date_to = date('Y-m-d');
}

// Get user performance with date filtering
$performance_stmt = $conn->prepare("
    SELECT 
        u.id,
        u.username,
        u.full_name,
        COUNT(DISTINCT CASE WHEN ta.task_id IS NOT NULL AND t.start_date BETWEEN ? AND ? THEN ta.task_id END) as total_tasks,
        COUNT(DISTINCT CASE WHEN tc.status = 'completed' AND tc.completion_date BETWEEN ? AND ? THEN tc.id END) as completed_tasks,
        (COUNT(DISTINCT CASE WHEN ta.task_id IS NOT NULL AND t.start_date BETWEEN ? AND ? THEN ta.task_id END) - 
         COUNT(DISTINCT CASE WHEN tc.status = 'completed' AND tc.completion_date BETWEEN ? AND ? THEN tc.id END)) as pending_tasks,
        COALESCE(SUM(CASE WHEN tc.status = 'completed' AND tc.completion_date BETWEEN ? AND ? THEN tc.task_points_earned ELSE 0 END), 0) as task_points,
        COALESCE(SUM(CASE WHEN tc.status = 'completed' AND tc.completion_date BETWEEN ? AND ? THEN tc.quality_points_earned ELSE 0 END), 0) as quality_points
    FROM users u
    LEFT JOIN task_assignments ta ON u.id = ta.user_id
    LEFT JOIN tasks t ON ta.task_id = t.id AND t.is_enabled = 1
    LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
        AND ta.user_id = tc.user_id
    WHERE u.role = 'user'
    GROUP BY u.id, u.username, u.full_name
    ORDER BY task_points DESC, quality_points DESC
");

// Bind parameters (date_from and date_to used multiple times)
$performance_stmt->bind_param("ssssssssss", 
    $date_from, $date_to,  // total_tasks
    $date_from, $date_to,  // completed_tasks
    $date_from, $date_to,  // pending_tasks (total)
    $date_from, $date_to,  // pending_tasks (completed)
    $date_from, $date_to,  // task_points
    $date_from, $date_to   // quality_points
);

$performance_stmt->execute();
$performance_result = $performance_stmt->get_result();

$performance = [];
while ($row = $performance_result->fetch_assoc()) {
    $total = $row['total_tasks'];
    $completion_rate = $total > 0 
        ? round(($row['completed_tasks'] / $total) * 100, 1)
        : 0;
    $row['completion_rate'] = $completion_rate;
    $row['total_points'] = $row['task_points'] + $row['quality_points'];
    $performance[] = $row;
}

echo json_encode(['success' => true, 'performance' => $performance, 'filter' => $filter]);

$conn->close();
?>
