<?php
require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$conn = getDBConnection();
$user_id = getCurrentUserId();
$period = $_GET['period'] ?? 'week';
$date_from = $_GET['date_from'] ?? null;
$date_to = $_GET['date_to'] ?? null;

// Calculate date range based on period
if (!$date_from || !$date_to) {
    switch ($period) {
        case 'today':
            $date_from = $date_to = date('Y-m-d');
            break;
        case 'week':
            $date_from = date('Y-m-d', strtotime('monday this week'));
            $date_to = date('Y-m-d', strtotime('sunday this week'));
            break;
        case 'month':
            $date_from = date('Y-m-01');
            $date_to = date('Y-m-t');
            break;
        case 'year':
            $date_from = date('Y-01-01');
            $date_to = date('Y-12-31');
            break;
        default:
            $date_from = date('Y-m-d', strtotime('-7 days'));
            $date_to = date('Y-m-d');
    }
}

// Get completion stats
$completion_stmt = $conn->prepare("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
        SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending
    FROM (
        SELECT 
            ta.task_id,
            COALESCE(tc.status, 'pending') as status
        FROM task_assignments ta
        JOIN tasks t ON ta.task_id = t.id
        LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
            AND ta.user_id = tc.user_id
            AND tc.completion_date BETWEEN ? AND ?
        WHERE ta.user_id = ? 
            AND t.start_date BETWEEN ? AND ?
            AND t.is_enabled = 1
    ) as task_status
");
$completion_stmt->bind_param("ssiss", $date_from, $date_to, $user_id, $date_from, $date_to);
$completion_stmt->execute();
$completion_result = $completion_stmt->get_result();
$completion_stats = $completion_result->fetch_assoc();

// Get points breakdown
$points_stmt = $conn->prepare("
    SELECT 
        SUM(task_points_earned) as task_points,
        SUM(quality_points_earned) as quality_points,
        SUM(lost_task_points) as lost_task_points
    FROM task_completions
    WHERE user_id = ? AND completion_date BETWEEN ? AND ?
");
$points_stmt->bind_param("iss", $user_id, $date_from, $date_to);
$points_stmt->execute();
$points_result = $points_stmt->get_result();
$points = $points_result->fetch_assoc();

$progress = [
    'period' => $period,
    'date_from' => $date_from,
    'date_to' => $date_to,
    'completion_stats' => [
        'total' => (int)$completion_stats['total'],
        'completed' => (int)$completion_stats['completed'],
        'failed' => (int)$completion_stats['failed'],
        'pending' => (int)$completion_stats['pending']
    ],
    'points' => [
        'task_points' => (int)($points['task_points'] ?? 0),
        'quality_points' => (int)($points['quality_points'] ?? 0),
        'lost_task_points' => (int)($points['lost_task_points'] ?? 0)
    ]
];

// Calculate percentages
$total = $progress['completion_stats']['total'];
if ($total > 0) {
    $progress['completion_percentage'] = round(($progress['completion_stats']['completed'] / $total) * 100, 1);
} else {
    $progress['completion_percentage'] = 0;
}

echo json_encode(['success' => true, 'progress' => $progress]);

$conn->close();
?>
