<?php
require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn() || !isAdmin()) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Forbidden']);
    exit;
}

$conn = getDBConnection();
$date_from = $_GET['date_from'] ?? date('Y-m-01');
$date_to = $_GET['date_to'] ?? date('Y-m-t');

// User performance report
$performance_stmt = $conn->prepare("
    SELECT 
        u.id,
        u.username,
        u.full_name,
        COUNT(DISTINCT ta.task_id) as total_tasks,
        COUNT(DISTINCT CASE WHEN tc.status = 'completed' THEN tc.id END) as completed_tasks,
        COUNT(DISTINCT CASE WHEN tc.status = 'failed' THEN tc.id END) as failed_tasks,
        COALESCE(SUM(CASE WHEN tc.status = 'completed' THEN tc.task_points_earned ELSE 0 END), 0) as task_points,
        COALESCE(SUM(CASE WHEN tc.status = 'completed' THEN tc.quality_points_earned ELSE 0 END), 0) as quality_points
    FROM users u
    LEFT JOIN task_assignments ta ON u.id = ta.user_id
    LEFT JOIN tasks t ON ta.task_id = t.id AND t.start_date BETWEEN ? AND ? AND t.is_enabled = 1
    LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
        AND ta.user_id = tc.user_id
        AND tc.completion_date BETWEEN ? AND ?
    WHERE u.role = 'user'
    GROUP BY u.id, u.username, u.full_name
    ORDER BY task_points DESC, quality_points DESC
");
$performance_stmt->bind_param("ssss", $date_from, $date_to, $date_from, $date_to);
$performance_stmt->execute();
$performance_result = $performance_stmt->get_result();

$performance = [];
while ($row = $performance_result->fetch_assoc()) {
    $completion_rate = $row['total_tasks'] > 0 
        ? round(($row['completed_tasks'] / $row['total_tasks']) * 100, 1)
        : 0;
    $row['completion_rate'] = $completion_rate;
    $performance[] = $row;
}

// Task summary report
$task_summary_stmt = $conn->prepare("
    SELECT 
        t.id,
        t.name,
        t.priority,
        COUNT(DISTINCT ta.user_id) as assigned_users,
        COUNT(DISTINCT CASE WHEN tc.status = 'completed' THEN tc.id END) as completed_count,
        COUNT(DISTINCT ta.user_id) - COUNT(DISTINCT CASE WHEN tc.status = 'completed' THEN tc.id END) as pending_count
    FROM tasks t
    LEFT JOIN task_assignments ta ON t.id = ta.task_id
    LEFT JOIN task_completions tc ON t.id = tc.task_id AND tc.completion_date BETWEEN ? AND ?
    WHERE t.start_date BETWEEN ? AND ? AND t.is_enabled = 1
    GROUP BY t.id, t.name, t.priority
    ORDER BY t.start_date DESC
");
$task_summary_stmt->bind_param("ssss", $date_from, $date_to, $date_from, $date_to);
$task_summary_stmt->execute();
$task_summary_result = $task_summary_stmt->get_result();

$task_summary = [];
while ($row = $task_summary_result->fetch_assoc()) {
    $task_summary[] = $row;
}

echo json_encode([
    'success' => true,
    'reports' => [
        'user_performance' => $performance,
        'task_summary' => $task_summary,
        'date_from' => $date_from,
        'date_to' => $date_to
    ]
]);

$conn->close();
?>
