<?php
// Suppress error display, log errors instead
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn() || !isAdmin()) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Forbidden']);
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    echo json_encode(['success' => false, 'message' => 'Database connection failed']);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$user_id = getCurrentUserId();

switch ($method) {
    case 'PUT':
        // Enable/Disable task
        $data = json_decode(file_get_contents('php://input'), true);
        
        $task_id = $data['task_id'] ?? null;
        $is_enabled = $data['is_enabled'] ?? 1;
        $admin_password = $data['admin_password'] ?? '';
        $disable_reason = $data['disable_reason'] ?? null;
        
        if (!$task_id) {
            echo json_encode(['success' => false, 'message' => 'Task ID is required']);
            exit;
        }
        
        if (empty($admin_password)) {
            echo json_encode(['success' => false, 'message' => 'Admin password is required']);
            exit;
        }
        
        // If disabling, require a reason
        if ($is_enabled == 0 && empty($disable_reason)) {
            echo json_encode(['success' => false, 'message' => 'Please provide a reason for disabling this task']);
            exit;
        }
        
        // Verify admin password
        $admin_stmt = $conn->prepare("SELECT password FROM users WHERE id = ? AND role = 'admin'");
        $admin_stmt->bind_param("i", $user_id);
        $admin_stmt->execute();
        $admin_result = $admin_stmt->get_result();
        $admin = $admin_result->fetch_assoc();
        
        if (!$admin || $admin_password !== $admin['password']) {
            echo json_encode(['success' => false, 'message' => 'Invalid admin password']);
            exit;
        }
        
        // Update task status and reason
        $is_enabled_int = $is_enabled ? 1 : 0;
        
        // Clean the disable reason
        $disable_reason_clean = !empty($disable_reason) ? trim($disable_reason) : '';
        
        try {
            // Check if disable_reason column exists
            $checkColumn = $conn->query("SHOW COLUMNS FROM tasks LIKE 'disable_reason'");
            $columnExists = $checkColumn && $checkColumn->num_rows > 0;
            
            if ($is_enabled_int == 0) {
                // Disabling - save reason
                if ($columnExists) {
                    // Ensure disable_reason is not empty
                    if (empty($disable_reason_clean)) {
                        throw new Exception("Disable reason is required");
                    }
                    $stmt = $conn->prepare("UPDATE tasks SET is_enabled = ?, disable_reason = ? WHERE id = ?");
                    if (!$stmt) {
                        throw new Exception("Prepare failed: " . $conn->error);
                    }
                    $stmt->bind_param("isi", $is_enabled_int, $disable_reason_clean, $task_id);
                } else {
                    // Column doesn't exist - return error asking to run migration
                    throw new Exception("disable_reason column does not exist. Please run the migration to add this column.");
                }
            } else {
                // Enabling - clear reason
                if ($columnExists) {
                    $stmt = $conn->prepare("UPDATE tasks SET is_enabled = ?, disable_reason = NULL WHERE id = ?");
                    if (!$stmt) {
                        throw new Exception("Prepare failed: " . $conn->error);
                    }
                    $stmt->bind_param("ii", $is_enabled_int, $task_id);
                } else {
                    // Column doesn't exist, just update status
                    $stmt = $conn->prepare("UPDATE tasks SET is_enabled = ? WHERE id = ?");
                    if (!$stmt) {
                        throw new Exception("Prepare failed: " . $conn->error);
                    }
                    $stmt->bind_param("ii", $is_enabled_int, $task_id);
                }
            }
            
            if ($stmt->execute()) {
                $status_text = $is_enabled ? 'enabled' : 'disabled';
                
                // Log for debugging (remove in production)
                if ($is_enabled_int == 0 && $columnExists) {
                    error_log("Task disabled - ID: $task_id, Reason: " . substr($disable_reason_clean, 0, 50));
                }
                
                echo json_encode([
                    'success' => true, 
                    'message' => "Task {$status_text} successfully",
                    'is_enabled' => $is_enabled_int
                ]);
            } else {
                throw new Exception("Execute failed: " . $stmt->error);
            }
            $stmt->close();
        } catch (Exception $e) {
            error_log("Task status update error: " . $e->getMessage());
            echo json_encode([
                'success' => false, 
                'message' => 'Failed to update task status: ' . $e->getMessage()
            ]);
        }
        break;
        
    case 'GET':
        // Get disabled tasks list
        $disabled_only = $_GET['disabled_only'] ?? '0';
        
        if ($disabled_only === '1') {
            // Get only disabled tasks
            $stmt = $conn->query("
                SELECT t.*, 
                       GROUP_CONCAT(DISTINCT u.id) as assigned_user_ids,
                       GROUP_CONCAT(DISTINCT u.username) as assigned_usernames,
                       GROUP_CONCAT(DISTINCT u.full_name) as assigned_full_names
                FROM tasks t
                LEFT JOIN task_assignments ta ON t.id = ta.task_id
                LEFT JOIN users u ON ta.user_id = u.id
                WHERE t.is_enabled = 0
                GROUP BY t.id
                ORDER BY t.created_at DESC
            ");
        } else {
            // Get all tasks with their status
            $stmt = $conn->query("
                SELECT t.*, 
                       GROUP_CONCAT(DISTINCT u.id) as assigned_user_ids,
                       GROUP_CONCAT(DISTINCT u.username) as assigned_usernames,
                       GROUP_CONCAT(DISTINCT u.full_name) as assigned_full_names
                FROM tasks t
                LEFT JOIN task_assignments ta ON t.id = ta.task_id
                LEFT JOIN users u ON ta.user_id = u.id
                GROUP BY t.id
                ORDER BY t.is_enabled DESC, t.created_at DESC
            ");
        }
        
        $tasks = [];
        while ($row = $stmt->fetch_assoc()) {
            $row['assigned_user_ids'] = $row['assigned_user_ids'] ? explode(',', $row['assigned_user_ids']) : [];
            $row['assigned_usernames'] = $row['assigned_usernames'] ? explode(',', $row['assigned_usernames']) : [];
            $row['assigned_full_names'] = $row['assigned_full_names'] ? explode(',', $row['assigned_full_names']) : [];
            $tasks[] = $row;
        }
        
        echo json_encode(['success' => true, 'tasks' => $tasks]);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
}

$conn->close();
?>
