<?php
require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$conn = getDBConnection();
$method = $_SERVER['REQUEST_METHOD'];
$user_id = getCurrentUserId();
$is_admin = isAdmin();

switch ($method) {
    case 'GET':
        // Get tasks
        $task_id = $_GET['id'] ?? null;
        $status = $_GET['status'] ?? 'all';
        $date = $_GET['date'] ?? null;
        $priority = $_GET['priority'] ?? 'all';
        
        if ($task_id) {
            // Get single task
            $query = "
                SELECT t.*, GROUP_CONCAT(DISTINCT u.id) as assigned_user_ids,
                       GROUP_CONCAT(DISTINCT u.username) as assigned_usernames
                FROM tasks t
                LEFT JOIN task_assignments ta ON t.id = ta.task_id
                LEFT JOIN users u ON ta.user_id = u.id
                WHERE t.id = ?
            ";
            
            // Exclude disabled tasks from user views
            if (!$is_admin) {
                $query .= " AND t.is_enabled = 1";
            }
            
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $task_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $task = $result->fetch_assoc();
            
            if ($task) {
                $task['assigned_user_ids'] = $task['assigned_user_ids'] ? explode(',', $task['assigned_user_ids']) : [];
                $task['assigned_usernames'] = $task['assigned_usernames'] ? explode(',', $task['assigned_usernames']) : [];
            }
            
            echo json_encode(['success' => true, 'task' => $task]);
        } else {
            // Get list of tasks
            $target_user_id = $_GET['user_id'] ?? null;
            $filter_user_id = $target_user_id ? intval($target_user_id) : ($is_admin ? null : $user_id);
            
            $query = "
                SELECT DISTINCT t.*, 
                       GROUP_CONCAT(DISTINCT u.id) as assigned_user_ids,
                       GROUP_CONCAT(DISTINCT u.username) as assigned_usernames,
                       GROUP_CONCAT(DISTINCT u.full_name) as assigned_full_names";
            
            // For date-filtered queries, join with recurring_task_instances to match stats query logic
            if ($date) {
                $query .= ",
                       rti.instance_date as effective_date,
                       rti.start_time as effective_start_time,
                       rti.end_time as effective_end_time,
                       rti.status as instance_status
                ";
            }
            
            $query .= " FROM tasks t
                LEFT JOIN task_assignments ta ON t.id = ta.task_id
                LEFT JOIN users u ON ta.user_id = u.id";
            
            if ($date) {
                $query .= " LEFT JOIN recurring_task_instances rti ON t.id = rti.parent_task_id 
                    AND rti.instance_date = ?";
            }
            
            $query .= " WHERE 1=1";
            
            // Exclude disabled tasks from user views (admins can see all)
            if (!$is_admin) {
                $query .= " AND t.is_enabled = 1";
            }
            
            $params = [];
            $types = '';
            
            // Add date parameter for recurring_task_instances join if date is provided
            if ($date) {
                $params[] = $date;
                $types .= 's';
            }
            
            // Filter by user_id (admin can view any user's tasks, regular users can only view their own)
            if ($filter_user_id) {
                if ($is_admin || $filter_user_id == $user_id) {
                    $query .= " AND ta.user_id = ?";
                    $params[] = $filter_user_id;
                    $types .= 'i';
                } else {
                    // Non-admin trying to view another user's tasks
                    http_response_code(403);
                    echo json_encode(['success' => false, 'message' => 'Forbidden']);
                    exit;
                }
            } else if (!$is_admin) {
                $query .= " AND ta.user_id = ?";
                $params[] = $user_id;
                $types .= 'i';
            }
            
            // Date filter - for today/tomorrow, check if task is scheduled for that date
            // Also match stats query logic: only show one-time tasks OR recurring tasks with instances
            if ($date) {
                $query .= " AND t.start_date <= ? AND (t.end_date IS NULL OR t.end_date >= ?)";
                $params[] = $date;
                $params[] = $date;
                $types .= 'ss';
            }
            
            if ($priority !== 'all') {
                $query .= " AND t.priority = ?";
                $params[] = $priority;
                $types .= 's';
            }
            
            // Exclude disabled tasks from user views (already added above)
            // Exclude tasks on leave days (but don't exclude completed tasks if status filter is 'completed' or 'all')
            if ($filter_user_id && $date && $status !== 'completed' && $status !== 'all') {
                $query .= " AND NOT EXISTS (
                    SELECT 1 FROM task_completions tc 
                    WHERE tc.task_id = t.id 
                    AND tc.user_id = ? 
                    AND tc.status = 'completed'
                )";
                $params[] = $filter_user_id;
                $types .= 'i';
            }
            
            // Always exclude tasks on leave days
            if ($filter_user_id && $date) {
                $query .= " AND NOT EXISTS (
                    SELECT 1 FROM user_leaves ul 
                    WHERE ul.user_id = ? 
                    AND ul.leave_date = ?
                )";
                $params[] = $filter_user_id;
                $params[] = $date;
                $types .= 'is';
            }
            
            $query .= " GROUP BY t.id ORDER BY t.start_date DESC, t.priority DESC";
            
            $stmt = $conn->prepare($query);
            if (!empty($params)) {
                $stmt->bind_param($types, ...$params);
            }
            $stmt->execute();
            $result = $stmt->get_result();
            
            // Get today's day of week for filtering recurring tasks (0 = Sunday, 1 = Monday, ..., 6 = Saturday)
            $date_day_of_week = null;
            if ($date) {
                try {
                    $date_obj = new DateTime($date, new DateTimeZone('Asia/Kolkata'));
                    $date_day_of_week = (int)$date_obj->format('w');
                } catch (Exception $e) {
                    // If date parsing fails, skip day of week check
                }
            }
            
            $tasks = [];
            $check_user_id = $filter_user_id ? $filter_user_id : $user_id;
            while ($row = $result->fetch_assoc()) {
                // For recurring tasks, check if the date is a recurring day
                if ($row['task_type'] === 'recurring' && $date && $date_day_of_week !== null) {
                    $recurring_days = $row['skip_days'] ? explode(',', $row['skip_days']) : [];
                    // Check if the date's day of week is in the recurring days list
                    if (empty($recurring_days) || !in_array((string)$date_day_of_week, $recurring_days)) {
                        // This date is not a recurring day, skip this task
                        continue;
                    }
                }
                
                // For date-filtered queries, check completion status for that specific date
                // Recurring tasks are already filtered in the WHERE clause to only include those with instances
                $effective_date = $date;
                
                // Check completion status for the specific date
                if ($date) {
                    $completion_stmt = $conn->prepare("
                        SELECT status, completion_date 
                        FROM task_completions 
                        WHERE task_id = ? AND user_id = ? AND completion_date = ?
                        ORDER BY completed_at DESC LIMIT 1
                    ");
                    $completion_stmt->bind_param("iis", $row['id'], $check_user_id, $date);
                } else {
                    $completion_stmt = $conn->prepare("
                        SELECT status, completion_date 
                        FROM task_completions 
                        WHERE task_id = ? AND user_id = ? 
                        ORDER BY completed_at DESC LIMIT 1
                    ");
                    $completion_stmt->bind_param("ii", $row['id'], $check_user_id);
                }
                $completion_stmt->execute();
                $completion_result = $completion_stmt->get_result();
                $completion = $completion_result->fetch_assoc();
                $completion_stmt->close();
                
                // If no completion found and date is provided, check if task is failed
                if (!$completion && $date) {
                    // Use effective instance date/time for recurring tasks
                    $task_end_date = $row['effective_date'] ?? ($row['end_date'] ?? $row['start_date']);
                    $task_end_time = $row['effective_end_time'] ?? ($row['end_time'] ?? null);
                    
                    // Get current time in Kolkata timezone
                    $now_kolkata = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
                    $current_date_kolkata = $now_kolkata->format('Y-m-d');
                    $current_time_kolkata = $now_kolkata->format('H:i:s');
                    
                    if (!empty($row['instance_status'])) {
                        $row['status'] = $row['instance_status'];
                    } elseif ($task_end_date < $current_date_kolkata || 
                        ($task_end_date == $current_date_kolkata && $task_end_time && $task_end_time < $current_time_kolkata)) {
                        $row['status'] = 'failed';
                    } else {
                        $row['status'] = 'pending';
                    }
                } else {
                    $row['status'] = $completion ? $completion['status'] : 'pending';
                }
                $row['completion_date'] = $completion ? $completion['completion_date'] : null;
                $row['assigned_user_ids'] = $row['assigned_user_ids'] ? explode(',', $row['assigned_user_ids']) : [];
                $row['assigned_usernames'] = $row['assigned_usernames'] ? explode(',', $row['assigned_usernames']) : [];
                $row['assigned_full_names'] = $row['assigned_full_names'] ? explode(',', $row['assigned_full_names']) : [];
                
                // Filter by status if needed
                if ($status !== 'all' && $row['status'] !== $status) {
                    continue;
                }
                
                $tasks[] = $row;
            }
            
            echo json_encode(['success' => true, 'tasks' => $tasks]);
        }
        break;
        
    case 'POST':
        // Create new task (admin only)
        if (!$is_admin) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Forbidden']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        $name = $data['name'] ?? '';
        $description = $data['description'] ?? '';
        $priority = $data['priority'] ?? 'medium';
        $start_date = $data['start_date'] ?? date('Y-m-d');
        $end_date = $data['end_date'] ?? null;
        $start_time = $data['start_time'] ?? null;
        $end_time = $data['end_time'] ?? null;
        $task_type = $data['task_type'] ?? 'one-time';
        $recurring_interval = null; // No longer used, kept for backward compatibility
        $recurring_days = $data['recurring_days'] ?? [];
        $skip_days = null; // Repurposed: now stores the days to repeat on
        $task_points = $data['task_points'] ?? 1;
        $quality_points = $data['quality_points'] ?? 10;
        $lost_task_points = $data['lost_task_points'] ?? -1;
        $notes = $data['notes'] ?? null;
        $assigned_users = $data['assigned_users'] ?? [];
        
        if (empty($name) || empty($description)) {
            echo json_encode(['success' => false, 'message' => 'Name and description are required']);
            exit;
        }
        
        // Handle recurring days - store in skip_days field (repurposed)
        if ($task_type === 'recurring') {
            if (is_array($recurring_days) && !empty($recurring_days)) {
                $skip_days = implode(',', array_map('intval', $recurring_days));
            } else {
                echo json_encode(['success' => false, 'message' => 'At least one repeat day must be selected for recurring tasks']);
                exit;
            }
        }
        
        // Generate unique task number
        $task_number = generateTaskNumber();
        
        $stmt = $conn->prepare("
            INSERT INTO tasks (task_number, name, description, priority, start_date, end_date, start_time, end_time, task_type, 
                            recurring_interval, skip_days, task_points, quality_points, 
                            lost_task_points, notes, created_by, is_enabled)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1)
        ");
        $stmt->bind_param("sssssssssisiiisi", $task_number, $name, $description, $priority, $start_date, $end_date, 
                         $start_time, $end_time, $task_type, $recurring_interval, $skip_days, $task_points, $quality_points, 
                         $lost_task_points, $notes, $user_id);
        
        if ($stmt->execute()) {
            $task_id = $conn->insert_id;
            
            // Assign users
            if (!empty($assigned_users)) {
                $assign_stmt = $conn->prepare("INSERT INTO task_assignments (task_id, user_id) VALUES (?, ?)");
                foreach ($assigned_users as $uid) {
                    $assign_stmt->bind_param("ii", $task_id, $uid);
                    $assign_stmt->execute();
                    
                    // Create notification
                    $notif_stmt = $conn->prepare("
                        INSERT INTO notifications (user_id, title, message, type)
                        VALUES (?, 'New Task Assigned', ?, 'info')
                    ");
                    $message = "You have been assigned a new task: " . $name;
                    $notif_stmt->bind_param("is", $uid, $message);
                    $notif_stmt->execute();
                }
                $assign_stmt->close();
            }
            
            echo json_encode(['success' => true, 'message' => 'Task created successfully', 'task_id' => $task_id]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to create task']);
        }
        break;
        
    case 'PUT':
        // Update task (admin only)
        if (!$is_admin) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Forbidden']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $task_id = $data['id'] ?? null;
        
        if (!$task_id) {
            echo json_encode(['success' => false, 'message' => 'Task ID is required']);
            exit;
        }
        
        // Similar to POST but UPDATE instead of INSERT
        $name = $data['name'] ?? '';
        $description = $data['description'] ?? '';
        $priority = $data['priority'] ?? 'medium';
        $start_date = $data['start_date'] ?? date('Y-m-d');
        $end_date = $data['end_date'] ?? null;
        $start_time = $data['start_time'] ?? null;
        $end_time = $data['end_time'] ?? null;
        $task_type = $data['task_type'] ?? 'one-time';
        $recurring_interval = null; // No longer used, kept for backward compatibility
        $recurring_days = $data['recurring_days'] ?? [];
        $skip_days = null; // Repurposed: now stores the days to repeat on
        $task_points = $data['task_points'] ?? 1;
        $quality_points = $data['quality_points'] ?? 10;
        $lost_task_points = $data['lost_task_points'] ?? -1;
        $notes = $data['notes'] ?? null;
        $assigned_users = $data['assigned_users'] ?? [];
        
        // Handle recurring days - store in skip_days field (repurposed)
        if ($task_type === 'recurring') {
            if (is_array($recurring_days) && !empty($recurring_days)) {
                $skip_days = implode(',', array_map('intval', $recurring_days));
            } else {
                echo json_encode(['success' => false, 'message' => 'At least one repeat day must be selected for recurring tasks']);
                exit;
            }
        }
        
        $stmt = $conn->prepare("
            UPDATE tasks SET name=?, description=?, priority=?, start_date=?, end_date=?, start_time=?, end_time=?, 
                           task_type=?, recurring_interval=?, skip_days=?, task_points=?, 
                           quality_points=?, lost_task_points=?, notes=?
            WHERE id=?
        ");
        $stmt->bind_param("ssssssssisiiisi", $name, $description, $priority, $start_date, $end_date, 
                         $start_time, $end_time, $task_type, $recurring_interval, $skip_days, $task_points, $quality_points, 
                         $lost_task_points, $notes, $task_id);
        
        if ($stmt->execute()) {
            // Update assignments
            $conn->query("DELETE FROM task_assignments WHERE task_id = $task_id");
            if (!empty($assigned_users)) {
                $assign_stmt = $conn->prepare("INSERT INTO task_assignments (task_id, user_id) VALUES (?, ?)");
                foreach ($assigned_users as $uid) {
                    $assign_stmt->bind_param("ii", $task_id, $uid);
                    $assign_stmt->execute();
                }
                $assign_stmt->close();
            }
            
            echo json_encode(['success' => true, 'message' => 'Task updated successfully']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to update task']);
        }
        break;
        
    case 'DELETE':
        // Delete task (admin only)
        if (!$is_admin) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Forbidden']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $task_id = $data['id'] ?? null;
        $admin_password = $data['admin_password'] ?? '';
        
        if (!$task_id) {
            echo json_encode(['success' => false, 'message' => 'Task ID is required']);
            exit;
        }
        
        // Verify admin password
        $admin_stmt = $conn->prepare("SELECT password FROM users WHERE id = ? AND role = 'admin'");
        $admin_stmt->bind_param("i", $user_id);
        $admin_stmt->execute();
        $admin_result = $admin_stmt->get_result();
        $admin = $admin_result->fetch_assoc();
        
        if (!$admin || $admin_password !== $admin['password']) {
            echo json_encode(['success' => false, 'message' => 'Invalid admin password']);
            exit;
        }
        
        $stmt = $conn->prepare("DELETE FROM tasks WHERE id = ?");
        $stmt->bind_param("i", $task_id);
        
        if ($stmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'Task deleted successfully']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to delete task']);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
}

$conn->close();
?>
