<?php
require_once '../config.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$conn = getDBConnection();
$method = $_SERVER['REQUEST_METHOD'];
$is_admin = isAdmin();

switch ($method) {
    case 'GET':
        // Get users list (admin only) or current user info
        if ($is_admin) {
            $stmt = $conn->query("
                SELECT u.*,
                       (SELECT COUNT(*) FROM task_assignments WHERE user_id = u.id) as total_assigned_works,
                       (SELECT COUNT(*) FROM task_assignments ta
                        JOIN tasks t ON ta.task_id = t.id
                        WHERE ta.user_id = u.id 
                        AND t.start_date <= CURDATE() 
                        AND (t.end_date IS NULL OR t.end_date >= CURDATE())
                        AND NOT EXISTS (
                            SELECT 1 FROM task_completions tc 
                            WHERE tc.task_id = t.id 
                            AND tc.user_id = u.id 
                            AND tc.status = 'completed'
                        )
                        AND NOT EXISTS (
                            SELECT 1 FROM user_leaves ul 
                            WHERE ul.user_id = u.id 
                            AND ul.leave_date = CURDATE()
                        )) as today_assigned_works,
                       (SELECT COUNT(*) FROM task_assignments ta
                        JOIN tasks t ON ta.task_id = t.id
                        WHERE ta.user_id = u.id 
                        AND t.start_date <= DATE_ADD(CURDATE(), INTERVAL 1 DAY)
                        AND (t.end_date IS NULL OR t.end_date >= DATE_ADD(CURDATE(), INTERVAL 1 DAY))
                        AND NOT EXISTS (
                            SELECT 1 FROM task_completions tc 
                            WHERE tc.task_id = t.id 
                            AND tc.user_id = u.id 
                            AND tc.status = 'completed'
                        )
                        AND NOT EXISTS (
                            SELECT 1 FROM user_leaves ul 
                            WHERE ul.user_id = u.id 
                            AND ul.leave_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
                        )) as tomorrow_assigned_works
                FROM users u
                ORDER BY u.role ASC, u.created_at DESC
            ");
            
            // Count total admins
            $admin_count_result = $conn->query("SELECT COUNT(*) as count FROM users WHERE role = 'admin'");
            $admin_count = $admin_count_result->fetch_assoc()['count'] ?? 0;
            
            $users = [];
            while ($row = $stmt->fetch_assoc()) {
                // Get leave days (today and future dates only)
                $leave_stmt = $conn->prepare("
                    SELECT leave_date 
                    FROM user_leaves 
                    WHERE user_id = ? AND leave_date >= CURDATE()
                    ORDER BY leave_date ASC
                ");
                $leave_stmt->bind_param("i", $row['id']);
                $leave_stmt->execute();
                $leave_result = $leave_stmt->get_result();
                $leave_dates = [];
                while ($leave_row = $leave_result->fetch_assoc()) {
                    $leave_dates[] = $leave_row['leave_date'];
                }
                $leave_stmt->close();
                
                $row['leave_dates'] = $leave_dates;
                $row['is_admin'] = ($row['role'] === 'admin');
                $row['can_delete'] = !($row['role'] === 'admin' && $admin_count <= 1);
                $users[] = $row;
            }
            
            echo json_encode(['success' => true, 'users' => $users]);
        } else {
            // Get current user info
            $user_id = getCurrentUserId();
            $stmt = $conn->prepare("
                SELECT u.*, 
                       COALESCE(up.total_task_points, 0) as total_task_points,
                       COALESCE(up.total_quality_points, 0) as total_quality_points,
                       COALESCE(up.total_lost_task_points, 0) as total_lost_task_points
                FROM users u
                LEFT JOIN user_points up ON u.id = up.user_id
                WHERE u.id = ?
            ");
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();
            
            echo json_encode(['success' => true, 'user' => $user]);
        }
        break;
        
    case 'POST':
        // Create new user (admin only)
        if (!$is_admin) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Forbidden']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        $username = $data['username'] ?? '';
        $password = $data['password'] ?? '';
        $role = $data['role'] ?? 'user';
        $email = $data['email'] ?? null;
        $full_name = $data['full_name'] ?? null;
        
        if (empty($username) || empty($password)) {
            echo json_encode(['success' => false, 'message' => 'Username and password are required']);
            exit;
        }
        
        // Check if username exists
        $check_stmt = $conn->prepare("SELECT id FROM users WHERE username = ?");
        $check_stmt->bind_param("s", $username);
        $check_stmt->execute();
        if ($check_stmt->get_result()->num_rows > 0) {
            echo json_encode(['success' => false, 'message' => 'Username already exists']);
            exit;
        }
        
        $stmt = $conn->prepare("
            INSERT INTO users (username, password, role, email, full_name)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->bind_param("sssss", $username, $password, $role, $email, $full_name);
        
        if ($stmt->execute()) {
            $user_id = $conn->insert_id;
            
            // Initialize user points
            $points_stmt = $conn->prepare("INSERT INTO user_points (user_id) VALUES (?)");
            $points_stmt->bind_param("i", $user_id);
            $points_stmt->execute();
            
            echo json_encode(['success' => true, 'message' => 'User created successfully', 'user_id' => $user_id]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to create user']);
        }
        break;
        
    case 'PUT':
        // Update user (admin only)
        if (!$is_admin) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Forbidden']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $user_id = $data['id'] ?? null;
        
        if (!$user_id) {
            echo json_encode(['success' => false, 'message' => 'User ID is required']);
            exit;
        }
        
        $username = $data['username'] ?? '';
        $password = $data['password'] ?? '';
        $role = $data['role'] ?? 'user';
        $email = $data['email'] ?? null;
        $full_name = $data['full_name'] ?? null;
        
        if (empty($username)) {
            echo json_encode(['success' => false, 'message' => 'Username is required']);
            exit;
        }
        
        // Check if username exists (excluding current user)
        $check_stmt = $conn->prepare("SELECT id FROM users WHERE username = ? AND id != ?");
        $check_stmt->bind_param("si", $username, $user_id);
        $check_stmt->execute();
        if ($check_stmt->get_result()->num_rows > 0) {
            echo json_encode(['success' => false, 'message' => 'Username already exists']);
            exit;
        }
        
        if (!empty($password)) {
            $stmt = $conn->prepare("
                UPDATE users SET username=?, password=?, role=?, email=?, full_name=?
                WHERE id=?
            ");
            $stmt->bind_param("sssssi", $username, $password, $role, $email, $full_name, $user_id);
        } else {
            $stmt = $conn->prepare("
                UPDATE users SET username=?, role=?, email=?, full_name=?
                WHERE id=?
            ");
            $stmt->bind_param("ssssi", $username, $role, $email, $full_name, $user_id);
        }
        
        if ($stmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'User updated successfully']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to update user']);
        }
        break;
        
    case 'DELETE':
        // Delete user (admin only)
        if (!$is_admin) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Forbidden']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $user_id_to_delete = $data['id'] ?? null;
        $admin_password = $data['admin_password'] ?? '';
        
        if (!$user_id_to_delete) {
            echo json_encode(['success' => false, 'message' => 'User ID is required']);
            exit;
        }
        
        // Verify admin password
        $current_user_id = getCurrentUserId();
        $admin_stmt = $conn->prepare("SELECT password FROM users WHERE id = ? AND role = 'admin'");
        $admin_stmt->bind_param("i", $current_user_id);
        $admin_stmt->execute();
        $admin_result = $admin_stmt->get_result();
        $admin = $admin_result->fetch_assoc();
        
        if (!$admin || $admin_password !== $admin['password']) {
            echo json_encode(['success' => false, 'message' => 'Invalid admin password']);
            exit;
        }
        
        $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
        $stmt->bind_param("i", $user_id_to_delete);
        
        if ($stmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'User deleted successfully']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to delete user']);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
}

$conn->close();
?>
