// Minimal JavaScript - Only for SPA navigation, theme toggle, modals, and native behaviors

// Custom Toast Notification System
function showToast(message, type = 'success', title = null) {
    // Remove default browser alerts
    const toastContainer = getOrCreateToastContainer();
    
    const toast = document.createElement('div');
    toast.className = `toast ${type}`;
    
    const icons = {
        success: '✅',
        error: '❌',
        info: 'ℹ️',
        warning: '⚠️'
    };
    
    const defaultTitles = {
        success: 'Success',
        error: 'Error',
        info: 'Info',
        warning: 'Warning'
    };
    
    toast.innerHTML = `
        <span class="toast-icon">${icons[type] || icons.info}</span>
        <div class="toast-content">
            ${title ? `<div class="toast-title">${escapeHtml(title)}</div>` : ''}
            <div class="toast-message">${escapeHtml(message)}</div>
        </div>
        <button class="toast-close" onclick="this.parentElement.remove()">&times;</button>
    `;
    
    toastContainer.appendChild(toast);
    
    // Auto remove after 2 seconds (2000ms)
    setTimeout(() => {
        toast.classList.add('hiding');
        setTimeout(() => {
            if (toast.parentElement) {
                toast.remove();
            }
        }, 300);
    }, 2000);
    
    return toast;
}

function getOrCreateToastContainer() {
    let container = document.getElementById('toastContainer');
    if (!container) {
        container = document.createElement('div');
        container.id = 'toastContainer';
        container.className = 'toast-container';
        document.body.appendChild(container);
    }
    return container;
}

// Replace all alert() calls with custom toast
function showSuccessAlert(message) {
    showToast(message, 'success');
}

function showErrorAlert(message) {
    showToast(message, 'error');
}

function showInfoAlert(message) {
    showToast(message, 'info');
}

// Store native confirm before overriding (must be before showIOSConfirm)
const nativeConfirm = window.confirm.bind(window);

// iOS Style Confirmation Dialog (replaces confirm())
function showIOSConfirm(message, title = 'Confirm') {
    return new Promise((resolve) => {
        const modal = document.getElementById('iosConfirmModal');
        const titleEl = document.getElementById('iosConfirmTitle');
        const messageEl = document.getElementById('iosConfirmMessage');
        const cancelBtn = document.getElementById('iosConfirmCancel');
        const okBtn = document.getElementById('iosConfirmOk');
        
        if (!modal || !titleEl || !messageEl || !cancelBtn || !okBtn) {
            // Fallback to native browser confirm if modal not found
            resolve(nativeConfirm(message));
            return;
        }
        
        // Set content
        titleEl.textContent = title;
        messageEl.textContent = message;
        
        // Remove existing listeners
        const newCancelBtn = cancelBtn.cloneNode(true);
        const newOkBtn = okBtn.cloneNode(true);
        cancelBtn.parentNode.replaceChild(newCancelBtn, cancelBtn);
        okBtn.parentNode.replaceChild(newOkBtn, okBtn);
        
        // Add new listeners
        newCancelBtn.addEventListener('click', () => {
            closeModal('iosConfirmModal');
            resolve(false);
        });
        
        newOkBtn.addEventListener('click', () => {
            closeModal('iosConfirmModal');
            resolve(true);
        });
        
        // Open modal
        openModal('iosConfirmModal');
    });
}

// Override browser confirm() globally - returns Promise
// Note: This changes confirm() from synchronous to asynchronous
// All confirm() calls should use await or .then()
window.confirm = function(message) {
    // Use iOS style confirm
    return showIOSConfirm(message, 'Confirm');
};

// Override browser alert() globally
window.alert = function(message) {
    // Use toast instead
    showToast(message, 'info');
};

// Theme Management (localStorage - native behavior)
function initTheme() {
    const savedTheme = localStorage.getItem('theme') || 'light';
    document.documentElement.setAttribute('data-theme', savedTheme);
    updateThemeIcon(savedTheme);
}

function toggleTheme() {
    const currentTheme = document.documentElement.getAttribute('data-theme');
    const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
    document.documentElement.setAttribute('data-theme', newTheme);
    localStorage.setItem('theme', newTheme);
    updateThemeIcon(newTheme);
}

function updateThemeIcon(theme) {
    const themeIcon = document.getElementById('themeIcon');
    if (themeIcon) {
        themeIcon.textContent = theme === 'dark' ? '☀️' : '🌙';
    }
}

// Helper function to refresh current page data
function refreshCurrentPage() {
    if (currentPage) {
        console.log('Refreshing page:', currentPage);
        // Special handling for admin dashboard - refresh users list
        if (currentPage === 'admin-dashboard') {
            if (typeof loadAdminUsers === 'function') {
                loadAdminUsersRetryCount = 0;
                loadAdminUsers();
            }
            // Also reload the page to refresh stats and performance
            loadPage('admin-dashboard');
        } else {
            loadPage(currentPage);
        }
    }
}

// SPA Router - Minimal page loading (only if needed for smooth transitions)
let currentPage = 'dashboard';
const pageRoutes = {
    'dashboard': 'user/user-dashboard.php',
    'tasks': 'user/user-tasks.php',
    'progress': 'user/user-progress.php',
    'history': 'user/user-history.php',
    'admin-dashboard': 'admin/admin-dashboard.php',
    'admin-tasks': 'admin/admin-tasks.php',
    'admin-tasks-disabled': 'admin/admin-tasks-disabled.php',
    'admin-users': 'admin/admin-users.php',
    'admin-reports': 'admin/admin-reports.php'
};

async function loadPage(pageId) {
    const pageContainer = document.getElementById('pageContainer');
    if (!pageContainer) return;

    // Handle query parameters in pageId (e.g., "progress?period=today")
    let actualPageId = pageId;
    let queryString = '';
    if (pageId.includes('?')) {
        const parts = pageId.split('?');
        actualPageId = parts[0];
        queryString = '?' + parts[1];
    }

    const pagePath = pageRoutes[actualPageId];
    if (!pagePath) {
        console.error(`Page not found: ${actualPageId}`);
        return;
    }

    try {
        pageContainer.innerHTML = '<div class="empty-state"><div class="empty-state-icon">⏳</div><h3>Loading...</h3></div>';
        
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000);
        
        const response = await fetch(pagePath + queryString, { signal: controller.signal });
        clearTimeout(timeoutId);
        
        if (!response.ok) {
            throw new Error(`Failed to load page: ${pagePath}`);
        }
        
        const html = await response.text();
        pageContainer.innerHTML = html;
        currentPage = actualPageId;
        
        // Update URL hash to persist page on refresh (without query params)
        // Only update hash if we're in SPA mode (not on standalone pages)
        if (pageContainer && window.location.hash !== `#${actualPageId}`) {
            window.location.hash = actualPageId;
        }
        
        // Set active nav link
        const activeLink = document.querySelector(`[data-page="${actualPageId}"]`);
        if (activeLink) {
            if (activeLink.classList.contains('nav-link')) {
                setActiveNavLink(activeLink);
            } else if (activeLink.classList.contains('bottom-nav-link')) {
                setActiveBottomNavLink(activeLink);
            }
        }
        
        // Re-initialize countdown timers if on dashboard or tasks page
        if (pageId === 'dashboard' || pageId === 'tasks' || pageId === 'admin-dashboard') {
            setTimeout(() => initCountdownTimers(), 100);
        }
        
        // Initialize page-specific functionality
        setTimeout(() => {
            if (actualPageId === 'progress') {
                initProgressPage();
            } else if (actualPageId === 'history') {
                initHistoryPage();
            } else if (actualPageId === 'tasks') {
                initTasksPage();
            }
        }, 100);
        
        // Load users for admin dashboard
        if (pageId === 'admin-dashboard') {
            // Reset retry count when loading admin dashboard
            loadAdminUsersRetryCount = 0;
            // Use requestAnimationFrame to ensure DOM is ready
            requestAnimationFrame(() => {
                setTimeout(() => {
                    loadAdminUsers();
                    
                    // Get the selected filter value (default to 'today')
                    const performanceFilter = document.getElementById('performanceFilter');
                    const selectedFilter = performanceFilter ? performanceFilter.value : 'today';
                    
                    // Ensure dropdown is set to 'today' if not already set
                    if (performanceFilter && !performanceFilter.value) {
                        performanceFilter.value = 'today';
                    }
                    
                    // Load performance data with the selected filter
                    loadUserPerformance(selectedFilter);
                    
                    // Attach performance filter event listener with retry mechanism
                    let retryCount = 0;
                    const maxRetries = 10;
                    const attachPerformanceFilter = () => {
                        const performanceFilter = document.getElementById('performanceFilter');
                        if (performanceFilter) {
                            // Remove any existing handler first
                            if (performanceFilter._filterChangeHandler) {
                                performanceFilter.removeEventListener('change', performanceFilter._filterChangeHandler);
                            }
                            
                            // Create and attach new handler
                            const handleFilterChange = function(e) {
                                const filter = e.target.value;
                                console.log('Performance filter changed to:', filter);
                                loadUserPerformance(filter);
                            };
                            
                            performanceFilter.addEventListener('change', handleFilterChange);
                            performanceFilter._filterChangeHandler = handleFilterChange;
                            console.log('Performance filter event listener attached successfully');
                        } else if (retryCount < maxRetries) {
                            // Retry if element not found yet
                            retryCount++;
                            setTimeout(attachPerformanceFilter, 100);
                        } else {
                            console.error('Failed to attach performance filter listener after', maxRetries, 'retries');
                        }
                    };
                    
                    attachPerformanceFilter();
                }, 50);
            });
        }
        
    } catch (error) {
        console.error('Error loading page:', error);
        pageContainer.innerHTML = `
            <div class="empty-state">
                <div class="empty-state-icon">❌</div>
                <h3>Error Loading Page</h3>
                <p>${error.message}</p>
                <button class="btn btn-primary" onclick="loadPage('${pageId}')">Retry</button>
            </div>
        `;
    }
}

// Navigation Management
function initNavigation() {
    // Check if we're in SPA mode (has pageContainer) or standalone page mode
    const pageContainer = document.getElementById('pageContainer');
    const isSPAMode = pageContainer !== null;
    
    const navLinks = document.querySelectorAll('.nav-link');
    navLinks.forEach(link => {
        link.addEventListener('click', (e) => {
            const page = link.getAttribute('data-page');
            const href = link.getAttribute('href');
            
            // If in SPA mode and has data-page, use SPA navigation
            if (isSPAMode && page) {
                e.preventDefault();
                loadPage(page);
                setActiveNavLink(link);
            }
            // If standalone page mode (has href but no data-page), allow normal navigation
            else if (!isSPAMode && href && !href.startsWith('#')) {
                // Allow normal navigation - don't prevent default
                // The browser will handle the navigation
            }
            // If has data-page but we're in standalone mode, convert to href navigation
            else if (!isSPAMode && page && pageRoutes[page]) {
                e.preventDefault();
                window.location.href = pageRoutes[page];
            }
            // Fallback: if has href, allow normal navigation
            else if (href && !href.startsWith('#')) {
                // Allow normal navigation
            }
            // Otherwise prevent default for hash links
            else if (!href || href.startsWith('#')) {
                e.preventDefault();
            }
        });
    });

    const bottomNavLinks = document.querySelectorAll('.bottom-nav-link');
    bottomNavLinks.forEach(link => {
        link.addEventListener('click', (e) => {
            const page = link.getAttribute('data-page');
            const href = link.getAttribute('href');
            
            // If in SPA mode and has data-page, use SPA navigation
            if (isSPAMode && page) {
                e.preventDefault();
                loadPage(page);
                setActiveBottomNavLink(link);
            }
            // If standalone page mode (has href but no data-page), allow normal navigation
            else if (!isSPAMode && href && !href.startsWith('#')) {
                // Allow normal navigation - don't prevent default
                // The browser will handle the navigation
            }
            // If has data-page but we're in standalone mode, convert to href navigation
            else if (!isSPAMode && page && pageRoutes[page]) {
                e.preventDefault();
                window.location.href = pageRoutes[page];
            }
            // Fallback: if has href, allow normal navigation
            else if (href && !href.startsWith('#')) {
                // Allow normal navigation
            }
            // Otherwise prevent default for hash links
            else if (!href || href.startsWith('#')) {
                e.preventDefault();
            }
        });
    });
    
    // Listen for hash changes (back/forward buttons) - only in SPA mode
    if (isSPAMode) {
        window.addEventListener('hashchange', () => {
            const hash = window.location.hash.substring(1);
            if (hash && pageRoutes[hash] && hash !== currentPage) {
                loadPage(hash);
            }
        });
    }
}

function setActiveNavLink(activeLink) {
    document.querySelectorAll('.nav-link').forEach(link => link.classList.remove('active'));
    if (activeLink) activeLink.classList.add('active');
}

function setActiveBottomNavLink(activeLink) {
    document.querySelectorAll('.bottom-nav-link').forEach(link => link.classList.remove('active'));
    if (activeLink) activeLink.classList.add('active');
}

// Modal Management (native behavior)
function initModals() {
    // Close modals when clicking outside
    document.querySelectorAll('.modal').forEach(modal => {
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                closeModal(modal.id);
            }
        });
    });

    // Close buttons
    document.querySelectorAll('.modal-close').forEach(btn => {
        btn.addEventListener('click', (e) => {
            const modal = e.target.closest('.modal');
            if (modal) closeModal(modal.id);
        });
    });

    // Image preview for completion form
    const completionImage = document.getElementById('completionImage');
    const imagePreview = document.getElementById('imagePreview');
    const previewImg = document.getElementById('previewImg');
    const removeImageBtn = document.getElementById('removeImage');

    if (completionImage && previewImg) {
        completionImage.addEventListener('change', (e) => {
            const file = e.target.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = (event) => {
                    previewImg.src = event.target.result;
                    if (imagePreview) imagePreview.style.display = 'block';
                };
                reader.readAsDataURL(file);
            }
        });
    }

    if (removeImageBtn && completionImage) {
        removeImageBtn.addEventListener('click', () => {
            completionImage.value = '';
            if (imagePreview) imagePreview.style.display = 'none';
            if (previewImg) previewImg.src = '';
        });
    }
}

function openModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.add('active');
        document.body.style.overflow = 'hidden';
    }
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.remove('active');
        document.body.style.overflow = '';
        
        // Reset forms when closing modals
        if (modalId === 'addTaskModal') {
            const form = document.getElementById('addTaskForm');
            if (form) {
                form.reset();
                form.action = 'add_task.php';
                document.getElementById('hiddenTaskId').value = '';
                document.getElementById('taskModalTitle').textContent = 'Add New Task';
                document.getElementById('submitTaskBtn').textContent = 'Create Task';
                const recurringDaysGroup = document.getElementById('recurringDaysGroup');
                if (recurringDaysGroup) recurringDaysGroup.style.display = 'none';
            }
        }
        
        if (modalId === 'addUserModal') {
            const form = document.getElementById('addUserForm');
            if (form) {
                form.reset();
                form.action = 'add_user.php';
                document.getElementById('hiddenUserId').value = '';
                document.getElementById('addUserModalTitle').textContent = 'Add New User';
                document.getElementById('submitUserBtn').textContent = 'Create User';
                document.getElementById('newPasswordLabel').textContent = 'Password *';
                document.getElementById('newPassword').setAttribute('required', 'required');
                document.getElementById('passwordHint').style.display = 'none';
            }
        }
        
        if (modalId === 'deleteTaskModal') {
            const form = document.getElementById('deleteTaskForm');
            if (form) {
                form.reset();
                document.getElementById('hiddenDeleteTaskId').value = '';
            }
        }
        
        if (modalId === 'disableTaskModal') {
            const form = document.getElementById('disableTaskForm');
            if (form) {
                form.reset();
                document.getElementById('hiddenDisableTaskId').value = '';
                const disableReason = document.getElementById('disableReason');
                if (disableReason) disableReason.value = '';
            }
        }
        
        if (modalId === 'enableTaskModal') {
            const form = document.getElementById('enableTaskForm');
            if (form) {
                form.reset();
                document.getElementById('hiddenEnableTaskId').value = '';
            }
        }
        
        if (modalId === 'deleteUserModal') {
            const form = document.getElementById('deleteUserForm');
            if (form) {
                form.reset();
                document.getElementById('hiddenDeleteUserId').value = '';
            }
        }
        
        if (modalId === 'leaveModal') {
            const form = document.getElementById('addLeaveForm');
            if (form) {
                form.reset();
                document.getElementById('hiddenLeaveUserId').value = '';
                document.getElementById('leavesList').innerHTML = '<p style="color: #999; padding: 10px;">Loading...</p>';
            }
        }
        
        if (modalId === 'markCompleteModal') {
            const form = document.getElementById('markCompleteForm');
            const hiddenTaskId = document.getElementById('hiddenCompleteTaskId');
            const existingTaskId = hiddenTaskId ? hiddenTaskId.value : '';
            if (form) {
                form.reset();
                if (hiddenTaskId) {
                    hiddenTaskId.value = existingTaskId;
                }
                const imagePreview = document.getElementById('imagePreview');
                const previewImg = document.getElementById('previewImg');
                if (imagePreview) imagePreview.style.display = 'none';
                if (previewImg) previewImg.src = '';
            }
        }
        
        if (modalId === 'userTasksModal') {
            const modalBody = document.getElementById('userTasksModalBody');
            if (modalBody) {
                modalBody.innerHTML = '<div class="loading-spinner"><div class="spinner"></div><p>Loading tasks...</p></div>';
            }
        }
    }
}

// Remaining Time Display (replaces countdown)
let remainingTimeIntervals = [];

function initCountdownTimers() {
    // Clear existing intervals
    remainingTimeIntervals.forEach(interval => clearInterval(interval));
    remainingTimeIntervals = [];

    // Remaining time displays with date and time
    const remainingTimeDisplays = document.querySelectorAll('.countdown-timer[data-end-date]');
    remainingTimeDisplays.forEach(timer => {
        const taskCard = timer.closest('.task-card');
        const taskId = taskCard?.getAttribute('data-task-id');
        const taskType = taskCard?.getAttribute('data-task-type') || 'one-time';
        const skipDays = taskCard?.getAttribute('data-skip-days') || '';
        const taskEndDate = taskCard?.getAttribute('data-task-end-date') || '';
        const effectiveDate = taskCard?.getAttribute('data-effective-date') || '';
        
        let endDate = taskCard?.getAttribute('data-end-date');
        let endTime = taskCard?.getAttribute('data-end-time') || '';
        
        if (!endDate) return;

        // Helper function to calculate next recurring occurrence date
        function getNextRecurringDate(instanceDate, recurringDays, taskEndDateStr) {
            if (!recurringDays || !taskEndDateStr) return null;
            
            const days = recurringDays.split(',').map(d => parseInt(d.trim())).filter(d => !isNaN(d));
            if (days.length === 0) return null;
            
            const now = typeof getKolkataTime === 'function' ? getKolkataTime() : new Date();
            const today = new Date(now);
            today.setHours(0, 0, 0, 0);
            
            const taskEndDate = new Date(taskEndDateStr + 'T23:59:59');
            if (taskEndDate < today) return null; // Task has ended
            
            // If instanceDate exists and is today or in the future, use it
            if (instanceDate) {
                const instanceDateObj = new Date(instanceDate + 'T00:00:00');
                instanceDateObj.setHours(0, 0, 0, 0);
                const instanceDayOfWeek = instanceDateObj.getDay();
                
                // If instance date is today or in the future and is a recurring day, use it
                if (instanceDateObj >= today && days.includes(instanceDayOfWeek) && instanceDateObj <= taskEndDate) {
                    return instanceDate;
                }
            }
            
            // Check if today is a recurring day
            const todayDayOfWeek = today.getDay(); // 0 = Sunday, 1 = Monday, etc.
            if (days.includes(todayDayOfWeek)) {
                // Today is a recurring day, use today's date
                return today.toISOString().split('T')[0];
            }
            
            // Find next recurring day
            let nextDate = new Date(today);
            let attempts = 0;
            const maxAttempts = 14; // Check up to 2 weeks ahead
            
            while (attempts < maxAttempts) {
                nextDate.setDate(nextDate.getDate() + 1);
                const dayOfWeek = nextDate.getDay();
                
                if (days.includes(dayOfWeek) && nextDate <= taskEndDate) {
                    return nextDate.toISOString().split('T')[0];
                }
                attempts++;
            }
            
            return null;
        }

        const updateRemainingTime = () => {
            // Get current time in Kolkata timezone
            const nowKolkata = typeof getKolkataTime === 'function' ? getKolkataTime() : new Date();
            
            // For recurring tasks, calculate the next occurrence date
            let targetEndDate = endDate;
            let targetEndTime = endTime;
            
            if (taskType === 'recurring' && skipDays && taskEndDate) {
                const nextOccurrenceDate = getNextRecurringDate(effectiveDate, skipDays, taskEndDate);
                if (nextOccurrenceDate) {
                    targetEndDate = nextOccurrenceDate;
                    // For recurring tasks, always use the task's end_time
                    // Get it from the original data attribute if not already set
                    if (taskCard) {
                        const originalEndTime = taskCard.getAttribute('data-end-time');
                        if (originalEndTime) {
                            targetEndTime = originalEndTime;
                        }
                    }
                }
            }
            
            // Store targetEndTime for use in end time display update
            const currentTargetEndTime = targetEndTime;
            
            // Build end date/time - treat stored time as Kolkata time
            // Ensure we use the correct date format and time format
            let endDateTime;
            if (targetEndTime) {
                // Ensure time is in HH:MM:SS format
                let timeStr = targetEndTime;
                if (timeStr.length === 5) {
                    // If time is HH:MM, add seconds
                    timeStr = timeStr + ':00';
                }
                
                if (typeof createKolkataDate === 'function') {
                    endDateTime = createKolkataDate(targetEndDate, timeStr);
                } else {
                    // Fallback: create date string in ISO format
                    const dateTimeStr = targetEndDate + 'T' + timeStr;
                    endDateTime = new Date(dateTimeStr);
                }
            } else {
                if (typeof createKolkataDate === 'function') {
                    endDateTime = createKolkataDate(targetEndDate, '23:59:59');
                } else {
                    endDateTime = new Date(targetEndDate + 'T23:59:59');
                }
            }
            
            // Debug: Log the calculation (remove in production)
            // console.log('Now:', nowKolkata, 'End:', endDateTime, 'Target Date:', targetEndDate, 'Target Time:', targetEndTime);
            
            const distance = endDateTime.getTime() - nowKolkata.getTime();
            const countdownValue = timer.querySelector('.countdown-value');
            
            // Debug logging - always log to help diagnose
            if (countdownValue) {
                // Both Date objects now store UTC internally (no offset manipulation)
                const nowUTCDate = new Date(nowKolkata.getTime());
                const endUTCDate = new Date(endDateTime.getTime());
                
                // Get IST times for display
                const nowIST = nowUTCDate.toLocaleTimeString('en-IN', {
                    timeZone: 'Asia/Kolkata',
                    hour: '2-digit',
                    minute: '2-digit',
                    hour12: false
                });
                const endIST = endUTCDate.toLocaleTimeString('en-IN', {
                    timeZone: 'Asia/Kolkata',
                    hour: '2-digit',
                    minute: '2-digit',
                    hour12: false
                });
                
                console.log('Time Debug:', {
                    'Now UTC': nowUTCDate.toISOString(),
                    'End UTC': endUTCDate.toISOString(),
                    'Now UTC hours': String(nowUTCDate.getUTCHours()).padStart(2, '0') + ':' + String(nowUTCDate.getUTCMinutes()).padStart(2, '0') + ' UTC',
                    'End UTC hours': String(endUTCDate.getUTCHours()).padStart(2, '0') + ':' + String(endUTCDate.getUTCMinutes()).padStart(2, '0') + ' UTC',
                    'Now IST': nowIST,
                    'End IST (should be 23:30)': endIST,
                    'Target Date': targetEndDate,
                    'Target Time': targetEndTime,
                    'Distance (ms)': distance,
                    'Distance (hours)': (distance / (1000 * 60 * 60)).toFixed(2)
                });
            }

            if (distance < 0) {
                // Time expired - keep countdown visible but do not auto-fail
                if (countdownValue) {
                    countdownValue.textContent = '0s';
                }
                timer.classList.remove('remaining-time-orange');
                timer.classList.add('remaining-time-red');
                return;
            }

            // Calculate total hours remaining for color coding
            const totalHours = distance / (1000 * 60 * 60);
            
            // Calculate time components for live countdown
            const days = Math.floor(distance / (1000 * 60 * 60 * 24));
            const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((distance % (1000 * 60)) / 1000);

            // Format live countdown text (e.g., "1d 19h 30m 45s")
            let timeText = '';
            if (days > 0) {
                timeText = `${days}d ${String(hours).padStart(2, '0')}h ${String(minutes).padStart(2, '0')}m ${String(seconds).padStart(2, '0')}s`;
            } else if (hours > 0) {
                timeText = `${hours}h ${String(minutes).padStart(2, '0')}m ${String(seconds).padStart(2, '0')}s`;
            } else if (minutes > 0) {
                timeText = `${minutes}m ${String(seconds).padStart(2, '0')}s`;
            } else {
                timeText = `${seconds}s`;
            }

            if (countdownValue) {
                countdownValue.textContent = timeText;
                
                // Update end time display if it exists
                const endTimeDisplay = timer.querySelector('.countdown-end-time');
                if (endTimeDisplay && currentTargetEndTime) {
                    // Format the end time for display
                    const endTimeParts = currentTargetEndTime.split(':');
                    if (endTimeParts.length >= 2) {
                        const hours = parseInt(endTimeParts[0]);
                        const minutes = parseInt(endTimeParts[1]);
                        const period = hours >= 12 ? 'PM' : 'AM';
                        const displayHours = hours > 12 ? hours - 12 : (hours === 0 ? 12 : hours);
                        const displayMinutes = String(minutes).padStart(2, '0');
                        const formattedTime = `${displayHours}:${displayMinutes} ${period}`;
                        
                        const endTimeText = endTimeDisplay.querySelector('.countdown-end-text');
                        if (endTimeText) {
                            endTimeText.textContent = formattedTime;
                        }
                    }
                }
                
                // Apply color classes based on remaining time
                // < 2 hours: red, < 5 hours: orange, >= 5 hours: normal black
                timer.classList.remove('remaining-time-red', 'remaining-time-orange');
                if (totalHours < 2) {
                    timer.classList.add('remaining-time-red');
                } else if (totalHours < 5) {
                    timer.classList.add('remaining-time-orange');
                }
                // If >= 5 hours, no class is added, so it uses default black color
            }
        };

        updateRemainingTime();
        const interval = setInterval(updateRemainingTime, 1000); // Update every second for live countdown
        remainingTimeIntervals.push(interval);
    });

    // Legacy timer support (for backward compatibility)
    const timers = document.querySelectorAll('.task-timer[data-due-time]');
    timers.forEach(timer => {
        const dueTime = parseInt(timer.getAttribute('data-due-time'));
        if (isNaN(dueTime)) return;

        const updateTimer = () => {
            const now = new Date().getTime();
            const distance = dueTime - now;

            if (distance < 0) {
                timer.textContent = '00:00:00';
                timer.closest('.task-card')?.classList.add('failed');
                return;
            }

            const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((distance % (1000 * 60)) / 1000);

            timer.textContent = 
                String(hours).padStart(2, '0') + ':' +
                String(minutes).padStart(2, '0') + ':' +
                String(seconds).padStart(2, '0');
        };

        updateTimer();
        const interval = setInterval(updateTimer, 1000);
        countdownIntervals.push(interval);
    });
}

// Auto-mark task as failed when time expires
let failedTasks = new Set(); // Track tasks already marked as failed to avoid duplicate API calls

async function markTaskAsFailed(taskId, instanceDate = null) {
    // Create unique key for recurring tasks (taskId + instanceDate)
    const taskKey = instanceDate ? `${taskId}_${instanceDate}` : taskId.toString();
    
    // Prevent duplicate API calls for the same task/instance
    if (failedTasks.has(taskKey)) {
        return;
    }
    failedTasks.add(taskKey);
    
    try {
        // Use instance date for recurring tasks, or today for one-time tasks
        const today = typeof getKolkataTime === 'function' 
            ? getKolkataTime().toISOString().split('T')[0]
            : new Date().toISOString().split('T')[0];
        
        const completionDate = instanceDate || today;
        
        // Determine API path based on current page location
        const apiPath = window.location.pathname.includes('/user/') || window.location.pathname.includes('/admin/')
            ? '../api/completions.php'
            : 'api/completions.php';
        
        const response = await fetch(apiPath, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                task_id: parseInt(taskId),
                status: 'failed',
                completion_date: completionDate,
                instance_date: instanceDate // For recurring tasks
            })
        });
        
        const result = await response.json();
        if (result.success) {
            console.log('Task marked as failed:', taskId, instanceDate ? `(instance: ${instanceDate})` : '');
            // For recurring tasks, the cronjob will create the next instance
            // Reload the page to show updated status after a short delay
            if (currentPage === 'tasks' || currentPage === 'dashboard') {
                setTimeout(() => {
                    if (typeof refreshCurrentPage === 'function') {
                        refreshCurrentPage();
                    } else {
                        window.location.reload();
                    }
                }, 2000);
            }
        } else {
            console.error('Failed to mark task as failed:', result.message);
            // Remove from set so it can be retried
            failedTasks.delete(taskKey);
        }
    } catch (error) {
        console.error('Error marking task as failed:', error);
        // Remove from set so it can be retried
        failedTasks.delete(taskKey);
    }
}

// Load notifications from API
async function loadNotifications() {
    const notificationList = document.getElementById('notificationList');
    if (!notificationList) return;

    try {
        // Determine API path based on current page location
        const apiPath = window.location.pathname.includes('/user/') || window.location.pathname.includes('/admin/') 
            ? '../api/notifications.php' 
            : 'api/notifications.php';
        const response = await fetch(apiPath);
        if (!response.ok) {
            throw new Error('Failed to load notifications');
        }
        
        const result = await response.json();
        if (result.success && result.notifications) {
            if (result.notifications.length === 0) {
                notificationList.innerHTML = '<div class="notification-item" style="padding: 20px; text-align: center; color: var(--text-secondary);">No notifications</div>';
            } else {
                notificationList.innerHTML = result.notifications.map(notif => {
                    const timeAgo = formatTimeAgo(notif.created_at);
                    return `
                        <div class="notification-item ${notif.is_read ? '' : 'unread'}" data-id="${notif.id}">
                            <div class="notification-content">
                                <p class="notification-message">${escapeHtml(notif.message)}</p>
                                <span class="notification-time">${timeAgo}</span>
                            </div>
                        </div>
                    `;
                }).join('');
                
                // Add click handlers to mark as read
                notificationList.querySelectorAll('.notification-item').forEach(item => {
                    item.addEventListener('click', async () => {
                        const notifId = item.dataset.id;
                        if (notifId) {
                            await markNotificationAsRead(notifId);
                            item.classList.remove('unread');
                        }
                    });
                });
            }
        }
    } catch (error) {
        console.error('Error loading notifications:', error);
        notificationList.innerHTML = '<div class="notification-item" style="padding: 20px; text-align: center; color: var(--danger);">Error loading notifications</div>';
    }
}

// Mark notification as read
async function markNotificationAsRead(notificationId) {
    try {
        const apiPath = window.location.pathname.includes('/user/') || window.location.pathname.includes('/admin/') 
            ? '../api/notifications.php' 
            : 'api/notifications.php';
        const response = await fetch(apiPath, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'read',
                id: parseInt(notificationId)
            })
        });
        
        if (response.ok) {
            // Update badge count
            const result = await response.json();
            if (result.success) {
                updateNotificationBadge();
            }
        }
    } catch (error) {
        console.error('Error marking notification as read:', error);
    }
}

// Update notification badge count
async function updateNotificationBadge() {
    try {
        const apiPath = window.location.pathname.includes('/user/') || window.location.pathname.includes('/admin/') 
            ? '../api/notifications.php' 
            : 'api/notifications.php';
        const response = await fetch(`${apiPath}?unread_only=true`);
        if (response.ok) {
            const result = await response.json();
            const notificationBtn = document.getElementById('notificationBtn');
            const badge = document.getElementById('notificationBadge');
            
            if (result.unread_count > 0) {
                // Show notification button
                if (notificationBtn) {
                    notificationBtn.style.display = 'flex';
                }
                
                // Update or create badge
                if (badge) {
                    badge.textContent = result.unread_count;
                    badge.style.display = 'block';
                } else if (notificationBtn) {
                    // Create badge if it doesn't exist
                    const newBadge = document.createElement('span');
                    newBadge.id = 'notificationBadge';
                    newBadge.className = 'badge';
                    newBadge.textContent = result.unread_count;
                    notificationBtn.appendChild(newBadge);
                }
            } else {
                // Hide notification button if no notifications
                if (notificationBtn) {
                    notificationBtn.style.display = 'none';
                }
                if (badge) badge.style.display = 'none';
            }
        }
    } catch (error) {
        console.error('Error updating notification badge:', error);
    }
}

// Format time ago
function formatTimeAgo(dateString) {
    const now = new Date();
    const date = new Date(dateString);
    const diffMs = now - date;
    const diffMins = Math.floor(diffMs / 60000);
    const diffHours = Math.floor(diffMs / 3600000);
    const diffDays = Math.floor(diffMs / 86400000);
    
    if (diffMins < 1) return 'Just now';
    if (diffMins < 60) return `${diffMins}m ago`;
    if (diffHours < 24) return `${diffHours}h ago`;
    if (diffDays < 7) return `${diffDays}d ago`;
    return date.toLocaleDateString();
}

// Notification Dropdown (native behavior)
function initNotifications() {
    const notificationBtn = document.getElementById('notificationBtn');
    const notificationDropdown = document.getElementById('notificationDropdown');
    const clearNotificationsBtn = document.getElementById('clearNotifications');

    if (notificationBtn && notificationDropdown) {
        notificationBtn.addEventListener('click', async (e) => {
            e.stopPropagation();
            const isHidden = notificationDropdown.classList.contains('hidden');
            notificationDropdown.classList.toggle('hidden');
            
            // Load notifications when opening dropdown
            if (isHidden) {
                await loadNotifications();
            }
        });

        document.addEventListener('click', (e) => {
            if (!notificationBtn.contains(e.target) && !notificationDropdown.contains(e.target)) {
                notificationDropdown.classList.add('hidden');
            }
        });
    }
    
    // Clear all notifications
    if (clearNotificationsBtn) {
        clearNotificationsBtn.addEventListener('click', async () => {
            if (confirm('Clear all notifications?')) {
                try {
                    const apiPath = window.location.pathname.includes('/user/') || window.location.pathname.includes('/admin/') 
                        ? '../api/notifications.php' 
                        : 'api/notifications.php';
                    const response = await fetch(apiPath, {
                        method: 'PUT',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            action: 'clear_all'
                        })
                    });
                    
                    if (response.ok) {
                        await loadNotifications();
                        await updateNotificationBadge();
                        // Hide button if no notifications left
                        const notificationBtn = document.getElementById('notificationBtn');
                        if (notificationBtn) {
                            notificationBtn.style.display = 'none';
                        }
                    }
                } catch (error) {
                    console.error('Error clearing notifications:', error);
                }
            }
        });
    }
    
    // Update badge on page load (this will show/hide button based on count)
    updateNotificationBadge();
}

// Recurring task form toggle (native behavior)
function initRecurringTaskToggle() {
    const taskType = document.getElementById('taskType');
    const recurringDaysGroup = document.getElementById('recurringDaysGroup');

    if (taskType && recurringDaysGroup) {
        taskType.addEventListener('change', (e) => {
            if (e.target.value === 'recurring') {
                recurringDaysGroup.style.display = 'block';
            } else {
                recurringDaysGroup.style.display = 'none';
            }
        });
    }
}

// Modal helper functions (for task completion, etc.)
let lastCompleteTaskId = '';
function openMarkCompleteModal(taskId) {
    const modal = document.getElementById('markCompleteModal');
    const form = document.getElementById('markCompleteForm');
    const hiddenTaskId = document.getElementById('hiddenCompleteTaskId');
    if (modal && form && hiddenTaskId) {
        lastCompleteTaskId = String(taskId);
        hiddenTaskId.value = taskId;
        form.reset();
        hiddenTaskId.value = taskId; // Reset after form reset
        // Reset image preview
        const imagePreview = document.getElementById('imagePreview');
        const previewImg = document.getElementById('previewImg');
        if (imagePreview) imagePreview.style.display = 'none';
        if (previewImg) previewImg.src = '';
        openModal('markCompleteModal');
    }
}

document.addEventListener('DOMContentLoaded', function () {
    const form = document.getElementById('markCompleteForm');
    if (!form) return;
    form.addEventListener('submit', function () {
        const hiddenTaskId = document.getElementById('hiddenCompleteTaskId');
        if (hiddenTaskId && !hiddenTaskId.value && lastCompleteTaskId) {
            hiddenTaskId.value = lastCompleteTaskId;
        }
    });
});

function viewTaskDetails(taskId) {
    const modal = document.getElementById('taskDetailModal');
    const content = document.getElementById('taskDetailContent');
    const title = document.getElementById('taskDetailTitle');
    const actions = document.getElementById('taskDetailActions');
    
    if (!modal || !content || !title) return;
    
    content.innerHTML = '<div class="loading-spinner"><div class="spinner"></div><p>Loading task details...</p></div>';
    actions.style.display = 'none';
    openModal('taskDetailModal');
    
    // Determine API path based on current page location
    const apiPath = window.location.pathname.includes('/user/') || window.location.pathname.includes('/admin/')
        ? '../api/tasks.php'
        : 'api/tasks.php';
    
    fetch(`${apiPath}?id=${taskId}`)
        .then(r => r.json())
        .then(data => {
            if (data.success && data.task) {
                const t = data.task;
                const priorityColors = {high: '#0f8a1f', medium: '#28a745', low: '#6fa86f'};
                const pColor = priorityColors[t.priority] || '#86868b';
                const startDate = new Date(t.start_date);
                const endDate = t.end_date ? new Date(t.end_date) : null;
                
                // Check if user is admin
                const isAdmin = typeof CURRENT_USER !== 'undefined' && CURRENT_USER.role === 'admin';
                
                title.textContent = t.name;
                // Format recurring days for display
                let recurringDaysDisplay = '';
                if (t.task_type === 'recurring' && t.skip_days) {
                    const dayNames = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                    const recurringDays = t.skip_days.split(',').map(d => parseInt(d));
                    const dayLabels = recurringDays.map(dayNum => dayNames[dayNum]).join(', ');
                    recurringDaysDisplay = `<span class="ios-detail-label">Repeat Days</span><span class="ios-detail-value">${escapeHtml(dayLabels)}</span>`;
                }
                
                // Format dates with time if available
                const startTime = t.start_time ? ` at ${new Date('2000-01-01T' + t.start_time).toLocaleTimeString('en-US', {hour: 'numeric', minute: '2-digit'})}` : '';
                const endTime = t.end_time ? ` at ${new Date('2000-01-01T' + t.end_time).toLocaleTimeString('en-US', {hour: 'numeric', minute: '2-digit'})}` : '';
                
                content.innerHTML = `
                    <div class="ios-task-detail">
                        <div class="ios-task-detail-hero">
                            <div class="ios-task-badge-group">
                                <span class="ios-task-priority" style="background: ${pColor}15; color: ${pColor}; border-color: ${pColor}30;">${t.priority}</span>
                                <span class="ios-task-type-badge">${t.task_type === 'recurring' ? '🔄 Recurring' : '📌 One-Time'}</span>
                            </div>
                            ${t.task_number ? `<div class="ios-task-number">${escapeHtml(t.task_number)}</div>` : ''}
                        </div>
                        
                        <div class="ios-detail-section">
                            <div class="ios-detail-section-title">Information</div>
                            <div class="ios-detail-item">
                                <div class="ios-detail-item-content">
                                    <span class="ios-detail-icon">📝</span>
                                    <div class="ios-detail-text">
                                        <span class="ios-detail-label">Description</span>
                                        <span class="ios-detail-value">${escapeHtml(t.description || 'No description')}</span>
                                    </div>
                                </div>
                            </div>
                            <div class="ios-detail-item">
                                <div class="ios-detail-item-content">
                                    <span class="ios-detail-icon">📅</span>
                                    <div class="ios-detail-text">
                                        <span class="ios-detail-label">Start Date</span>
                                        <span class="ios-detail-value">${startDate.toLocaleDateString('en-US', {weekday: 'short', month: 'short', day: 'numeric', year: 'numeric'})}${startTime}</span>
                                    </div>
                                </div>
                            </div>
                            ${endDate ? `<div class="ios-detail-item">
                                <div class="ios-detail-item-content">
                                    <span class="ios-detail-icon">⏰</span>
                                    <div class="ios-detail-text">
                                        <span class="ios-detail-label">End Date</span>
                                        <span class="ios-detail-value">${endDate.toLocaleDateString('en-US', {weekday: 'short', month: 'short', day: 'numeric', year: 'numeric'})}${endTime}</span>
                                    </div>
                                </div>
                            </div>` : ''}
                            ${recurringDaysDisplay ? `<div class="ios-detail-item">
                                <div class="ios-detail-item-content">
                                    <span class="ios-detail-icon">🔄</span>
                                    <div class="ios-detail-text">
                                        ${recurringDaysDisplay}
                                    </div>
                                </div>
                            </div>` : ''}
                        </div>
                        
                        <div class="ios-detail-section">
                            <div class="ios-detail-section-title">Assignment</div>
                            <div class="ios-detail-item">
                                <div class="ios-detail-item-content">
                                    <span class="ios-detail-icon">👥</span>
                                    <div class="ios-detail-text">
                                        <span class="ios-detail-label">Assigned To</span>
                                        <span class="ios-detail-value">${t.assigned_usernames && t.assigned_usernames.length ? escapeHtml(t.assigned_usernames.join(', ')) : 'Unassigned'}</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="ios-detail-section">
                            <div class="ios-detail-section-title">Points</div>
                            <div class="ios-detail-item">
                                <div class="ios-detail-item-content">
                                    <span class="ios-detail-icon">📊</span>
                                    <div class="ios-detail-text">
                                        <span class="ios-detail-label">Task Points</span>
                                        <span class="ios-detail-value ios-point-value">${t.task_points} TP</span>
                                    </div>
                                </div>
                            </div>
                            <div class="ios-detail-item">
                                <div class="ios-detail-item-content">
                                    <span class="ios-detail-icon">🏆</span>
                                    <div class="ios-detail-text">
                                        <span class="ios-detail-label">Quality Points</span>
                                        <span class="ios-detail-value ios-point-value">${t.quality_points} QP</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        ${t.notes ? `<div class="ios-detail-section">
                            <div class="ios-detail-section-title">Notes</div>
                            <div class="ios-detail-item">
                                <div class="ios-detail-item-content ios-notes-content">
                                    <span class="ios-detail-value ios-notes-text">${escapeHtml(t.notes)}</span>
                                </div>
                            </div>
                        </div>` : ''}
                    </div>
                `;
                // Only show edit/delete buttons for admin users
                if (isAdmin && actions) {
                    actions.style.display = 'flex';
                    actions.dataset.taskId = taskId;
                } else if (actions) {
                    actions.style.display = 'none';
                }
            }
        })
        .catch(e => {
            content.innerHTML = '<div class="ios-empty-state"><div class="ios-empty-icon">❌</div><h3>Error</h3><p>Failed to load task details</p></div>';
            showToast('Error loading task details', 'error');
        });
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
}

// Load users for admin dashboard (make it globally accessible)
let loadAdminUsersRetryCount = 0;
const MAX_RETRIES = 5;

window.loadAdminUsers = async function loadAdminUsers() {
    const adminUsersList = document.getElementById('adminUsersList');
    if (!adminUsersList) {
        // Only retry if we haven't exceeded max retries and we're on admin-dashboard page
        if (loadAdminUsersRetryCount < MAX_RETRIES && currentPage === 'admin-dashboard') {
            loadAdminUsersRetryCount++;
            setTimeout(() => {
                loadAdminUsers();
            }, 200);
        } else {
            // Element doesn't exist and we're not on admin-dashboard, so just return silently
            return;
        }
        return;
    }
    
    // Reset retry count on success
    loadAdminUsersRetryCount = 0;
    
    console.log('Loading users from API...');
    
    try {
        const response = await fetch('api/users.php');
        
        if (!response.ok) {
            const errorText = await response.text();
            throw new Error(`HTTP error! status: ${response.status} - ${errorText}`);
        }
        
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const text = await response.text();
            throw new Error(`Expected JSON but got: ${contentType}. Response: ${text.substring(0, 100)}`);
        }
        
        const data = await response.json();
        console.log('Users API response:', data);
        
        if (data.success && data.users) {
            console.log(`Loaded ${data.users.length} users`);
            renderAdminUsers(data.users);
        } else {
            console.error('API returned error:', data);
            adminUsersList.innerHTML = `
                <div class="empty-state">
                    <div class="empty-state-icon">❌</div>
                    <h3>Error loading users</h3>
                    <p>${data.message || 'Failed to load users'}</p>
                </div>
            `;
        }
    } catch (error) {
        console.error('Error loading users:', error);
        adminUsersList.innerHTML = `
            <div class="empty-state">
                <div class="empty-state-icon">❌</div>
                <h3>Error loading users</h3>
                <p>${error.message}</p>
                <button class="btn btn-primary btn-small" onclick="loadAdminUsers()" style="margin-top: 10px;">Retry</button>
            </div>
        `;
    }
}

function renderAdminUsers(users) {
    const adminUsersList = document.getElementById('adminUsersList');
    if (!adminUsersList) return;
    
    if (users.length === 0) {
        adminUsersList.innerHTML = `
            <div class="empty-state">
                <div class="empty-state-icon">👥</div>
                <h3>No users found</h3>
                <p>Click "Add User" to create a new user.</p>
            </div>
        `;
        return;
    }
    
    adminUsersList.innerHTML = users.map(user => {
        const userName = user.full_name || user.username;
        const createdDate = new Date(user.created_at);
        const formattedDate = createdDate.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
        const totalAssigned = user.total_assigned_works || 0;
        const todayAssigned = user.today_assigned_works || 0;
        const tomorrowAssigned = user.tomorrow_assigned_works || 0;
        const leaveDates = user.leave_dates || [];
        
        // Format leave dates
        let leaveDatesHtml = '';
        if (leaveDates.length > 0) {
            const today = new Date();
            today.setHours(0, 0, 0, 0);
            
            const leaveBadges = leaveDates.map(leaveDate => {
                const date = new Date(leaveDate);
                date.setHours(0, 0, 0, 0);
                const isToday = date.getTime() === today.getTime();
                const formattedLeaveDate = date.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
                return isToday 
                    ? `<span class="leave-badge today">Today</span>`
                    : `<span class="leave-badge">${formattedLeaveDate}</span>`;
            });
            leaveDatesHtml = leaveBadges.join(' ');
        }
        
        const canDelete = user.can_delete !== false;
        const isAdmin = user.role === 'admin';
        
        return `
            <div class="user-card ${isAdmin ? 'admin-card' : ''}">
                <div class="user-card-header">
                    <span class="user-name">${escapeHtml(userName)}</span>
                    <span class="user-role ${user.role}">${user.role}</span>
                </div>
                <div class="user-info-section">
                    <div class="user-info-item">
                        <span class="user-info-label">Username:</span>
                        <span class="user-info-value">${escapeHtml(user.username)}</span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Password:</span>
                        <span class="user-info-value password-field" data-password="${escapeHtml(user.password)}">
                            <span class="password-display">••••••••</span>
                            <button type="button" class="password-toggle" onclick="togglePassword(this)" title="Show/Hide Password">
                                <span class="eye-icon">👁️</span>
                            </button>
                        </span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Created Date:</span>
                        <span class="user-info-value">${formattedDate}</span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Total Assigned Works:</span>
                        <span class="user-info-value clickable-count" onclick="showUserTasks(${user.id}, 'all', '${escapeHtml(user.username)}')" title="Click to view all tasks">
                            ${totalAssigned}
                        </span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Today's Work:</span>
                        <span class="user-info-value clickable-count" onclick="showUserTasks(${user.id}, 'today', '${escapeHtml(user.username)}')" title="Click to view today's tasks">
                            ${todayAssigned}
                        </span>
                    </div>
                    <div class="user-info-item">
                        <span class="user-info-label">Tomorrow's Work:</span>
                        <span class="user-info-value clickable-count" onclick="showUserTasks(${user.id}, 'tomorrow', '${escapeHtml(user.username)}')" title="Click to view tomorrow's tasks">
                            ${tomorrowAssigned}
                        </span>
                    </div>
                    ${leaveDatesHtml ? `
                    <div class="user-info-item">
                        <span class="user-info-label">Leave Days:</span>
                        <span class="user-info-value leave-dates">${leaveDatesHtml}</span>
                    </div>
                    ` : ''}
                </div>
                <div class="task-actions" style="margin-top: 15px;">
                    <button class="btn btn-secondary btn-small" onclick="openEditUserModal(${user.id})">Edit</button>
                    ${canDelete ? `<button class="btn btn-danger btn-small" onclick="openDeleteUserModal(${user.id}, '${escapeHtml(user.username)}')">Delete</button>` : ''}
                    <button class="btn btn-info btn-small" onclick="openLeaveModal(${user.id}, '${escapeHtml(user.username)}')">Leave</button>
                </div>
            </div>
        `;
    }).join('');
    
    // Store users data for edit modal
    window.ALL_USERS_DATA = users;
}

// Load user performance data
window.loadUserPerformance = async function loadUserPerformance(filter = 'today') {
    const userPerformanceList = document.getElementById('userPerformanceList');
    if (!userPerformanceList) {
        console.warn('userPerformanceList element not found');
        return;
    }
    
    console.log('Loading user performance with filter:', filter);
    
    // Show loading state
    userPerformanceList.innerHTML = `
        <div class="empty-state">
            <div class="empty-state-icon">⏳</div>
            <h3>Loading performance data...</h3>
        </div>
    `;
    
    try {
        const response = await fetch(`api/performance.php?filter=${encodeURIComponent(filter)}`);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const data = await response.json();
        console.log('Performance data received:', data);
        
        if (data.success && data.performance) {
            renderUserPerformance(data.performance);
        } else {
            userPerformanceList.innerHTML = `
                <div class="empty-state">
                    <div class="empty-state-icon">❌</div>
                    <h3>Error loading performance data</h3>
                    <p>${data.message || 'Failed to load performance data'}</p>
                </div>
            `;
        }
    } catch (error) {
        console.error('Error loading user performance:', error);
        userPerformanceList.innerHTML = `
            <div class="empty-state">
                <div class="empty-state-icon">❌</div>
                <h3>Error loading performance data</h3>
                <p>${error.message}</p>
                <button class="btn btn-primary btn-small" onclick="loadUserPerformance('${filter}')" style="margin-top: 10px;">Retry</button>
            </div>
        `;
    }
}

function renderUserPerformance(performance) {
    const userPerformanceList = document.getElementById('userPerformanceList');
    if (!userPerformanceList) return;
    
    if (performance.length === 0) {
        userPerformanceList.innerHTML = `
            <div class="empty-state">
                <div class="empty-state-icon">📊</div>
                <h3>No performance data</h3>
                <p>No user performance data available for the selected period.</p>
            </div>
        `;
        return;
    }
    
    userPerformanceList.innerHTML = performance.map(user => {
        const userName = user.full_name || user.username;
        const completedTasks = user.completed_tasks || 0;
        const pendingTasks = user.pending_tasks || 0;
        const completionRate = user.completion_rate || 0;
        const totalPoints = user.total_points || 0;
        
        return `
            <div class="performance-item">
                <div class="performance-info">
                    <h4>${escapeHtml(userName)}</h4>
                    <p>${completedTasks} completed, ${pendingTasks} pending</p>
                </div>
                <div class="performance-stats">
                    <div class="stat-value">${completionRate}%</div>
                    <div class="stat-label">Completion Rate</div>
                    <div class="stat-value" style="margin-top: 5px;">${totalPoints}</div>
                    <div class="stat-label">Total Points</div>
                </div>
            </div>
        `;
    }).join('');
}

// Password toggle function (make it globally available)
window.togglePassword = function togglePassword(button) {
    const passwordField = button.closest('.password-field');
    if (!passwordField) return;
    
    const passwordDisplay = passwordField.querySelector('.password-display');
    const actualPassword = passwordField.getAttribute('data-password');
    
    if (!passwordDisplay || !actualPassword) return;
    
    if (passwordDisplay.textContent === '••••••••' || passwordDisplay.textContent.trim() === '') {
        passwordDisplay.textContent = actualPassword;
        const eyeIcon = button.querySelector('.eye-icon');
        if (eyeIcon) eyeIcon.textContent = '🙈';
    } else {
        passwordDisplay.textContent = '••••••••';
        const eyeIcon = button.querySelector('.eye-icon');
        if (eyeIcon) eyeIcon.textContent = '👁️';
    }
}

// Show user tasks in iOS-style modal
window.showUserTasks = async function showUserTasks(userId, filter, username) {
    const modal = document.getElementById('userTasksModal');
    const modalTitle = document.getElementById('userTasksModalTitle');
    const modalBody = document.getElementById('userTasksModalBody');
    const datePicker = document.getElementById('taskDatePicker');
    const clearDateBtn = document.getElementById('clearDateBtn');
    
    if (!modal || !modalTitle || !modalBody) {
        showToast('Tasks modal not found', 'error');
        return;
    }
    
    // Store current user context for date picker
    if (!modal.dataset.userId) {
        modal.dataset.userId = userId;
        modal.dataset.username = username;
    }
    
    // Set title based on filter
    const titles = {
        'all': `All Tasks - ${username}`,
        'today': `Today's Tasks - ${username}`,
        'tomorrow': `Tomorrow's Tasks - ${username}`
    };
    modalTitle.textContent = titles[filter] || `Tasks - ${username}`;
    
    // Set initial date based on filter
    if (datePicker) {
        const today = new Date().toISOString().split('T')[0];
        const tomorrow = new Date(Date.now() + 86400000).toISOString().split('T')[0];
        
        if (filter === 'today') {
            datePicker.value = today;
        } else if (filter === 'tomorrow') {
            datePicker.value = tomorrow;
        } else {
            datePicker.value = '';
        }
        
        // Show/hide clear button
        if (clearDateBtn) {
            clearDateBtn.style.display = datePicker.value ? 'inline-block' : 'none';
        }
    }
    
    // Show loading state
    modalBody.innerHTML = `
        <div class="loading-spinner">
            <div class="spinner"></div>
            <p>Loading tasks...</p>
        </div>
    `;
    
    // Open modal
    modal.classList.add('active');
    document.body.style.overflow = 'hidden';
    
    // Load tasks with current filter
    await loadUserTasksForModal(userId, filter);
    
    // Setup date picker event listener
    if (datePicker && !datePicker.dataset.listenerAttached) {
        datePicker.dataset.listenerAttached = 'true';
        datePicker.addEventListener('change', async () => {
            const selectedDate = datePicker.value;
            if (clearDateBtn) {
                clearDateBtn.style.display = selectedDate ? 'inline-block' : 'none';
            }
            // Reload tasks with selected date
            await loadUserTasksForModal(userId, selectedDate ? 'date' : 'all', selectedDate);
        });
    }
    
    // Setup clear date button
    if (clearDateBtn && !clearDateBtn.dataset.listenerAttached) {
        clearDateBtn.dataset.listenerAttached = 'true';
        clearDateBtn.addEventListener('click', async () => {
            if (datePicker) {
                datePicker.value = '';
                clearDateBtn.style.display = 'none';
                await loadUserTasksForModal(userId, 'all');
            }
        });
    }
}

// Load tasks for the modal
async function loadUserTasksForModal(userId, filter, customDate = null) {
    const modalBody = document.getElementById('userTasksModalBody');
    const modalTitle = document.getElementById('userTasksModalTitle');
    const modal = document.getElementById('userTasksModal');
    
    if (!modalBody) return;
    
    // Show loading state
    modalBody.innerHTML = `
        <div class="loading-spinner">
            <div class="spinner"></div>
            <p>Loading tasks...</p>
        </div>
    `;
    
    try {
        const today = new Date().toISOString().split('T')[0];
        const tomorrow = new Date(Date.now() + 86400000).toISOString().split('T')[0];
        
        let url = `api/tasks.php?user_id=${userId}`;
        let dateToUse = null;
        
        if (filter === 'today') {
            dateToUse = today;
        } else if (filter === 'tomorrow') {
            dateToUse = tomorrow;
        } else if (filter === 'date' && customDate) {
            dateToUse = customDate;
        }
        
        if (dateToUse) {
            url += `&date=${dateToUse}`;
        }
        
        const response = await fetch(url);
        const result = await response.json();
        
        if (result.success && result.tasks) {
            const tasks = result.tasks;
            
            // Update title if custom date is selected
            if (modalTitle && dateToUse && filter === 'date') {
                const dateObj = new Date(dateToUse);
                const username = modal.dataset.username || 'User';
                modalTitle.textContent = `Tasks - ${dateObj.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })} - ${username}`;
            }
            
            if (tasks.length === 0) {
                const emptyMessage = dateToUse 
                    ? `No tasks scheduled for ${new Date(dateToUse).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })}.`
                    : 'This user has no assigned tasks.';
                    
                modalBody.innerHTML = `
                    <div class="ios-empty-state">
                        <div class="ios-empty-icon">📝</div>
                        <h3>No tasks found</h3>
                        <p>${emptyMessage}</p>
                    </div>
                `;
            } else {
                // Count completed and pending tasks
                const completedCount = tasks.filter(t => t.status === 'completed').length;
                const pendingCount = tasks.filter(t => t.status !== 'completed').length;
                
                modalBody.innerHTML = `
                    <div class="ios-task-stats">
                        <div class="ios-stat-item completed">
                            <span class="ios-stat-icon">✓</span>
                            <div class="ios-stat-content">
                                <span class="ios-stat-label">Completed</span>
                                <span class="ios-stat-value">${completedCount}</span>
                            </div>
                        </div>
                        <div class="ios-stat-item pending">
                            <span class="ios-stat-icon">○</span>
                            <div class="ios-stat-content">
                                <span class="ios-stat-label">Pending</span>
                                <span class="ios-stat-value">${pendingCount}</span>
                            </div>
                        </div>
                    </div>
                ` + tasks.map(task => {
                    const startDate = new Date(task.start_date);
                    const endDate = task.end_date ? new Date(task.end_date) : null;
                    const priorityColors = {
                        'high': '#0f8a1f',
                        'medium': '#28a745',
                        'low': '#6fa86f'
                    };
                    const priorityColor = priorityColors[task.priority] || '#86868b';
                    
                    // Determine task status
                    const taskStatus = task.status || 'pending';
                    const isCompleted = taskStatus === 'completed';
                    const statusColor = isCompleted ? '#34c759' : '#ff9500';
                    const statusText = isCompleted ? 'Completed' : 'Pending';
                    const statusIcon = isCompleted ? '✓' : '○';
                    
                    return `
                        <div class="ios-task-item ${isCompleted ? 'ios-task-completed' : ''}">
                            <div class="ios-task-header">
                                <h4 class="ios-task-name">${escapeHtml(task.name)}</h4>
                                <div class="ios-task-badges">
                                    <span class="ios-task-priority" style="background: ${priorityColor}20; color: ${priorityColor}; border-color: ${priorityColor}40;">
                                        ${task.priority}
                                    </span>
                                    <span class="ios-task-status" style="background: ${statusColor}20; color: ${statusColor}; border-color: ${statusColor}40;">
                                        <span class="status-icon">${statusIcon}</span>
                                        ${statusText}
                                    </span>
                                </div>
                            </div>
                            <p class="ios-task-description">${escapeHtml(task.description || 'No description')}</p>
                            <div class="ios-task-meta">
                                <span class="ios-task-meta-item">
                                    <span class="ios-meta-icon">📅</span>
                                    <span>${startDate.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })}</span>
                                </span>
                                ${endDate ? `
                                <span class="ios-task-meta-item">
                                    <span class="ios-meta-icon">⏰</span>
                                    <span>${endDate.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })}</span>
                                </span>
                                ` : ''}
                                <span class="ios-task-meta-item">
                                    <span class="ios-meta-icon">${task.task_type === 'recurring' ? '🔄' : '📌'}</span>
                                    <span>${task.task_type === 'recurring' ? 'Recurring' : 'One-Time'}</span>
                                </span>
                            </div>
                            <div class="ios-task-points">
                                <span class="ios-point-badge">TP: ${task.task_points}</span>
                                <span class="ios-point-badge">QP: ${task.quality_points}</span>
                            </div>
                        </div>
                    `;
                }).join('');
            }
        } else {
            modalBody.innerHTML = `
                <div class="ios-empty-state">
                    <div class="ios-empty-icon">❌</div>
                    <h3>Error loading tasks</h3>
                    <p>${result.message || 'Failed to load tasks'}</p>
                </div>
            `;
        }
    } catch (error) {
        console.error('Error loading user tasks:', error);
        modalBody.innerHTML = `
            <div class="ios-empty-state">
                <div class="ios-empty-icon">❌</div>
                <h3>Error</h3>
                <p>${error.message}</p>
            </div>
        `;
    }
}

function openEditUserModal(userId) {
    // Try to get user data from window.ALL_USERS_DATA first
    let user = null;
    if (window.ALL_USERS_DATA && Array.isArray(window.ALL_USERS_DATA)) {
        user = window.ALL_USERS_DATA.find(u => u.id == userId);
    }
    
    // If not found, try to fetch from API
    if (!user) {
        fetch(`api/users.php`)
            .then(response => response.json())
            .then(data => {
                if (data.success && data.users) {
                    window.ALL_USERS_DATA = data.users;
                    const foundUser = data.users.find(u => u.id == userId);
                    if (foundUser) {
                        populateEditModal(foundUser, userId);
                    } else {
                        showToast('User not found', 'error');
                    }
                } else {
                    showToast('User data not loaded. Please refresh the page.', 'error');
                }
            })
            .catch(error => {
                console.error('Error fetching user data:', error);
                showToast('Error loading user data. Please refresh the page.', 'error');
            });
        return;
    }
    
    populateEditModal(user, userId);
}

function populateEditModal(user, userId) {
    // Populate form elements with null checks
    const usernameEl = document.getElementById('newUsername');
    const passwordEl = document.getElementById('newPassword');
    const roleEl = document.getElementById('newUserRole');
    const emailEl = document.getElementById('newUserEmail');
    const fullNameEl = document.getElementById('newUserFullName');
    
    if (!usernameEl || !passwordEl || !roleEl) {
        showToast('Edit form elements not found. Please refresh the page.', 'error');
        return;
    }
    
    usernameEl.value = user.username || '';
    passwordEl.value = '';
    roleEl.value = user.role || 'user';
    if (emailEl) emailEl.value = user.email || '';
    if (fullNameEl) fullNameEl.value = user.full_name || '';
    
    // Set form action and user ID
    const form = document.getElementById('addUserForm');
    if (!form) {
        showToast('User form not found. Please refresh the page.', 'error');
        return;
    }
    
    form.action = 'edit_user.php';
    let hiddenUserId = document.getElementById('hiddenUserId');
    if (!hiddenUserId) {
        // Create hidden input if it doesn't exist
        const input = document.createElement('input');
        input.type = 'hidden';
        input.id = 'hiddenUserId';
        input.name = 'user_id';
        try {
            form.appendChild(input);
            hiddenUserId = input;
        } catch (e) {
            console.error('Error appending hidden input:', e);
            showToast('Error initializing edit form. Please refresh the page.', 'error');
            return;
        }
    }
    hiddenUserId.value = userId;
    
    const modalTitle = document.getElementById('addUserModalTitle');
    const submitBtn = document.getElementById('submitUserBtn');
    const passwordLabel = document.getElementById('newPasswordLabel');
    const passwordHint = document.getElementById('passwordHint');
    
    if (modalTitle) modalTitle.textContent = 'Edit User';
    if (submitBtn) submitBtn.textContent = 'Update User';
    if (passwordLabel) passwordLabel.textContent = 'Password (leave blank to keep current)';
    if (passwordEl) passwordEl.removeAttribute('required');
    if (passwordHint) passwordHint.style.display = 'block';
    
    openModal('addUserModal');
}

function openDeleteUserModal(userId, username) {
    const hiddenDeleteUserId = document.getElementById('hiddenDeleteUserId');
    if (!hiddenDeleteUserId) {
        const form = document.getElementById('deleteUserForm');
        if (!form) {
            showToast('Delete form not found. Please refresh the page.', 'error');
            return;
        }
        const input = document.createElement('input');
        input.type = 'hidden';
        input.id = 'hiddenDeleteUserId';
        input.name = 'user_id';
        try {
            form.appendChild(input);
        } catch (e) {
            console.error('Error appending hidden input:', e);
            showToast('Error initializing delete form. Please refresh the page.', 'error');
            return;
        }
    }
    document.getElementById('hiddenDeleteUserId').value = userId;
    document.getElementById('deleteUserMessage').textContent = `Are you sure you want to delete user "${username}"? This action cannot be undone.`;
    openModal('deleteUserModal');
}

function openLeaveModal(userId, username) {
    // Try to find elements - start with the form since it's more likely to be unique
    const addLeaveForm = document.getElementById('addLeaveForm');
    let leaveModal = null;
    
    // If form exists, find the modal via the form
    if (addLeaveForm) {
        leaveModal = addLeaveForm.closest('.modal');
        if (leaveModal && !leaveModal.id) {
            // If modal doesn't have ID, set it
            leaveModal.id = 'leaveModal';
        }
    }
    
    // If still not found, try direct ID lookup
    if (!leaveModal) {
        leaveModal = document.getElementById('leaveModal');
    }
    
    // If still not found, try querySelector
    if (!leaveModal) {
        leaveModal = document.querySelector('#leaveModal');
    }
    
    // Debug: list all modals if still not found
    if (!leaveModal) {
        const allModals = document.querySelectorAll('.modal');
        const modalIds = Array.from(allModals).map(m => m.id || '(no id)');
        console.error('leaveModal not found in DOM');
        console.error('Total modals found:', allModals.length);
        console.error('Modal IDs:', modalIds);
        
        // Try to find modal containing the form
        if (addLeaveForm) {
            leaveModal = addLeaveForm.closest('div.modal');
            console.log('Found modal via form.closest(.modal):', leaveModal);
        }
    }
    
    // Get form elements
    if (!addLeaveForm) {
        console.error('addLeaveForm not found');
        showToast('Leave form not found. Please refresh the page.', 'error');
        return;
    }
    
    const hiddenLeaveUserId = document.getElementById('hiddenLeaveUserId');
    const leaveUserName = document.getElementById('leaveUserName');
    
    if (!hiddenLeaveUserId) {
        console.error('hiddenLeaveUserId element not found, creating it');
        // The element should exist, but if it doesn't, create it
        const input = document.createElement('input');
        input.type = 'hidden';
        input.id = 'hiddenLeaveUserId';
        input.name = 'user_id';
        addLeaveForm.insertBefore(input, addLeaveForm.firstChild);
        input.value = userId;
    } else {
        hiddenLeaveUserId.value = userId;
    }
    
    if (!leaveUserName) {
        console.error('leaveUserName element not found');
        showToast('Leave modal elements not found. Please refresh the page.', 'error');
        return;
    }
    
    try {
        leaveUserName.textContent = username;
        
        // Load existing leaves for this user
        loadUserLeaves(userId);
        
        // Open modal - use the modal we found or try by ID
        if (leaveModal) {
            leaveModal.classList.add('active');
            document.body.style.overflow = 'hidden';
        } else {
            // Fallback: try to open by ID anyway
            openModal('leaveModal');
        }
    } catch (e) {
        console.error('Error opening leave modal:', e);
        showToast('Error opening leave modal: ' + e.message, 'error');
    }
}

async function loadUserLeaves(userId) {
    try {
        const response = await fetch(`api/leaves.php?user_id=${userId}`);
        const result = await response.json();
        
        const leavesList = document.getElementById('leavesList');
        if (result.success && result.leaves && result.leaves.length > 0) {
            leavesList.innerHTML = result.leaves.map(leave => {
                const date = new Date(leave.leave_date);
                return `
                    <div class="leave-item" style="display: flex; justify-content: space-between; align-items: center; padding: 8px; margin: 5px 0; background: #f5f5f5; border-radius: 4px;">
                        <span>${date.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' })}</span>
                        <button class="btn btn-small btn-danger" onclick="deleteLeave(${leave.id}, ${userId})" style="padding: 4px 8px; font-size: 12px;">Remove</button>
                    </div>
                `;
            }).join('');
        } else {
            leavesList.innerHTML = '<p style="color: #999; padding: 10px;">No leave dates set</p>';
        }
    } catch (error) {
        console.error('Error loading leaves:', error);
        document.getElementById('leavesList').innerHTML = '<p style="color: #c33; padding: 10px;">Error loading leave dates</p>';
    }
}

async function deleteLeave(leaveId, userId) {
    const confirmed = await showIOSConfirm('Are you sure you want to remove this leave date?', 'Remove Leave Date');
    if (!confirmed) {
        return;
    }
    
    try {
        const response = await fetch('api/leaves.php', {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ id: leaveId })
        });
        
        const result = await response.json();
        
        if (result.success) {
            loadUserLeaves(userId);
            showToast('Leave date removed successfully', 'success');
        } else {
            showToast(result.message || 'Failed to remove leave date', 'error');
        }
    } catch (error) {
        console.error('Error deleting leave:', error);
        showToast('Error: ' + error.message, 'error');
    }
}

// Logout functionality
function initLogout() {
    const logoutBtn = document.getElementById('logoutBtn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', async (e) => {
            e.preventDefault();
            const confirmed = await showIOSConfirm('Are you sure you want to logout?', 'Logout');
            if (confirmed) {
                window.location.href = 'logout.php';
            }
        });
    }
}

// Show/hide admin links based on user role
function initAdminLinks() {
    const adminLinks = document.getElementById('adminLinks');
    if (adminLinks) {
        if (typeof CURRENT_USER !== 'undefined' && CURRENT_USER.role === 'admin') {
            adminLinks.classList.remove('hidden');
        } else {
            adminLinks.classList.add('hidden');
        }
    }
}

// Initialize leave form handler
function initLeaveForm() {
    const addLeaveForm = document.getElementById('addLeaveForm');
    if (addLeaveForm && !addLeaveForm.dataset.ajaxHandlerAttached) {
        addLeaveForm.dataset.ajaxHandlerAttached = 'true';
        addLeaveForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const userId = document.getElementById('hiddenLeaveUserId')?.value;
            const leaveDate = document.getElementById('leaveDate')?.value;
            
            if (!userId || !leaveDate) {
                showToast('User ID and leave date are required', 'warning');
                return;
            }
            
            try {
                const response = await fetch('api/leaves.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        user_id: parseInt(userId),
                        leave_date: leaveDate
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    document.getElementById('leaveDate').value = '';
                    loadUserLeaves(parseInt(userId));
                    showToast('Leave date added successfully', 'success');
                } else {
                    showToast(result.message || 'Failed to add leave date', 'error');
                }
            } catch (error) {
                console.error('Error adding leave:', error);
                showToast('Error: ' + error.message, 'error');
            }
        });
    }
    
    // Cancel button is handled in initCancelButtons() for consistency
}

// Initialize all cancel buttons to close their respective modals
function initCancelButtons() {
    // Cancel button for Add/Edit Task Modal
    const cancelTaskBtn = document.getElementById('cancelTaskBtn');
    if (cancelTaskBtn) {
        cancelTaskBtn.addEventListener('click', () => {
            closeModal('addTaskModal');
        });
    }
    
    // Cancel button for Add/Edit User Modal
    const cancelUserBtn = document.getElementById('cancelUserBtn');
    if (cancelUserBtn) {
        cancelUserBtn.addEventListener('click', () => {
            closeModal('addUserModal');
        });
    }
    
    // Cancel button for Delete User Modal
    const cancelDeleteUserBtn = document.getElementById('cancelDeleteUserBtn');
    if (cancelDeleteUserBtn) {
        cancelDeleteUserBtn.addEventListener('click', () => {
            closeModal('deleteUserModal');
        });
    }
    
    // Cancel button for Delete Task Modal
    const cancelDeleteBtn = document.getElementById('cancelDeleteBtn');
    if (cancelDeleteBtn) {
        cancelDeleteBtn.addEventListener('click', () => {
            closeModal('deleteTaskModal');
        });
    }
    
    // Cancel button for Disable Task Modal
    const cancelDisableTaskBtn = document.getElementById('cancelDisableTaskBtn');
    if (cancelDisableTaskBtn) {
        cancelDisableTaskBtn.addEventListener('click', () => {
            closeModal('disableTaskModal');
        });
    }
    
    // Cancel button for Enable Task Modal
    const cancelEnableTaskBtn = document.getElementById('cancelEnableTaskBtn');
    if (cancelEnableTaskBtn) {
        cancelEnableTaskBtn.addEventListener('click', () => {
            closeModal('enableTaskModal');
        });
    }
    
    // Cancel button for Mark Complete Modal
    const cancelMarkCompleteBtn = document.getElementById('cancelMarkCompleteBtn');
    if (cancelMarkCompleteBtn) {
        cancelMarkCompleteBtn.addEventListener('click', () => {
            closeModal('markCompleteModal');
        });
    }
    
    // Cancel button for Leave Modal (already handled in initLeaveForm, but adding here for consistency)
    const cancelLeaveBtn = document.getElementById('cancelLeaveBtn');
    if (cancelLeaveBtn && !cancelLeaveBtn.dataset.listenerAttached) {
        cancelLeaveBtn.dataset.listenerAttached = 'true';
        cancelLeaveBtn.addEventListener('click', () => {
            closeModal('leaveModal');
        });
    }
    
    // Enable/Disable Task Form Handlers
    const disableTaskForm = document.getElementById('disableTaskForm');
    if (disableTaskForm && !disableTaskForm.dataset.handlerAttached) {
        disableTaskForm.dataset.handlerAttached = 'true';
        disableTaskForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const taskId = document.getElementById('hiddenDisableTaskId')?.value;
            const adminPassword = document.getElementById('adminPasswordDisable')?.value;
            const disableReason = document.getElementById('disableReason')?.value;
            
            if (!taskId) {
                showToast('Task ID is required', 'warning');
                return;
            }
            
            if (!adminPassword) {
                showToast('Admin password is required', 'warning');
                return;
            }
            
            if (!disableReason || disableReason.trim() === '') {
                showToast('Please provide a reason for disabling this task', 'warning');
                return;
            }
            
            try {
                const response = await fetch('api/task_status.php', {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        task_id: parseInt(taskId),
                        is_enabled: 0,
                        admin_password: adminPassword,
                        disable_reason: disableReason.trim()
                    })
                });
                
                // Check if response is JSON
                const contentType = response.headers.get('content-type');
                if (!contentType || !contentType.includes('application/json')) {
                    const text = await response.text();
                    console.error('Non-JSON response:', text);
                    showToast('Server error: Please run the migration to add disable_reason column', 'error');
                    return;
                }
                
                const result = await response.json();
                
                if (result.success) {
                    closeModal('disableTaskModal');
                    showToast('Task disabled successfully', 'success');
                    // Reload the page to refresh task list
                    setTimeout(() => {
                        if (currentPage === 'admin-tasks') {
                            loadPage('admin-tasks');
                        } else if (currentPage === 'admin-tasks-disabled') {
                            loadPage('admin-tasks-disabled');
                        } else {
                            refreshCurrentPage();
                        }
                    }, 300);
                } else {
                    showToast(result.message || 'Failed to disable task', 'error');
                }
            } catch (error) {
                console.error('Error disabling task:', error);
                if (error.message.includes('JSON')) {
                    showToast('Server error: Please run the migration to add disable_reason column', 'error');
                } else {
                    showToast('Error: ' + error.message, 'error');
                }
            }
        });
    }
    
    const enableTaskForm = document.getElementById('enableTaskForm');
    if (enableTaskForm && !enableTaskForm.dataset.handlerAttached) {
        enableTaskForm.dataset.handlerAttached = 'true';
        enableTaskForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const taskId = document.getElementById('hiddenEnableTaskId')?.value;
            const adminPassword = document.getElementById('adminPasswordEnable')?.value;
            
            if (!taskId) {
                showToast('Task ID is required', 'warning');
                return;
            }
            
            if (!adminPassword) {
                showToast('Admin password is required', 'warning');
                return;
            }
            
            try {
                const response = await fetch('api/task_status.php', {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        task_id: parseInt(taskId),
                        is_enabled: 1,
                        admin_password: adminPassword
                    })
                });
                
                // Check if response is JSON
                const contentType = response.headers.get('content-type');
                if (!contentType || !contentType.includes('application/json')) {
                    const text = await response.text();
                    console.error('Non-JSON response:', text);
                    showToast('Server error: Please check database connection', 'error');
                    return;
                }
                
                const result = await response.json();
                
                if (result.success) {
                    closeModal('enableTaskModal');
                    showToast('Task enabled successfully', 'success');
                    // Reload the page to refresh task list
                    setTimeout(() => {
                        if (currentPage === 'admin-tasks') {
                            loadPage('admin-tasks');
                        } else if (currentPage === 'admin-tasks-disabled') {
                            loadPage('admin-tasks-disabled');
                        } else {
                            refreshCurrentPage();
                        }
                    }, 300);
                } else {
                    showToast(result.message || 'Failed to enable task', 'error');
                }
            } catch (error) {
                console.error('Error enabling task:', error);
                if (error.message.includes('JSON')) {
                    showToast('Server error: Please check database connection', 'error');
                } else {
                    showToast('Error: ' + error.message, 'error');
                }
            }
        });
    }
}

// Initialize task detail modal actions (minimal JS)
function initTaskDetailActions() {
    const editBtn = document.getElementById('taskDetailEditBtn');
    const deleteBtn = document.getElementById('taskDetailDeleteBtn');
    
    if (editBtn) {
        editBtn.addEventListener('click', () => {
            const taskId = document.getElementById('taskDetailActions')?.dataset.taskId;
            if (taskId && typeof openEditTaskModal === 'function') {
                closeModal('taskDetailModal');
                setTimeout(() => openEditTaskModal(parseInt(taskId)), 300);
            } else if (taskId) {
                closeModal('taskDetailModal');
                showToast('Edit function not available', 'warning');
            }
        });
    }
    
    if (deleteBtn) {
        deleteBtn.addEventListener('click', () => {
            const taskId = document.getElementById('taskDetailActions')?.dataset.taskId;
            if (taskId) {
                // Determine API path based on current page location
                const apiPath = window.location.pathname.includes('/user/') || window.location.pathname.includes('/admin/')
                    ? '../api/tasks.php'
                    : 'api/tasks.php';
                
                fetch(`${apiPath}?id=${taskId}`)
                    .then(r => r.json())
                    .then(data => {
                        if (data.success && data.task) {
                            closeModal('taskDetailModal');
                            if (typeof openDeleteTaskModal === 'function') {
                                setTimeout(() => openDeleteTaskModal(taskId, data.task.name), 300);
                            } else {
                                showToast('Delete function not available', 'warning');
                            }
                        }
                    })
                    .catch(() => showToast('Error loading task', 'error'));
            }
        });
    }
}

// Open Edit Task Modal (make globally available)
window.openEditTaskModal = function openEditTaskModal(taskId) {
    // Try to get task from window.ALL_TASKS first
    let task = window.ALL_TASKS?.find(t => t.id == taskId);
    
    // If not found, fetch from API
    if (!task) {
        // Determine API path based on current page location
        const apiPath = window.location.pathname.includes('/user/') || window.location.pathname.includes('/admin/')
            ? '../api/tasks.php'
            : 'api/tasks.php';
        
        fetch(`${apiPath}?id=${taskId}`)
            .then(r => r.json())
            .then(data => {
                if (data.success && data.task) {
                    populateEditTaskModal(data.task, taskId);
                } else {
                    showToast('Task not found', 'error');
                }
            })
            .catch(() => showToast('Error loading task', 'error'));
        return;
    }
    
    populateEditTaskModal(task, taskId);
}

function populateEditTaskModal(task, taskId) {
    const taskName = document.getElementById('taskName');
    const taskDescription = document.getElementById('taskDescription');
    const taskPriority = document.getElementById('taskPriority');
    const taskStartDate = document.getElementById('taskStartDate');
    const taskEndDate = document.getElementById('taskEndDate');
    const taskStartTime = document.getElementById('taskStartTime');
    const taskEndTime = document.getElementById('taskEndTime');
    const taskType = document.getElementById('taskType');
    const taskPoints = document.getElementById('taskPoints');
    const qualityPoints = document.getElementById('qualityPoints');
    const lostTaskPoints = document.getElementById('lostTaskPoints');
    const taskNotes = document.getElementById('taskNotes');
    const assignSelect = document.getElementById('taskAssignTo');
    const form = document.getElementById('addTaskForm');
    const hiddenTaskId = document.getElementById('hiddenTaskId');
    const taskModalTitle = document.getElementById('taskModalTitle');
    const submitTaskBtn = document.getElementById('submitTaskBtn');
    const recurringDaysGroup = document.getElementById('recurringDaysGroup');
    
    if (!taskName || !taskDescription || !form) {
        showToast('Edit form not found', 'error');
        return;
    }
    
    // Populate form fields
    if (taskName) taskName.value = task.name || '';
    if (taskDescription) taskDescription.value = task.description || '';
    if (taskPriority) taskPriority.value = task.priority || 'medium';
    if (taskStartDate) taskStartDate.value = task.start_date || '';
    if (taskEndDate) taskEndDate.value = task.end_date || '';
    if (taskStartTime) taskStartTime.value = task.start_time || '';
    if (taskEndTime) taskEndTime.value = task.end_time || '';
    if (taskType) taskType.value = task.task_type || 'one-time';
    if (taskPoints) taskPoints.value = task.task_points || 1;
    if (qualityPoints) qualityPoints.value = task.quality_points || 10;
    if (lostTaskPoints) lostTaskPoints.value = task.lost_task_points || -1;
    if (taskNotes) taskNotes.value = task.notes || '';
    
    // Handle recurring days (stored in skip_days field, repurposed)
    // Clear all checkboxes first
    ['repeatSunday', 'repeatMonday', 'repeatTuesday', 'repeatWednesday', 'repeatThursday', 'repeatFriday', 'repeatSaturday'].forEach((id) => {
        const checkbox = document.getElementById(id);
        if (checkbox) checkbox.checked = false;
    });
    
    if (task.skip_days) {
        const recurringDays = task.skip_days.split(',').map(d => parseInt(d));
        ['repeatSunday', 'repeatMonday', 'repeatTuesday', 'repeatWednesday', 'repeatThursday', 'repeatFriday', 'repeatSaturday'].forEach((id, index) => {
            const checkbox = document.getElementById(id);
            if (checkbox) checkbox.checked = recurringDays.includes(index);
        });
    }
    
    // Handle assigned users
    if (assignSelect) {
        const assignedIds = task.assigned_user_ids || [];
        Array.from(assignSelect.options).forEach(option => {
            option.selected = assignedIds.includes(parseInt(option.value));
        });
    }
    
    // Show/hide recurring fields
    if (task.task_type === 'recurring') {
        if (recurringDaysGroup) recurringDaysGroup.style.display = 'block';
    } else {
        if (recurringDaysGroup) recurringDaysGroup.style.display = 'none';
    }
    
    // Set form action and task ID
    if (form) form.action = 'edit_task.php';
    if (hiddenTaskId) hiddenTaskId.value = taskId;
    if (taskModalTitle) taskModalTitle.textContent = 'Edit Task';
    if (submitTaskBtn) submitTaskBtn.textContent = 'Update Task';
    
    openModal('addTaskModal');
}

// Open Delete Task Modal (make globally available)
window.openDeleteTaskModal = function openDeleteTaskModal(taskId, taskName) {
    const hiddenDeleteTaskId = document.getElementById('hiddenDeleteTaskId');
    const deleteTaskModal = document.getElementById('deleteTaskModal');
    const deleteTaskModalText = deleteTaskModal?.querySelector('p');
    
    if (!hiddenDeleteTaskId || !deleteTaskModal) {
        showToast('Delete modal not found', 'error');
        return;
    }
    
    hiddenDeleteTaskId.value = taskId;
    if (deleteTaskModalText) {
        deleteTaskModalText.textContent = `Are you sure you want to delete task "${taskName}"? This action requires admin password.`;
    }
    openModal('deleteTaskModal');
}

// Open Disable Task Modal (make globally available)
window.openDisableTaskModal = function openDisableTaskModal(taskId, taskName, taskNumber) {
    console.log('Opening disable modal for task:', taskId, taskName, taskNumber);
    
    // Wait for DOM to be ready if needed
    let retryCount = 0;
    const maxRetries = 10;
    
    const tryOpenModal = () => {
        const hiddenDisableTaskId = document.getElementById('hiddenDisableTaskId');
        const disableTaskModal = document.getElementById('disableTaskModal');
        const disableTaskMessage = document.getElementById('disableTaskMessage');
        
        console.log('Checking for modal elements:', {
            hiddenDisableTaskId: !!hiddenDisableTaskId,
            disableTaskModal: !!disableTaskModal,
            disableTaskMessage: !!disableTaskMessage,
            retryCount: retryCount
        });
        
        if (!hiddenDisableTaskId || !disableTaskModal) {
            if (retryCount < maxRetries) {
                retryCount++;
                setTimeout(tryOpenModal, 100);
                return;
            } else {
                console.error('Disable modal elements not found after', maxRetries, 'retries');
                console.error('Available modals:', document.querySelectorAll('.modal').length);
                showToast('Disable modal not found. Please refresh the page.', 'error');
                return;
            }
        }
        
        hiddenDisableTaskId.value = taskId;
        if (disableTaskMessage) {
            const taskInfo = taskNumber ? ` (${taskNumber})` : '';
            disableTaskMessage.textContent = `Are you sure you want to disable task "${taskName}"${taskInfo}? Disabled tasks will be removed from all users' working tasks. This action requires admin password.`;
        }
        console.log('Opening disable modal');
        openModal('disableTaskModal');
    };
    
    // Use requestAnimationFrame to ensure DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', tryOpenModal);
    } else {
        tryOpenModal();
    }
}

// Open Enable Task Modal (make globally available)
window.openEnableTaskModal = function openEnableTaskModal(taskId, taskName, taskNumber) {
    console.log('Opening enable modal for task:', taskId, taskName, taskNumber);
    
    // Wait for DOM to be ready if needed
    let retryCount = 0;
    const maxRetries = 10;
    
    const tryOpenModal = () => {
        const hiddenEnableTaskId = document.getElementById('hiddenEnableTaskId');
        const enableTaskModal = document.getElementById('enableTaskModal');
        const enableTaskMessage = document.getElementById('enableTaskMessage');
        
        console.log('Checking for modal elements:', {
            hiddenEnableTaskId: !!hiddenEnableTaskId,
            enableTaskModal: !!enableTaskModal,
            enableTaskMessage: !!enableTaskMessage,
            retryCount: retryCount
        });
        
        if (!hiddenEnableTaskId || !enableTaskModal) {
            if (retryCount < maxRetries) {
                retryCount++;
                setTimeout(tryOpenModal, 100);
                return;
            } else {
                console.error('Enable modal elements not found after', maxRetries, 'retries');
                console.error('Available modals:', document.querySelectorAll('.modal').length);
                showToast('Enable modal not found. Please refresh the page.', 'error');
                return;
            }
        }
        
        hiddenEnableTaskId.value = taskId;
        if (enableTaskMessage) {
            const taskInfo = taskNumber ? ` (${taskNumber})` : '';
            enableTaskMessage.textContent = `Are you sure you want to enable task "${taskName}"${taskInfo}? The task will be available to assigned users again. This action requires admin password.`;
        }
        console.log('Opening enable modal');
        openModal('enableTaskModal');
    };
    
    // Use requestAnimationFrame to ensure DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', tryOpenModal);
    } else {
        tryOpenModal();
    }
}

// Filter tasks (make globally available)
window.filterTasks = function filterTasks() {
    const searchQuery = (document.getElementById('searchTasks')?.value || '').toLowerCase().trim();
    const userFilter = document.getElementById('filterUser')?.value || '';
    const dateFilter = document.getElementById('filterDate')?.value || '';
    const priorityFilter = document.getElementById('filterPriority')?.value || '';
    const cards = document.querySelectorAll('.task-card-item');
    
    if (cards.length === 0) return;
    
    cards.forEach(card => {
        const userIds = (card.dataset.userIds || '').split(',').filter(Boolean);
        const cardDate = card.dataset.date || '';
        const cardPriority = card.dataset.priority || '';
        const cardTaskNumber = card.dataset.taskNumber || '';
        const cardName = card.dataset.name || '';
        const cardDescription = card.dataset.description || '';
        const cardNotes = card.dataset.notes || '';
        const cardAssignedUsers = card.dataset.assignedUsers || '';
        
        // Search filter (includes task number)
        const matchSearch = !searchQuery || 
            cardTaskNumber.includes(searchQuery) ||
            cardName.includes(searchQuery) || 
            cardDescription.includes(searchQuery) || 
            cardNotes.includes(searchQuery) ||
            cardAssignedUsers.includes(searchQuery);
        
        // Other filters
        const matchUser = !userFilter || userIds.includes(userFilter);
        const matchDate = !dateFilter || cardDate === dateFilter;
        const matchPriority = !priorityFilter || cardPriority === priorityFilter;
        
        card.style.display = (matchSearch && matchUser && matchDate && matchPriority) ? '' : 'none';
    });
}

// Clear filters (make globally available)
window.clearFilters = function clearFilters() {
    const searchTasks = document.getElementById('searchTasks');
    const filterUser = document.getElementById('filterUser');
    const filterDate = document.getElementById('filterDate');
    const filterPriority = document.getElementById('filterPriority');
    
    if (searchTasks) searchTasks.value = '';
    if (filterUser) filterUser.value = '';
    if (filterDate) filterDate.value = '';
    if (filterPriority) filterPriority.value = '';
    
    if (typeof filterTasks === 'function') {
        filterTasks();
    }
}

// Initialize user form handlers
function initUserForms() {
    const addUserForm = document.getElementById('addUserForm');
    if (addUserForm && !addUserForm.dataset.ajaxHandlerAttached) {
        addUserForm.dataset.ajaxHandlerAttached = 'true';
        addUserForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const usernameEl = document.getElementById('newUsername');
            const passwordEl = document.getElementById('newPassword');
            const roleEl = document.getElementById('newUserRole');
            const emailEl = document.getElementById('newUserEmail');
            const fullNameEl = document.getElementById('newUserFullName');
            const userIdEl = document.getElementById('hiddenUserId');
            
            const userId = userIdEl?.value;
            const isEdit = userId && userId !== '';
            
            const data = {
                username: usernameEl?.value || '',
                password: passwordEl?.value || '',
                role: roleEl?.value || 'user',
                email: emailEl?.value || null,
                full_name: fullNameEl?.value || null
            };
            
            if (isEdit && userId) {
                data.id = parseInt(userId);
            }
            
            // Validate required fields
            if (!data.username) {
                showToast('Username is required', 'warning');
                return;
            }
            
            if (!isEdit && !data.password) {
                showToast('Password is required', 'warning');
                return;
            }
            
            try {
                const url = 'api/users.php';
                const method = isEdit ? 'PUT' : 'POST';
                
                const response = await fetch(url, {
                    method: method,
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    closeModal('addUserModal');
                    showToast(result.message || (isEdit ? 'User updated successfully!' : 'User created successfully!'), 'success');
                    // Reload users if on admin dashboard
                    if (currentPage === 'admin-dashboard') {
                        loadAdminUsersRetryCount = 0;
                        loadAdminUsers();
                    } else if (currentPage === 'admin-users') {
                        // Reload the admin-users page to refresh the list
                        loadPage('admin-users');
                    }
                } else {
                    showToast(result.message || 'Operation failed', 'error');
                }
            } catch (error) {
                console.error('Error submitting user form:', error);
                showToast('Error: ' + error.message, 'error');
            }
        });
    }
    
    const deleteUserForm = document.getElementById('deleteUserForm');
    if (deleteUserForm && !deleteUserForm.dataset.ajaxHandlerAttached) {
        deleteUserForm.dataset.ajaxHandlerAttached = 'true';
        deleteUserForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const userId = document.getElementById('hiddenDeleteUserId')?.value;
            const adminPassword = document.getElementById('adminPasswordDelete')?.value;
            
            if (!userId) {
                showToast('User ID is required', 'warning');
                return;
            }
            
            if (!adminPassword) {
                showToast('Admin password is required', 'warning');
                return;
            }
            
            try {
                const response = await fetch('api/users.php', {
                    method: 'DELETE',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        id: userId,
                        admin_password: adminPassword
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    closeModal('deleteUserModal');
                    showToast(result.message || 'User deleted successfully!', 'success');
                    // Reload users if on admin dashboard
                    if (currentPage === 'admin-dashboard') {
                        loadAdminUsersRetryCount = 0;
                        loadAdminUsers();
                    } else if (currentPage === 'admin-users') {
                        // Reload the admin-users page to refresh the list
                        loadPage('admin-users');
                    }
                } else {
                    showToast(result.message || 'Failed to delete user', 'error');
                }
            } catch (error) {
                console.error('Error deleting user:', error);
                showToast('Error: ' + error.message, 'error');
            }
        });
    }
}

// Initialize everything
document.addEventListener('DOMContentLoaded', () => {
    initTheme();
    initNavigation();
    initModals();
    initNotifications();
    initRecurringTaskToggle();
    initLogout();
    initAdminLinks();
    initUserForms();
    initLeaveForm();
    initCancelButtons();
    initTaskDetailActions();
    
    // Initialize countdown timers on page load
    setTimeout(() => initCountdownTimers(), 500);
    
    // Initialize page-specific functionality on initial load
    const initialPage = currentPage || (window.location.hash ? window.location.hash.substring(1) : 'dashboard');
    setTimeout(() => {
        if (initialPage === 'progress') {
            initProgressPage();
        } else if (initialPage === 'history') {
            initHistoryPage();
        } else if (initialPage === 'tasks') {
            initTasksPage();
        }
    }, 600);
    
    // Check for success/error messages in URL and show toast, then clean URL and refresh
    const urlParams = new URLSearchParams(window.location.search);
    const successMsg = urlParams.get('success');
    const errorMsg = urlParams.get('error');
    const taskErrorMsg = urlParams.get('task_error');
    
    // Clean URL parameters after reading them
    let urlCleaned = false;
    let shouldRefresh = false;
    
    if (successMsg) {
        showToast(decodeURIComponent(successMsg), 'success');
        urlCleaned = true;
        shouldRefresh = true; // Refresh after success operations
    }
    
    if (errorMsg) {
        showToast(decodeURIComponent(errorMsg), 'error');
        urlCleaned = true;
    }
    
    if (taskErrorMsg) {
        showToast(decodeURIComponent(taskErrorMsg), 'error');
        urlCleaned = true;
    }
    
    // Clean URL by removing query parameters, keep hash
    if (urlCleaned) {
        const newUrl = window.location.pathname + (window.location.hash || '');
        window.history.replaceState({}, document.title, newUrl);
    }
    
    // Refresh page data after showing success message (for edit, delete, add operations)
    if (shouldRefresh) {
        // Small delay to ensure toast is visible, then refresh
        setTimeout(() => {
            refreshCurrentPage();
        }, 500);
    }
    
    // Load initial page if user is logged in
    if (typeof IS_LOGGED_IN !== 'undefined' && IS_LOGGED_IN) {
        // Check if we're in SPA mode (has pageContainer) or standalone page mode
        const pageContainer = document.getElementById('pageContainer');
        const isSPAMode = pageContainer !== null;
        
        // Only handle hash navigation in SPA mode
        if (isSPAMode) {
            // Check if user is admin and redirect to admin dashboard
            const isAdmin = typeof CURRENT_USER !== 'undefined' && CURRENT_USER.role === 'admin';
            const defaultPage = isAdmin ? 'admin-dashboard' : 'dashboard';
            
            // Get page from hash, or use default
            let initialPage = defaultPage;
            const hash = window.location.hash.substring(1);
            if (hash && pageRoutes[hash]) {
                // Validate that the page exists and user has access
                // Admin-only pages check
                const adminOnlyPages = ['admin-dashboard', 'admin-tasks', 'admin-users', 'admin-reports'];
                if (adminOnlyPages.includes(hash) && !isAdmin) {
                    // Non-admin trying to access admin page, redirect to default
                    initialPage = defaultPage;
                    window.location.hash = defaultPage;
                } else {
                    initialPage = hash;
                }
            } else if (!hash) {
                // No hash, set default page hash
                window.location.hash = defaultPage;
            }
            
            // If admin and no hash, make sure we're on admin dashboard
            if (isAdmin && !hash && initialPage === 'admin-dashboard') {
                // Set the active nav link to admin-dashboard
                const adminDashboardLink = document.querySelector('[data-page="admin-dashboard"]');
                if (adminDashboardLink) {
                    setActiveNavLink(adminDashboardLink);
                }
            }
            
            // Load the initial page immediately
            console.log('User is logged in, loading page:', initialPage);
            setTimeout(() => {
                loadPage(initialPage);
                // Also check if we're already on admin-dashboard and load users
                if (initialPage === 'admin-dashboard') {
                    setTimeout(() => {
                        loadAdminUsers();
                        // Get the selected filter value (default to 'today')
                        const performanceFilter = document.getElementById('performanceFilter');
                        const selectedFilter = performanceFilter ? (performanceFilter.value || 'today') : 'today';
                        loadUserPerformance(selectedFilter);
                    }, 300);
                }
            }, 100);
        }
        // If not in SPA mode (standalone pages), don't set hash or load pages
    } else {
        // Check if adminUsersList exists on page (server-side rendered)
        // Only check if we're on admin-dashboard page
        if (currentPage === 'admin-dashboard') {
            setTimeout(() => {
                const adminUsersList = document.getElementById('adminUsersList');
                if (adminUsersList && adminUsersList.innerHTML.includes('Loading users')) {
                    loadAdminUsersRetryCount = 0;
                    loadAdminUsers();
                }
            }, 500);
        }
    }

    // Theme toggle button
    const themeToggle = document.getElementById('themeToggle');
    if (themeToggle) {
        themeToggle.addEventListener('click', toggleTheme);
    }

    // Performance filter dropdown - This will only work if page is loaded on initial page load
    // For dynamically loaded pages, the event listener is attached in loadPage function
    const performanceFilter = document.getElementById('performanceFilter');
    if (performanceFilter && !performanceFilter.dataset.initialListenerAttached) {
        performanceFilter.dataset.initialListenerAttached = 'true';
        performanceFilter.addEventListener('change', (e) => {
            const filter = e.target.value;
            console.log('Performance filter changed to:', filter);
            loadUserPerformance(filter);
        });
    }
});
