<?php
// Setup Check Script
require_once 'config.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>Setup Check</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .success { color: green; }
        .error { color: red; }
        .warning { color: orange; }
        table { border-collapse: collapse; width: 100%; margin: 20px 0; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
    </style>
</head>
<body>
    <h1>Task Management System - Setup Check</h1>
    
    <h2>1. Database Connection</h2>
    <?php
    try {
        $conn = getDBConnection();
        if ($conn) {
            echo "<p class='success'>✓ Database connection successful</p>";
            echo "<p>Host: " . DB_HOST . "</p>";
            echo "<p>Database: " . DB_NAME . "</p>";
        } else {
            echo "<p class='error'>✗ Database connection failed</p>";
        }
    } catch (Exception $e) {
        echo "<p class='error'>✗ Database error: " . htmlspecialchars($e->getMessage()) . "</p>";
    }
    ?>
    
    <h2>2. Database Tables</h2>
    <?php
    if (isset($conn)) {
        $tables = ['users', 'tasks', 'task_assignments', 'task_completions', 'notifications', 'user_points'];
        $missing_tables = [];
        
        foreach ($tables as $table) {
            $result = $conn->query("SHOW TABLES LIKE '$table'");
            if ($result->num_rows > 0) {
                echo "<p class='success'>✓ Table '$table' exists</p>";
            } else {
                echo "<p class='error'>✗ Table '$table' is missing</p>";
                $missing_tables[] = $table;
            }
        }
        
        if (empty($missing_tables)) {
            echo "<p class='success'><strong>All required tables exist!</strong></p>";
        } else {
            echo "<p class='error'><strong>Please run database.sql to create missing tables.</strong></p>";
        }
    }
    ?>
    
    <h2>3. Users in Database</h2>
    <?php
    if (isset($conn)) {
        $result = $conn->query("SELECT COUNT(*) as count FROM users");
        $row = $result->fetch_assoc();
        $user_count = $row['count'];
        
        if ($user_count > 0) {
            echo "<p class='success'>✓ Found $user_count user(s) in database</p>";
            
            echo "<table>";
            echo "<tr><th>ID</th><th>Username</th><th>Password Preview</th><th>Role</th><th>Full Name</th></tr>";
            
            $users_result = $conn->query("SELECT id, username, password, role, full_name FROM users");
            while ($user = $users_result->fetch_assoc()) {
                $pwd_preview = strlen($user['password']) > 15 ? substr($user['password'], 0, 15) . '...' : $user['password'];
                echo "<tr>";
                echo "<td>" . htmlspecialchars($user['id']) . "</td>";
                echo "<td>" . htmlspecialchars($user['username']) . "</td>";
                echo "<td>" . htmlspecialchars($pwd_preview) . " (length: " . strlen($user['password']) . ")</td>";
                echo "<td>" . htmlspecialchars($user['role']) . "</td>";
                echo "<td>" . htmlspecialchars($user['full_name'] ?? 'N/A') . "</td>";
                echo "</tr>";
            }
            echo "</table>";
            
            // Test password for admin
            echo "<h3>Password Test for 'admin' user:</h3>";
            $test_stmt = $conn->prepare("SELECT password FROM users WHERE username = 'admin'");
            if ($test_stmt) {
                $test_stmt->execute();
                $test_result = $test_stmt->get_result();
                if ($test_result->num_rows > 0) {
                    $test_user = $test_result->fetch_assoc();
                    $expected = 'password123';
                    $actual = $test_user['password'];
                    
                    echo "<p>Expected password: <code>$expected</code></p>";
                    echo "<p>Stored password: <code>" . htmlspecialchars($actual) . "</code></p>";
                    echo "<p>Password length: " . strlen($actual) . " characters</p>";
                    
                    if ($expected === $actual) {
                        echo "<p class='success'>✓ Password matches correctly!</p>";
                    } else {
                        echo "<p class='error'>✗ Password does NOT match!</p>";
                        echo "<p class='warning'>The stored password is different from 'password123'. Please check insert_dummy_data.sql</p>";
                    }
                }
            }
        } else {
            echo "<p class='error'>✗ No users found in database</p>";
            echo "<p class='warning'>Please run insert_dummy_data.sql to add users</p>";
        }
    }
    ?>
    
    <h2>4. Session Configuration</h2>
    <?php
    if (session_status() === PHP_SESSION_ACTIVE) {
        echo "<p class='success'>✓ Session is active</p>";
        echo "<p>Session ID: " . session_id() . "</p>";
    } else {
        echo "<p class='warning'>⚠ Session is not active</p>";
    }
    ?>
    
    <h2>5. File Permissions</h2>
    <?php
    $dirs_to_check = ['uploads', 'uploads/completions'];
    foreach ($dirs_to_check as $dir) {
        if (file_exists($dir)) {
            if (is_writable($dir)) {
                echo "<p class='success'>✓ Directory '$dir' is writable</p>";
            } else {
                echo "<p class='error'>✗ Directory '$dir' is NOT writable</p>";
            }
        } else {
            echo "<p class='warning'>⚠ Directory '$dir' does not exist</p>";
        }
    }
    ?>
    
    <h2>6. Quick Login Test</h2>
    <form method="POST" action="login.php" id="testLoginForm">
        <p>
            <label>Username: <input type="text" name="username" value="admin" required></label>
        </p>
        <p>
            <label>Password: <input type="password" name="password" value="password123" required></label>
        </p>
        <p>
            <button type="button" onclick="testLogin()">Test Login (AJAX)</button>
        </p>
    </form>
    <div id="loginResult"></div>
    
    <script>
        async function testLogin() {
            const form = document.getElementById('testLoginForm');
            const formData = new FormData(form);
            const username = formData.get('username');
            const password = formData.get('password');
            
            const resultDiv = document.getElementById('loginResult');
            resultDiv.innerHTML = '<p>Testing login...</p>';
            
            try {
                const response = await fetch('login.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ username, password })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    resultDiv.innerHTML = '<p class="success">✓ Login successful! Response: ' + JSON.stringify(data, null, 2) + '</p>';
                } else {
                    resultDiv.innerHTML = '<p class="error">✗ Login failed: ' + data.message + '</p>' +
                                         '<p>Debug info: ' + JSON.stringify(data.debug || {}, null, 2) + '</p>';
                }
            } catch (error) {
                resultDiv.innerHTML = '<p class="error">✗ Error: ' + error.message + '</p>';
            }
        }
    </script>
    
    <hr>
    <p><a href="index.php">Go to Main Application</a></p>
</body>
</html>
