<?php
/**
 * Demo Data Inserter for Users 1 and 2
 * ------------------------------------
 * When you open this file in the browser, it will insert demo tasks and completions
 * for user IDs 1 and 2, covering statuses: completed, failed, pending.
 *
 * Safe to run multiple times: it checks by task_number prefix and completion_date
 * and will skip inserting duplicates.
 */

require_once 'config.php';

$conn = getDBConnection();
if (!$conn) {
    die("Database connection failed: " . mysqli_connect_error());
}

// Force timezone to match app
date_default_timezone_set('Asia/Kolkata');

echo "<h2>Inserting Demo Tasks for Users 1 and 2</h2>";
echo "<pre>";

$conn->begin_transaction();

try {
    $today = date('Y-m-d');
    $yesterday = date('Y-m-d', strtotime('-1 day'));
    $two_days_ago = date('Y-m-d', strtotime('-2 days'));

    // ------------------------------------------------------------------
    // 1) Create a few demo tasks (both one-time and recurring)
    // ------------------------------------------------------------------
    $demoPrefix = 'DEMO-U12-';

    // Check existing demo tasks (avoid duplicates if run twice)
    $existingTaskNumbers = [];
    $checkTasksStmt = $conn->prepare("SELECT task_number FROM tasks WHERE task_number LIKE CONCAT(?, '%')");
    $checkTasksStmt->bind_param("s", $demoPrefix);
    $checkTasksStmt->execute();
    $checkResult = $checkTasksStmt->get_result();
    while ($row = $checkResult->fetch_assoc()) {
        $existingTaskNumbers[] = $row['task_number'];
    }
    $checkResult->free();
    $checkTasksStmt->close();

    // Ensure admin exists as creator
    $admin_id = 1;
    $adminRes = $conn->query("SELECT id FROM users WHERE role = 'admin' LIMIT 1");
    if ($adminRes && $adminRes->num_rows > 0) {
        $row = $adminRes->fetch_assoc();
        $admin_id = (int)$row['id'];
        $adminRes->free();
    }

    $demoTasks = [
        // For user 1
        [
            'task_number' => $demoPrefix . 'U1-COMPLETE',
            'name' => 'Demo Task - User1 Completed',
            'description' => 'Demo one-time task for User 1 (completed).',
            'priority' => 'high',
            'start_date' => $two_days_ago,
            'end_date' => $yesterday,
            'task_type' => 'one-time',
            'recurring_interval' => null,
            'skip_days' => null,
            'task_points' => 3,
            'quality_points' => 15,
            'lost_task_points' => -1,
            'notes' => 'Demo: completed',
        ],
        [
            'task_number' => $demoPrefix . 'U1-FAILED',
            'name' => 'Demo Task - User1 Failed',
            'description' => 'Demo one-time task for User 1 (failed).',
            'priority' => 'medium',
            'start_date' => $two_days_ago,
            'end_date' => $yesterday,
            'task_type' => 'one-time',
            'recurring_interval' => null,
            'skip_days' => null,
            'task_points' => 2,
            'quality_points' => 10,
            'lost_task_points' => -2,
            'notes' => 'Demo: failed',
        ],
        [
            'task_number' => $demoPrefix . 'U1-PENDING',
            'name' => 'Demo Task - User1 Pending Today',
            'description' => 'Demo one-time task for User 1 (pending, today).',
            'priority' => 'low',
            'start_date' => $today,
            'end_date' => $today,
            'task_type' => 'one-time',
            'recurring_interval' => null,
            'skip_days' => null,
            'task_points' => 1,
            'quality_points' => 5,
            'lost_task_points' => -1,
            'notes' => 'Demo: pending today',
        ],

        // For user 2
        [
            'task_number' => $demoPrefix . 'U2-COMPLETE',
            'name' => 'Demo Task - User2 Completed',
            'description' => 'Demo one-time task for User 2 (completed).',
            'priority' => 'high',
            'start_date' => $two_days_ago,
            'end_date' => $yesterday,
            'task_type' => 'one-time',
            'recurring_interval' => null,
            'skip_days' => null,
            'task_points' => 4,
            'quality_points' => 20,
            'lost_task_points' => -2,
            'notes' => 'Demo: completed',
        ],
        [
            'task_number' => $demoPrefix . 'U2-PENDING-REC',
            'name' => 'Demo Recurring Task - User2 Pending',
            'description' => 'Demo recurring task for User 2 (pending instance today).',
            'priority' => 'medium',
            'start_date' => $today,
            'end_date' => null,
            'task_type' => 'recurring',
            'recurring_interval' => 1,
            'skip_days' => '0,6',
            'task_points' => 2,
            'quality_points' => 10,
            'lost_task_points' => -1,
            'notes' => 'Demo: recurring pending',
        ],
    ];

    $taskInsert = $conn->prepare("
        INSERT INTO tasks (
            task_number, is_enabled, disable_reason, name, description, priority,
            start_date, end_date, start_time, end_time, task_type, recurring_interval,
            skip_days, task_points, quality_points, lost_task_points, notes, created_by
        )
        VALUES (?, 1, NULL, ?, ?, ?, ?, ?, NULL, NULL, ?, ?, ?, ?, ?, ?, ?, ?)
    ");

    $createdTasks = [];

    foreach ($demoTasks as $task) {
        if (in_array($task['task_number'], $existingTaskNumbers, true)) {
            echo "Task already exists: {$task['task_number']}\n";
            continue;
        }

        $taskInsert->bind_param(
            "sssssssisiisisi",
            $task['task_number'],
            $task['name'],
            $task['description'],
            $task['priority'],
            $task['start_date'],
            $task['end_date'],
            $task['task_type'],
            $task['recurring_interval'],
            $task['skip_days'],
            $task['task_points'],
            $task['quality_points'],
            $task['lost_task_points'],
            $task['notes'],
            $admin_id
        );

        if ($taskInsert->execute()) {
            $taskId = $conn->insert_id;
            $createdTasks[$task['task_number']] = $taskId;
            echo "Created task {$task['task_number']} (ID: {$taskId})\n";
        }
    }

    $taskInsert->close();

    // ------------------------------------------------------------------
    // 2) Assign tasks to users 1 and 2
    // ------------------------------------------------------------------
    $assignInsert = $conn->prepare("
        INSERT IGNORE INTO task_assignments (task_id, user_id)
        VALUES (?, ?)
    ");

    // Map: task_number => user_id
    $assignMap = [
        $demoPrefix . 'U1-COMPLETE'      => 1,
        $demoPrefix . 'U1-FAILED'        => 1,
        $demoPrefix . 'U1-PENDING'       => 1,
        $demoPrefix . 'U2-COMPLETE'      => 2,
        $demoPrefix . 'U2-PENDING-REC'   => 2,
    ];

    foreach ($assignMap as $taskNumber => $userId) {
        if (!isset($createdTasks[$taskNumber])) {
            // Task may have already existed; try to find its ID
            $stmt = $conn->prepare("SELECT id FROM tasks WHERE task_number = ?");
            $stmt->bind_param("s", $taskNumber);
            $stmt->execute();
            $res = $stmt->get_result();
            if ($res && $row = $res->fetch_assoc()) {
                $createdTasks[$taskNumber] = (int)$row['id'];
            }
            if ($res) {
                $res->free();
            }
            $stmt->close();
        }

        if (!isset($createdTasks[$taskNumber])) {
            echo "Skipping assignment; task not found: {$taskNumber}\n";
            continue;
        }

        $taskId = $createdTasks[$taskNumber];
        $assignInsert->bind_param("ii", $taskId, $userId);
        if ($assignInsert->execute()) {
            echo "Assigned task {$taskNumber} (ID: {$taskId}) to user {$userId}\n";
        }
    }

    $assignInsert->close();

    // ------------------------------------------------------------------
    // 3) Insert completions (completed + failed) for users 1 and 2
    // ------------------------------------------------------------------
    $completionInsert = $conn->prepare("
        INSERT INTO task_completions (
            task_id, user_id, completion_date, completion_url, completion_text,
            completion_image, status, task_points_earned, quality_points_earned,
            lost_task_points
        )
        VALUES (?, ?, ?, ?, ?, NULL, ?, ?, ?, ?)
    ");

    // Helper to avoid duplicate completion for same task/user/date
    $completionExists = $conn->prepare("
        SELECT id FROM task_completions
        WHERE task_id = ? AND user_id = ? AND completion_date = ?
    ");

    // User 1: one completed, one failed (yesterday)
    $u1CompletedTaskId = $createdTasks[$demoPrefix . 'U1-COMPLETE'] ?? null;
    $u1FailedTaskId    = $createdTasks[$demoPrefix . 'U1-FAILED'] ?? null;

    if ($u1CompletedTaskId) {
        $completionExists->bind_param("iis", $u1CompletedTaskId, $u1 = 1, $yesterday);
        $completionExists->execute();
        $res = $completionExists->get_result();
        if ($res->num_rows === 0) {
            $res->free();
            $completionInsert->bind_param(
                "iisssii i",
                $u1CompletedTaskId,
                $u1,
                $yesterday,
                $url1 = 'https://example.com/demo/u1-complete',
                $txt1 = 'Demo: User 1 completed this task.',
                $status1 = 'completed',
                $tp1 = 3,
                $qp1 = 15,
                $lost1 = 0
            );
            $completionInsert->execute();
            echo "Inserted COMPLETED completion for user 1, task DEMO-U12-U1-COMPLETE\n";
        } else {
            $res->free();
        }
    }

    if ($u1FailedTaskId) {
        $completionExists->bind_param("iis", $u1FailedTaskId, $u1 = 1, $yesterday);
        $completionExists->execute();
        $res = $completionExists->get_result();
        if ($res->num_rows === 0) {
            $res->free();
            $completionInsert->bind_param(
                "iisssii i",
                $u1FailedTaskId,
                $u1,
                $yesterday,
                $url2 = 'https://example.com/demo/u1-failed',
                $txt2 = 'Demo: User 1 failed this task.',
                $status2 = 'failed',
                $tp2 = 0,
                $qp2 = 0,
                $lost2 = 2
            );
            $completionInsert->execute();
            echo "Inserted FAILED completion for user 1, task DEMO-U12-U1-FAILED\n";
        } else {
            $res->free();
        }
    }

    // User 2: one completed, one failed (two_days_ago)
    $u2CompletedTaskId = $createdTasks[$demoPrefix . 'U2-COMPLETE'] ?? null;

    if ($u2CompletedTaskId) {
        $completionExists->bind_param("iis", $u2CompletedTaskId, $u2 = 2, $two_days_ago);
        $completionExists->execute();
        $res = $completionExists->get_result();
        if ($res->num_rows === 0) {
            $res->free();
            $completionInsert->bind_param(
                "iisssii i",
                $u2CompletedTaskId,
                $u2,
                $two_days_ago,
                $url3 = 'https://example.com/demo/u2-complete',
                $txt3 = 'Demo: User 2 completed this task.',
                $status3 = 'completed',
                $tp3 = 4,
                $qp3 = 20,
                $lost3 = 0
            );
            $completionInsert->execute();
            echo "Inserted COMPLETED completion for user 2, task DEMO-U12-U2-COMPLETE\n";
        } else {
            $res->free();
        }
    }

    // Pending examples:
    // - User 1: DEMO-U12-U1-PENDING has no completion record => pending
    // - User 2: DEMO-U12-U2-PENDING-REC is recurring starting today, no completion => pending

    $completionExists->close();
    $completionInsert->close();

    $conn->commit();

    echo "\nDone. Demo data inserted for users 1 and 2.\n";
    echo "You can now login as user1/user2 and see completed, failed, and pending tasks.\n";
    echo "</pre>";

} catch (Exception $e) {
    $conn->rollback();
    echo "ERROR: " . $e->getMessage() . "\n";
    echo "Transaction rolled back.\n";
    echo "</pre>";
}

$conn->close();
?>

