<?php
/**
 * Insert Dummy Data Script
 * This script inserts comprehensive dummy data for testing all features
 * Run this file once to populate the database with test data
 */

require_once 'config.php';

// Check if admin is logged in (optional - you can remove this check if needed)
// if (!isLoggedIn() || !isAdmin()) {
//     die("Please login as admin to run this script");
// }

$conn = getDBConnection();
if (!$conn) {
    die("Database connection failed: " . mysqli_connect_error());
}

// Start transaction
$conn->begin_transaction();

try {
    echo "<h2>Inserting Dummy Data...</h2>";
    echo "<pre>";
    
    // ============================================
    // 1. INSERT 5 DUMMY USERS
    // ============================================
    echo "1. Inserting 5 dummy users...\n";
    
    $users = [
        ['username' => 'john_doe', 'password' => 'password123', 'role' => 'user', 'email' => 'john.doe@example.com', 'full_name' => 'John Doe'],
        ['username' => 'jane_smith', 'password' => 'password123', 'role' => 'user', 'email' => 'jane.smith@example.com', 'full_name' => 'Jane Smith'],
        ['username' => 'bob_johnson', 'password' => 'password123', 'role' => 'user', 'email' => 'bob.johnson@example.com', 'full_name' => 'Bob Johnson'],
        ['username' => 'alice_williams', 'password' => 'password123', 'role' => 'user', 'email' => 'alice.williams@example.com', 'full_name' => 'Alice Williams'],
        ['username' => 'charlie_brown', 'password' => 'password123', 'role' => 'user', 'email' => 'charlie.brown@example.com', 'full_name' => 'Charlie Brown']
    ];
    
    $user_ids = [];
    $check_stmt = $conn->prepare("SELECT id FROM users WHERE username = ?");
    $user_stmt = $conn->prepare("INSERT INTO users (username, password, role, email, full_name) VALUES (?, ?, ?, ?, ?)");
    $points_stmt = $conn->prepare("INSERT INTO user_points (user_id) VALUES (?)");
    
    foreach ($users as $user) {
        // Check if user exists
        $check_stmt->bind_param("s", $user['username']);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if ($check_result && $check_result->num_rows > 0) {
            $result = $check_result->fetch_assoc();
            $user_ids[] = $result['id'];
            echo "  User '{$user['username']}' already exists (ID: {$result['id']})\n";
            $check_result->free();
            continue;
        }
        if ($check_result) {
            $check_result->free();
        }
        
        // Insert new user
        $user_stmt->bind_param("sssss", $user['username'], $user['password'], $user['role'], $user['email'], $user['full_name']);
        
        if ($user_stmt->execute()) {
            $user_id = $conn->insert_id;
            $user_ids[] = $user_id;
            
            // Initialize user points
            $points_stmt->bind_param("i", $user_id);
            $points_stmt->execute();
            
            echo "  ✓ Created user: {$user['username']} (ID: $user_id)\n";
        }
    }
    
    // Close statements
    $check_stmt->close();
    $user_stmt->close();
    $points_stmt->close();
    
    // Get admin user ID (assuming admin exists)
    $admin_result = $conn->query("SELECT id FROM users WHERE role = 'admin' LIMIT 1");
    $admin_id = 1; // Default
    if ($admin_result && $admin_result->num_rows > 0) {
        $admin_row = $admin_result->fetch_assoc();
        $admin_id = $admin_row['id'];
        $admin_result->free();
    }
    
    echo "  Total users ready: " . count($user_ids) . "\n\n";
    
    // ============================================
    // 2. INSERT 25+ DUMMY TASKS (covering all options)
    // ============================================
    echo "2. Inserting 25+ dummy tasks...\n";
    
    $today = date('Y-m-d');
    $yesterday = date('Y-m-d', strtotime('-1 day'));
    $tomorrow = date('Y-m-d', strtotime('+1 day'));
    $next_week = date('Y-m-d', strtotime('+7 days'));
    $last_week = date('Y-m-d', strtotime('-7 days'));
    
    $tasks = [
        // High Priority One-Time Tasks
        ['name' => 'Critical Bug Fix - Payment Gateway', 'description' => 'Fix critical bug in payment gateway that prevents transactions from processing. This is urgent and affects revenue.', 'priority' => 'high', 'start_date' => $yesterday, 'end_date' => $tomorrow, 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 10, 'quality_points' => 50, 'lost_task_points' => -5, 'notes' => 'Priority fix - affects production'],
        ['name' => 'Security Audit - Database Access', 'description' => 'Conduct comprehensive security audit of database access controls and permissions. Review all user roles and access levels.', 'priority' => 'high', 'start_date' => $today, 'end_date' => $next_week, 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 15, 'quality_points' => 75, 'lost_task_points' => -8, 'notes' => 'Security critical'],
        ['name' => 'Client Presentation - Q4 Results', 'description' => 'Prepare and deliver presentation to major client showcasing Q4 results and upcoming roadmap. Include metrics and achievements.', 'priority' => 'high', 'start_date' => $today, 'end_date' => date('Y-m-d', strtotime('+3 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 12, 'quality_points' => 60, 'lost_task_points' => -6, 'notes' => 'Client meeting scheduled'],
        
        // Medium Priority One-Time Tasks
        ['name' => 'Code Review - New Feature Branch', 'description' => 'Review pull request for new user authentication feature. Check code quality, security, and best practices.', 'priority' => 'medium', 'start_date' => $today, 'end_date' => $tomorrow, 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 5, 'quality_points' => 25, 'lost_task_points' => -2, 'notes' => 'PR #1234'],
        ['name' => 'Documentation Update - API Endpoints', 'description' => 'Update API documentation with new endpoints and request/response examples. Ensure all parameters are documented.', 'priority' => 'medium', 'start_date' => $today, 'end_date' => $next_week, 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 6, 'quality_points' => 30, 'lost_task_points' => -3, 'notes' => 'API v2 documentation'],
        ['name' => 'Database Optimization - Query Performance', 'description' => 'Optimize slow database queries identified in performance monitoring. Add indexes where needed.', 'priority' => 'medium', 'start_date' => $tomorrow, 'end_date' => date('Y-m-d', strtotime('+5 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 8, 'quality_points' => 40, 'lost_task_points' => -4, 'notes' => 'Performance improvement'],
        ['name' => 'UI/UX Improvements - Dashboard', 'description' => 'Implement UI/UX improvements for user dashboard based on user feedback. Improve navigation and visual design.', 'priority' => 'medium', 'start_date' => $today, 'end_date' => date('Y-m-d', strtotime('+4 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 7, 'quality_points' => 35, 'lost_task_points' => -3, 'notes' => 'User feedback implementation'],
        ['name' => 'Testing - New Feature Integration', 'description' => 'Write and execute comprehensive tests for new feature integration. Unit tests, integration tests, and E2E tests.', 'priority' => 'medium', 'start_date' => $tomorrow, 'end_date' => date('Y-m-d', strtotime('+6 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 9, 'quality_points' => 45, 'lost_task_points' => -4, 'notes' => 'Test coverage target: 80%'],
        
        // Low Priority One-Time Tasks
        ['name' => 'Email Template Design', 'description' => 'Design new email templates for user notifications. Ensure responsive design and brand consistency.', 'priority' => 'low', 'start_date' => $today, 'end_date' => $next_week, 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 3, 'quality_points' => 15, 'lost_task_points' => -1, 'notes' => 'Marketing team review needed'],
        ['name' => 'Archive Old Data', 'description' => 'Archive old database records older than 2 years to improve performance. Backup before archiving.', 'priority' => 'low', 'start_date' => $tomorrow, 'end_date' => date('Y-m-d', strtotime('+10 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 4, 'quality_points' => 20, 'lost_task_points' => -2, 'notes' => 'Low priority maintenance'],
        ['name' => 'Update Dependencies', 'description' => 'Update project dependencies to latest stable versions. Test for compatibility issues.', 'priority' => 'low', 'start_date' => $today, 'end_date' => date('Y-m-d', strtotime('+7 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 5, 'quality_points' => 25, 'lost_task_points' => -2, 'notes' => 'Dependency audit'],
        
        // Recurring Tasks - Daily (Weekdays)
        ['name' => 'Daily Standup Meeting', 'description' => 'Daily team standup meeting to discuss progress, blockers, and plans for the day. 15 minutes max.', 'priority' => 'low', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 1, 'skip_days' => '0,6', 'task_points' => 1, 'quality_points' => 5, 'lost_task_points' => -1, 'notes' => 'Every weekday at 9 AM'],
        ['name' => 'Daily Code Review', 'description' => 'Review at least 2 pull requests daily. Provide constructive feedback and ensure code quality standards.', 'priority' => 'medium', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 1, 'skip_days' => '0,6', 'task_points' => 2, 'quality_points' => 10, 'lost_task_points' => -1, 'notes' => 'Weekdays only'],
        ['name' => 'Daily Backup Verification', 'description' => 'Verify that daily database backups completed successfully. Check backup logs and test restore if needed.', 'priority' => 'high', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 1, 'skip_days' => '0,6', 'task_points' => 3, 'quality_points' => 15, 'lost_task_points' => -2, 'notes' => 'Critical - verify daily'],
        
        // Recurring Tasks - Weekly
        ['name' => 'Weekly Team Meeting', 'description' => 'Weekly team sync meeting to discuss progress, blockers, and upcoming priorities. Prepare agenda beforehand.', 'priority' => 'medium', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 7, 'skip_days' => null, 'task_points' => 4, 'quality_points' => 20, 'lost_task_points' => -2, 'notes' => 'Every Monday at 10 AM'],
        ['name' => 'Weekly Performance Report', 'description' => 'Generate and review weekly performance metrics report. Analyze trends and identify areas for improvement.', 'priority' => 'medium', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 7, 'skip_days' => null, 'task_points' => 5, 'quality_points' => 25, 'lost_task_points' => -2, 'notes' => 'Every Friday'],
        ['name' => 'Weekly Security Scan', 'description' => 'Run weekly security vulnerability scan on all systems. Review and address any critical findings.', 'priority' => 'high', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 7, 'skip_days' => null, 'task_points' => 6, 'quality_points' => 30, 'lost_task_points' => -3, 'notes' => 'Security compliance'],
        
        // Recurring Tasks - Bi-weekly
        ['name' => 'Bi-weekly Sprint Planning', 'description' => 'Plan upcoming sprint tasks, assign priorities, and estimate effort. Review previous sprint results.', 'priority' => 'high', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 14, 'skip_days' => null, 'task_points' => 8, 'quality_points' => 40, 'lost_task_points' => -4, 'notes' => 'Every 2 weeks'],
        ['name' => 'Bi-weekly Client Update', 'description' => 'Send bi-weekly progress update to clients. Include completed tasks, upcoming milestones, and any blockers.', 'priority' => 'medium', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 14, 'skip_days' => null, 'task_points' => 5, 'quality_points' => 25, 'lost_task_points' => -2, 'notes' => 'Client communication'],
        
        // Recurring Tasks - Monthly
        ['name' => 'Monthly System Maintenance', 'description' => 'Perform monthly system maintenance including updates, patches, and performance optimization.', 'priority' => 'medium', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 30, 'skip_days' => null, 'task_points' => 10, 'quality_points' => 50, 'lost_task_points' => -5, 'notes' => 'First of each month'],
        ['name' => 'Monthly Financial Report', 'description' => 'Generate monthly financial report with revenue, expenses, and profit analysis. Present to management.', 'priority' => 'high', 'start_date' => $today, 'end_date' => null, 'task_type' => 'recurring', 'recurring_interval' => 30, 'skip_days' => null, 'task_points' => 12, 'quality_points' => 60, 'lost_task_points' => -6, 'notes' => 'End of month'],
        
        // More One-Time Tasks with Various Dates
        ['name' => 'Deploy New Feature to Production', 'description' => 'Deploy new user authentication feature to production environment. Follow deployment checklist.', 'priority' => 'high', 'start_date' => $tomorrow, 'end_date' => date('Y-m-d', strtotime('+2 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 15, 'quality_points' => 75, 'lost_task_points' => -8, 'notes' => 'Production deployment'],
        ['name' => 'User Training Session', 'description' => 'Conduct training session for new users on system features and best practices. Prepare training materials.', 'priority' => 'medium', 'start_date' => date('Y-m-d', strtotime('+3 days')), 'end_date' => date('Y-m-d', strtotime('+5 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 7, 'quality_points' => 35, 'lost_task_points' => -3, 'notes' => 'New user onboarding'],
        ['name' => 'Bug Fix - Login Issue', 'description' => 'Fix reported bug where users cannot login with special characters in password. Test and deploy fix.', 'priority' => 'high', 'start_date' => $yesterday, 'end_date' => $today, 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 6, 'quality_points' => 30, 'lost_task_points' => -3, 'notes' => 'User reported bug'],
        ['name' => 'Content Update - Website', 'description' => 'Update website content with latest product information and pricing. Ensure SEO optimization.', 'priority' => 'low', 'start_date' => $today, 'end_date' => date('Y-m-d', strtotime('+8 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 4, 'quality_points' => 20, 'lost_task_points' => -2, 'notes' => 'Marketing content'],
        ['name' => 'Server Migration Planning', 'description' => 'Plan migration to new server infrastructure. Document migration steps and create rollback plan.', 'priority' => 'high', 'start_date' => $today, 'end_date' => date('Y-m-d', strtotime('+14 days')), 'task_type' => 'one-time', 'recurring_interval' => null, 'skip_days' => null, 'task_points' => 20, 'quality_points' => 100, 'lost_task_points' => -10, 'notes' => 'Infrastructure upgrade'],
    ];
    
    $task_ids = [];
    $task_stmt = $conn->prepare("
        INSERT INTO tasks (task_number, name, description, priority, start_date, end_date, task_type, 
                         recurring_interval, skip_days, task_points, quality_points, 
                         lost_task_points, notes, created_by, is_enabled)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1)
    ");
    
    foreach ($tasks as $task) {
        $task_number = generateTaskNumber();
        $task_stmt->bind_param("sssssssisiiisi", 
            $task_number,
            $task['name'], 
            $task['description'], 
            $task['priority'], 
            $task['start_date'], 
            $task['end_date'], 
            $task['task_type'], 
            $task['recurring_interval'], 
            $task['skip_days'], 
            $task['task_points'], 
            $task['quality_points'], 
            $task['lost_task_points'], 
            $task['notes'], 
            $admin_id
        );
        
        if ($task_stmt->execute()) {
            $task_id = $conn->insert_id;
            $task_ids[] = $task_id;
            echo "  ✓ Created task: {$task['name']} (ID: $task_id)\n";
        }
    }
    
    echo "  Total tasks created: " . count($task_ids) . "\n\n";
    
    // Close task insert statement
    $task_stmt->close();
    
    // ============================================
    // 3. INSERT TASK ASSIGNMENTS
    // ============================================
    echo "3. Inserting task assignments...\n";
    
    $assign_stmt = $conn->prepare("INSERT INTO task_assignments (task_id, user_id) VALUES (?, ?)");
    $assign_check_stmt = $conn->prepare("SELECT id FROM task_assignments WHERE task_id = ? AND user_id = ?");
    $assignment_count = 0;
    
    // Assign tasks to users (mix of single and multiple assignments)
    foreach ($task_ids as $index => $task_id) {
        // Assign to 1-3 random users per task
        $num_assignments = rand(1, min(3, count($user_ids)));
        $selected_users = array_rand($user_ids, $num_assignments);
        
        if (!is_array($selected_users)) {
            $selected_users = [$selected_users];
        }
        
        foreach ($selected_users as $user_index) {
            $user_id = $user_ids[$user_index];
            
            // Check if assignment already exists
            $assign_check_stmt->bind_param("ii", $task_id, $user_id);
            $assign_check_stmt->execute();
            $assign_check_result = $assign_check_stmt->get_result();
            
            if ($assign_check_result->num_rows > 0) {
                $assign_check_result->free();
                continue;
            }
            $assign_check_result->free();
            
            $assign_stmt->bind_param("ii", $task_id, $user_id);
            if ($assign_stmt->execute()) {
                $assignment_count++;
            }
        }
    }
    
    // Close statement
    $assign_check_stmt->close();
    $assign_stmt->close();
    
    echo "  Total assignments created: $assignment_count\n\n";
    
    // ============================================
    // 4. INSERT TASK COMPLETIONS (mix of completed and failed)
    // ============================================
    echo "4. Inserting task completions...\n";
    
    $completion_stmt = $conn->prepare("
        INSERT INTO task_completions (task_id, user_id, completion_date, completion_url, 
                                     completion_text, status, task_points_earned, 
                                     quality_points_earned, lost_task_points)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $completion_count = 0;
    // Complete some tasks (about 30% of assigned tasks)
    $completion_samples = min(15, count($task_ids));
    
    $assigned_users_stmt = $conn->prepare("SELECT user_id FROM task_assignments WHERE task_id = ?");
    $task_details_stmt = $conn->prepare("SELECT task_points, quality_points, lost_task_points FROM tasks WHERE id = ?");
    $completion_check_stmt = $conn->prepare("SELECT id FROM task_completions WHERE task_id = ? AND user_id = ?");
    $update_points_stmt = $conn->prepare("
        UPDATE user_points 
        SET total_task_points = total_task_points + ?,
            total_quality_points = total_quality_points + ?,
            total_lost_task_points = total_lost_task_points + ?
        WHERE user_id = ?
    ");
    
    for ($i = 0; $i < $completion_samples; $i++) {
        $task_id = $task_ids[array_rand($task_ids)];
        
        // Get users assigned to this task
        $assigned_users_stmt->bind_param("i", $task_id);
        $assigned_users_stmt->execute();
        $assigned_result = $assigned_users_stmt->get_result();
        
        if ($assigned_result->num_rows > 0) {
            $assigned_row = $assigned_result->fetch_assoc();
            $user_id = $assigned_row['user_id'];
            $assigned_result->free();
            
            // Get task details for points
            $task_details_stmt->bind_param("i", $task_id);
            $task_details_stmt->execute();
            $task_result = $task_details_stmt->get_result()->fetch_assoc();
            
            $status = (rand(1, 10) > 1) ? 'completed' : 'failed'; // 90% completed, 10% failed
            $completion_date = date('Y-m-d', strtotime('-' . rand(0, 30) . ' days'));
            
            $completion_url = ($status === 'completed') ? 'https://example.com/task-' . $task_id : null;
            $completion_text = ($status === 'completed') ? 'Task completed successfully. All requirements met.' : 'Task failed due to technical issues.';
            
            $task_points_earned = ($status === 'completed') ? $task_result['task_points'] : 0;
            $quality_points_earned = ($status === 'completed') ? $task_result['quality_points'] : 0;
            $lost_points = ($status === 'failed') ? abs($task_result['lost_task_points']) : 0;
            
            // Check if completion already exists
            $completion_check_stmt->bind_param("ii", $task_id, $user_id);
            $completion_check_stmt->execute();
            $completion_check_result = $completion_check_stmt->get_result();
            
            if ($completion_check_result->num_rows > 0) {
                $completion_check_result->free();
                continue;
            }
            $completion_check_result->free();
            
            $completion_stmt->bind_param("iissssiii", 
                $task_id, 
                $user_id, 
                $completion_date, 
                $completion_url, 
                $completion_text, 
                $status, 
                $task_points_earned, 
                $quality_points_earned, 
                $lost_points
            );
            
            if ($completion_stmt->execute()) {
                $completion_count++;
                
                // Update user points
                $update_points_stmt->bind_param("iiii", $task_points_earned, $quality_points_earned, $lost_points, $user_id);
                $update_points_stmt->execute();
            }
        } else {
            $assigned_result->free();
        }
    }
    
    // Close statements
    $assigned_users_stmt->close();
    $task_details_stmt->close();
    $completion_check_stmt->close();
    $update_points_stmt->close();
    $completion_stmt->close();
    
    echo "  Total completions created: $completion_count\n\n";
    
    // ============================================
    // 5. INSERT NOTIFICATIONS
    // ============================================
    echo "5. Inserting notifications...\n";
    
    $notif_stmt = $conn->prepare("
        INSERT INTO notifications (user_id, title, message, type, is_read)
        VALUES (?, ?, ?, ?, ?)
    ");
    
    $notification_types = ['info', 'success', 'warning', 'danger'];
    $notifications = [
        ['title' => 'New Task Assigned', 'message' => 'You have been assigned a new high-priority task. Please review and start working on it.', 'type' => 'info'],
        ['title' => 'Task Completed', 'message' => 'Great job! You have successfully completed a task and earned points.', 'type' => 'success'],
        ['title' => 'Task Deadline Approaching', 'message' => 'You have a task deadline approaching soon. Please prioritize accordingly.', 'type' => 'warning'],
        ['title' => 'Task Overdue', 'message' => 'Warning: You have an overdue task. Please complete it as soon as possible.', 'type' => 'danger'],
        ['title' => 'Points Updated', 'message' => 'Your points have been updated based on recent task completions.', 'type' => 'info'],
        ['title' => 'Weekly Report Available', 'message' => 'Your weekly performance report is now available for review.', 'type' => 'info'],
    ];
    
    $notif_count = 0;
    foreach ($user_ids as $user_id) {
        // Create 3-5 notifications per user
        $num_notifs = rand(3, 5);
        for ($i = 0; $i < $num_notifs; $i++) {
            $notif = $notifications[array_rand($notifications)];
            $is_read = rand(0, 1); // Random read/unread
            
            $notif_stmt->bind_param("isssi", $user_id, $notif['title'], $notif['message'], $notif['type'], $is_read);
            if ($notif_stmt->execute()) {
                $notif_count++;
            }
        }
    }
    
    echo "  Total notifications created: $notif_count\n\n";
    // Close notifications statement
    $notif_stmt->close();
    
    // Commit transaction
    $conn->commit();
    
    echo "========================================\n";
    echo "✓ DUMMY DATA INSERTION COMPLETED!\n";
    echo "========================================\n";
    echo "\nSummary:\n";
    echo "- Users: " . count($user_ids) . " (plus existing admin)\n";
    echo "- Tasks: " . count($task_ids) . "\n";
    echo "- Task Assignments: $assignment_count\n";
    echo "- Task Completions: $completion_count\n";
    echo "- Notifications: $notif_count\n";
    echo "\nYou can now login and test all features!\n";
    echo "</pre>";
    
} catch (Exception $e) {
    // Rollback on error
    $conn->rollback();
    echo "<pre>";
    echo "ERROR: " . $e->getMessage() . "\n";
    echo "Transaction rolled back.\n";
    echo "</pre>";
}

$conn->close();
?>
