<?php
// Set timezone to Asia/Kolkata (Kolkata) for all date/time operations
// Use both date_default_timezone_set and ini_set to ensure it's set correctly
ini_set('date.timezone', 'Asia/Kolkata');
date_default_timezone_set('Asia/Kolkata');

// Database Configuration
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'newpanca_taskmanagement');

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    // Configure session settings
    ini_set('session.cookie_httponly', 1);
    ini_set('session.use_only_cookies', 1);
    ini_set('session.cookie_samesite', 'Lax');
    
    session_start();
}

// Database connection function
function getDBConnection() {
    static $conn = null;
    if ($conn === null) {
        try {
            $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
            if ($conn->connect_error) {
                error_log("Database connection failed: " . $conn->connect_error);
                return null;
            }
            $conn->set_charset("utf8mb4");
            // Set MySQL session timezone to Asia/Kolkata
            $conn->query("SET time_zone = '+05:30'");
        } catch (Exception $e) {
            error_log("Database connection error: " . $e->getMessage());
            return null;
        }
    }
    return $conn;
}

// Check if user is logged in
function isLoggedIn() {
    return isset($_SESSION['user_id']) && isset($_SESSION['username']);
}

// Check if user is admin
function isAdmin() {
    return isset($_SESSION['role']) && $_SESSION['role'] === 'admin';
}

// Get current user ID
function getCurrentUserId() {
    return $_SESSION['user_id'] ?? null;
}

// Get current username
function getCurrentUsername() {
    return $_SESSION['username'] ?? 'Guest';
}

// Get current user role
function getCurrentUserRole() {
    return $_SESSION['role'] ?? 'user';
}

// Check if user is on leave on a specific date
function isUserOnLeave($user_id, $date) {
    $conn = getDBConnection();
    if (!$conn) return false;
    
    $stmt = $conn->prepare("SELECT id FROM user_leaves WHERE user_id = ? AND leave_date = ?");
    $stmt->bind_param("is", $user_id, $date);
    $stmt->execute();
    $result = $stmt->get_result();
    $is_on_leave = $result->num_rows > 0;
    $result->free();
    $stmt->close();
    
    return $is_on_leave;
}

// Get all leave dates for a user within a date range
function getUserLeaveDates($user_id, $start_date = null, $end_date = null) {
    $conn = getDBConnection();
    if (!$conn) return [];
    
    $leaves = [];
    if ($start_date && $end_date) {
        $stmt = $conn->prepare("SELECT leave_date FROM user_leaves WHERE user_id = ? AND leave_date BETWEEN ? AND ? ORDER BY leave_date ASC");
        $stmt->bind_param("iss", $user_id, $start_date, $end_date);
    } else {
        $stmt = $conn->prepare("SELECT leave_date FROM user_leaves WHERE user_id = ? ORDER BY leave_date ASC");
        $stmt->bind_param("i", $user_id);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $leaves[] = $row['leave_date'];
    }
    $result->free();
    $stmt->close();
    
    return $leaves;
}

// Generate unique task number (format: TSK + 12 digits + 4 random digits)
function generateTaskNumber() {
    $conn = getDBConnection();
    if (!$conn) {
        // Fallback if DB not available
        return 'TSK' . str_pad(time(), 12, '0', STR_PAD_LEFT) . str_pad(rand(0, 9999), 4, '0', STR_PAD_LEFT);
    }
    
    // Generate unique task number
    $maxAttempts = 10;
    $attempt = 0;
    
    while ($attempt < $maxAttempts) {
        // Format: TSK + timestamp (12 digits) + random (4 digits)
        $timestamp = time();
        $random = rand(1000, 9999);
        $taskNumber = 'TSK' . str_pad($timestamp, 12, '0', STR_PAD_LEFT) . $random;
        
        // Check if it already exists
        $stmt = $conn->prepare("SELECT id FROM tasks WHERE task_number = ?");
        $stmt->bind_param("s", $taskNumber);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            $stmt->close();
            return $taskNumber;
        }
        
        $stmt->close();
        $attempt++;
        usleep(1000); // Small delay to ensure different timestamp
    }
    
    // Fallback if all attempts failed
    return 'TSK' . str_pad(time(), 12, '0', STR_PAD_LEFT) . str_pad(rand(0, 9999), 4, '0', STR_PAD_LEFT);
}

// Create next recurring task instance when current one fails
function createNextRecurringInstance($parent_task_id, $current_instance_date) {
    $conn = getDBConnection();
    if (!$conn) return false;
    
    try {
        // Get parent task details
        $task_stmt = $conn->prepare("SELECT start_time, end_time, skip_days FROM tasks WHERE id = ? AND task_type = 'recurring'");
        $task_stmt->bind_param("i", $parent_task_id);
        $task_stmt->execute();
        $task_result = $task_stmt->get_result();
        $task = $task_result->fetch_assoc();
        
        if (!$task) return false;
        
        // Get recurring days (stored in skip_days field, repurposed)
        $recurring_days = $task['skip_days'] ? explode(',', $task['skip_days']) : [];
        if (empty($recurring_days)) {
            return false; // No recurring days defined
        }
        
        // Find next occurrence date that matches one of the recurring days
        $instance_date = new DateTime($current_instance_date, new DateTimeZone('Asia/Kolkata'));
        $next_date = clone $instance_date;
        $max_date = date('Y-m-d', strtotime('+30 days'));
        $max_attempts = 14; // Check up to 2 weeks ahead
        $found_next = false;
        
        for ($i = 0; $i < $max_attempts; $i++) {
            $next_date->modify("+1 day");
            $next_date_str = $next_date->format('Y-m-d');
            $next_day_of_week = (int)$next_date->format('w');
            
            // Check if this day is in the recurring days list
            if (in_array((string)$next_day_of_week, $recurring_days) && $next_date_str <= $max_date) {
                $found_next = true;
                break;
            }
        }
        
        if (!$found_next) {
            return false; // Couldn't find next occurrence within 2 weeks
        }
        
        // Check if next instance already exists
        $check_stmt = $conn->prepare("SELECT id FROM recurring_task_instances WHERE parent_task_id = ? AND instance_date = ?");
        $check_stmt->bind_param("is", $parent_task_id, $next_date_str);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if ($check_result->num_rows === 0 && $next_date_str <= date('Y-m-d', strtotime('+30 days'))) {
            // Create next instance
            $insert_stmt = $conn->prepare("
                INSERT INTO recurring_task_instances 
                (parent_task_id, instance_date, start_time, end_time, status)
                VALUES (?, ?, ?, ?, 'pending')
            ");
            $insert_stmt->bind_param("isss", 
                $parent_task_id,
                $next_date_str,
                $task['start_time'],
                $task['end_time']
            );
            
            if ($insert_stmt->execute()) {
                return true;
            }
        }
        
        return false;
    } catch (Exception $e) {
        error_log("Error creating next recurring instance: " . $e->getMessage());
        return false;
    }
}
?>
