<?php
/**
 * Cronjob for Recurring Tasks
 * 
 * This script should be run via cron to automatically generate recurring tasks.
 * 
 * Setup cron job (run daily at midnight):
 * 0 0 * * * /usr/bin/php /path/to/your/project/cronjob.php
 * 
 * Or run every hour:
 * 0 * * * * /usr/bin/php /path/to/your/project/cronjob.php
 */

require_once 'config.php';

// Set execution time limit
set_time_limit(300); // 5 minutes

// Log file
$log_file = __DIR__ . '/cronjob.log';

function logMessage($message) {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] $message\n";
    file_put_contents($log_file, $log_entry, FILE_APPEND);
    echo $log_entry;
}

logMessage("Cronjob started");

try {
    $conn = getDBConnection();
    
    if (!$conn) {
        logMessage("ERROR: Database connection failed");
        exit(1);
    }
    
    // Get current date and time in Kolkata timezone
    $today = date('Y-m-d');
    $now = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
    $day_of_week = (int)$now->format('w'); // 0 = Sunday, 1 = Monday, ..., 6 = Saturday
    
    logMessage("Processing recurring tasks for date: $today (Day of week: $day_of_week)");
    
    // Get all enabled recurring tasks
    $stmt = $conn->prepare("
        SELECT * FROM tasks 
        WHERE task_type = 'recurring' 
        AND is_enabled = 1
        AND start_date <= ?
    ");
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $tasks_processed = 0;
    $instances_created = 0;
    
    while ($task = $result->fetch_assoc()) {
        $tasks_processed++;
        
        // Get recurring days (stored in skip_days field, repurposed)
        $recurring_days = $task['skip_days'] ? explode(',', $task['skip_days']) : [];
        
        // Check if today's day of week is in the recurring days list
        if (empty($recurring_days) || !in_array((string)$day_of_week, $recurring_days)) {
            logMessage("Skipping task ID {$task['id']} - day $day_of_week is not in recurring days list");
            continue;
        }
        
        // Check if task has started (start_date must be today or earlier)
        $start_date = new DateTime($task['start_date'], new DateTimeZone('Asia/Kolkata'));
        if ($start_date->format('Y-m-d') > $today) {
            logMessage("Skipping task ID {$task['id']} - start date is in the future");
            continue;
        }
        
        // Today matches one of the recurring days, create instance
        {
        // Check if instance already exists for today
        $check_stmt = $conn->prepare("
            SELECT id FROM recurring_task_instances 
            WHERE parent_task_id = ? AND instance_date = ?
        ");
        $check_stmt->bind_param("is", $task['id'], $today);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if ($check_result->num_rows === 0) {
            // Create a new task instance for today
            // For recurring tasks, each day gets its own instance with the same end time
            $instance_end_date = $today; // End date is same as start date for daily recurring tasks
            
            // Create the instance
            $insert_stmt = $conn->prepare("
                INSERT INTO recurring_task_instances 
                (parent_task_id, instance_date, start_time, end_time, status)
                VALUES (?, ?, ?, ?, 'pending')
            ");
            $insert_stmt->bind_param("isss", 
                $task['id'], 
                $today, 
                $task['start_time'], 
                $task['end_time']
            );
            
            if ($insert_stmt->execute()) {
                $instances_created++;
                logMessage("Created recurring task instance for task ID {$task['id']} on date $today");
                
                // Create notifications for assigned users (skip if user is on leave)
                $assign_stmt = $conn->prepare("SELECT user_id FROM task_assignments WHERE task_id = ?");
                $assign_stmt->bind_param("i", $task['id']);
                $assign_stmt->execute();
                $assign_result = $assign_stmt->get_result();
                
                while ($assignment = $assign_result->fetch_assoc()) {
                    $user_id = $assignment['user_id'];
                    
                    // Check if user is on leave today
                    if (isUserOnLeave($user_id, $today)) {
                        logMessage("Skipping notification for user ID $user_id - user is on leave today");
                        continue;
                    }
                    
                    $notif_stmt = $conn->prepare("
                        INSERT INTO notifications (user_id, title, message, type)
                        VALUES (?, 'Recurring Task Due', ?, 'info')
                    ");
                    $message = "Your recurring task '{$task['name']}' is due today.";
                    $notif_stmt->bind_param("is", $user_id, $message);
                    $notif_stmt->execute();
                }
            } else {
                logMessage("ERROR: Failed to create instance for task ID {$task['id']}: " . $conn->error);
            }
        } else {
            logMessage("Instance for task ID {$task['id']} on date $today already exists - skipping");
        }
    }
    
    // Also check for failed instances and create next occurrence if needed
    logMessage("Checking for failed instances to create next occurrences...");
    $failed_check = $conn->query("
        SELECT rti.*, t.start_time, t.end_time, t.skip_days
        FROM recurring_task_instances rti
        JOIN tasks t ON rti.parent_task_id = t.id
        WHERE rti.status = 'failed'
        AND rti.instance_date < CURDATE()
        AND t.task_type = 'recurring'
        AND t.is_enabled = 1
    ");
    
    while ($failed_instance = $failed_check->fetch_assoc()) {
        // Get recurring days (stored in skip_days field)
        $recurring_days = $failed_instance['skip_days'] ? explode(',', $failed_instance['skip_days']) : [];
        if (empty($recurring_days)) {
            continue; // No recurring days defined, skip
        }
        
        // Find next occurrence date that matches one of the recurring days
        $instance_date = new DateTime($failed_instance['instance_date'], new DateTimeZone('Asia/Kolkata'));
        $next_date = clone $instance_date;
        $max_attempts = 14; // Check up to 2 weeks ahead
        $found_next = false;
        
        for ($i = 0; $i < $max_attempts; $i++) {
            $next_date->modify("+1 day");
            $next_date_str = $next_date->format('Y-m-d');
            $next_day_of_week = (int)$next_date->format('w');
            
            // Check if this day is in the recurring days list
            if (in_array((string)$next_day_of_week, $recurring_days)) {
                $found_next = true;
                break;
            }
        }
        
        if (!$found_next) {
            continue; // Couldn't find next occurrence within 2 weeks
        }
        
        // Check if next instance already exists
        $check_next = $conn->prepare("
            SELECT id FROM recurring_task_instances 
            WHERE parent_task_id = ? AND instance_date = ?
        ");
        $check_next->bind_param("is", $failed_instance['parent_task_id'], $next_date_str);
        $check_next->execute();
        $check_next_result = $check_next->get_result();
        
        if ($check_next_result->num_rows === 0 && $next_date_str <= date('Y-m-d', strtotime('+30 days'))) {
            // Create next instance
            $create_next = $conn->prepare("
                INSERT INTO recurring_task_instances 
                (parent_task_id, instance_date, start_time, end_time, status)
                VALUES (?, ?, ?, ?, 'pending')
            ");
            $create_next->bind_param("isss", 
                $failed_instance['parent_task_id'],
                $next_date_str,
                $failed_instance['start_time'],
                $failed_instance['end_time']
            );
            
            if ($create_next->execute()) {
                $instances_created++;
                logMessage("Created next instance for failed task ID {$failed_instance['parent_task_id']} on date $next_date_str");
            }
        }
    }
    
    logMessage("Cronjob completed: Processed $tasks_processed tasks, Created $instances_created instances");
    
    $conn->close();
    
} catch (Exception $e) {
    logMessage("ERROR: " . $e->getMessage());
    exit(1);
}

logMessage("Cronjob finished successfully");
exit(0);
?>
