<?php
require_once 'config.php';

if (!isLoggedIn() || !isAdmin()) {
    header('Location: index.php?error=' . urlencode('Unauthorized'));
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: index.php?error=' . urlencode('Method not allowed'));
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    header('Location: index.php?task_error=' . urlencode('Database connection failed'));
    exit;
}

$user_id = getCurrentUserId();
$task_id = intval($_POST['task_id'] ?? 0);

if (!$task_id) {
    header('Location: index.php?task_error=' . urlencode('Task ID is required'));
    exit;
}

$name = $_POST['name'] ?? '';
$description = $_POST['description'] ?? '';
$priority = $_POST['priority'] ?? 'medium';
$start_date = $_POST['start_date'] ?? date('Y-m-d');
$end_date = $_POST['end_date'] ?? null;
$start_time = $_POST['start_time'] ?? null;
$end_time = $_POST['end_time'] ?? null;
$task_type = $_POST['task_type'] ?? 'one-time';
$recurring_interval = null; // No longer used, kept for backward compatibility
$recurring_days = $_POST['recurring_days'] ?? [];
$skip_days = null; // Repurposed: now stores the days to repeat on
$task_points = intval($_POST['task_points'] ?? 1);
$quality_points = intval($_POST['quality_points'] ?? 10);
$lost_task_points = intval($_POST['lost_task_points'] ?? -1);
$notes = $_POST['notes'] ?? null;
$assigned_users = $_POST['assigned_users'] ?? [];

if (empty($name) || empty($description)) {
    header('Location: index.php?task_error=' . urlencode('Name and description are required'));
    exit;
}

if (empty($assigned_users)) {
    header('Location: index.php?task_error=' . urlencode('At least one user must be assigned'));
    exit;
}

// Handle recurring days - store in skip_days field (repurposed)
if ($task_type === 'recurring') {
    if (is_array($recurring_days) && !empty($recurring_days)) {
        // Validate that at least one day is selected
        $skip_days = implode(',', array_map('intval', $recurring_days));
    } else {
        header('Location: index.php?task_error=' . urlencode('At least one repeat day must be selected for recurring tasks'));
        exit;
    }
} else {
    $recurring_interval = null;
    $skip_days = null;
}

$stmt = $conn->prepare("
    UPDATE tasks SET name=?, description=?, priority=?, start_date=?, end_date=?, start_time=?, end_time=?, 
                   task_type=?, recurring_interval=?, skip_days=?, task_points=?, 
                   quality_points=?, lost_task_points=?, notes=?
    WHERE id=?
");
$stmt->bind_param("ssssssssisiiisi", $name, $description, $priority, $start_date, $end_date, 
                 $start_time, $end_time, $task_type, $recurring_interval, $skip_days, $task_points, $quality_points, 
                 $lost_task_points, $notes, $task_id);

if ($stmt->execute()) {
    // Update assignments
    $delete_stmt = $conn->prepare("DELETE FROM task_assignments WHERE task_id = ?");
    $delete_stmt->bind_param("i", $task_id);
    $delete_stmt->execute();
    $delete_stmt->close();
    
    if (!empty($assigned_users)) {
        $assign_stmt = $conn->prepare("INSERT INTO task_assignments (task_id, user_id) VALUES (?, ?)");
        if ($assign_stmt) {
            foreach ($assigned_users as $uid) {
                $uid = intval($uid);
                if ($uid > 0) {
                    $assign_stmt->bind_param("ii", $task_id, $uid);
                    if (!$assign_stmt->execute()) {
                        error_log("Failed to assign task $task_id to user $uid: " . $assign_stmt->error);
                    }
                }
            }
            $assign_stmt->close();
        }
    }
    
    header('Location: index.php?success=' . urlencode('Task updated successfully!') . '#admin-tasks');
    exit;
} else {
    header('Location: index.php?task_error=' . urlencode('Failed to update task: ' . $conn->error));
    exit;
}

$conn->close();
?>
