<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Set error handler to catch fatal errors
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        error_log("Fatal error in index.php: " . $error['message'] . " in " . $error['file'] . " on line " . $error['line']);
        http_response_code(500);
        echo "<!DOCTYPE html><html><head><title>Error</title></head><body>";
        echo "<h1>500 Internal Server Error</h1>";
        echo "<p>An error occurred. Please check the error log for details.</p>";
        echo "<p>Error: " . htmlspecialchars($error['message']) . "</p>";
        echo "<p>File: " . htmlspecialchars($error['file']) . "</p>";
        echo "<p>Line: " . $error['line'] . "</p>";
        echo "</body></html>";
        exit;
    }
});

require_once 'config.php';

// Helper function for HTML escaping
if (!function_exists('escapeHtml')) {
    function escapeHtml($text) {
        if ($text === null) return '';
        return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
    }
}

try {
    $is_logged_in = isLoggedIn();
    $user_role = getCurrentUserRole();
    $username = getCurrentUsername();
} catch (Exception $e) {
    error_log("Error in index.php initialization: " . $e->getMessage());
    http_response_code(500);
    die("An error occurred during initialization. Please check the error log.");
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Task Management System</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <!-- Login Screen -->
    <div id="loginScreen" class="screen <?php echo $is_logged_in ? '' : 'active'; ?>">
        <div class="login-container">
            <div class="login-card">
                <h1>Task Management</h1>
                <form method="POST" action="login.php">
                    <?php if (isset($_GET['error'])): ?>
                        <div class="alert alert-error" style="margin-bottom: 15px; padding: 10px; background: #fee; color: #c33; border-radius: 4px;">
                            <?php echo htmlspecialchars($_GET['error']); ?>
                        </div>
                    <?php endif; ?>
                    <div class="form-group">
                        <label for="username">Username</label>
                        <input type="text" id="username" name="username" required>
                    </div>
                    <div class="form-group">
                        <label for="password">Password</label>
                        <input type="password" id="password" name="password" required>
                    </div>
                    <button type="submit" class="btn btn-primary">Login</button>
                </form>
                <p class="login-hint">Default: admin/password123 or user1/password123</p>
            </div>
        </div>
    </div>

    <!-- Main App Screen -->
    <div id="appScreen" class="screen <?php echo $is_logged_in ? 'active' : ''; ?>">
        <!-- Navbar -->
        <nav class="navbar">
            <div class="nav-left">
                <h2 class="nav-title">Task Manager</h2>
            </div>
            <div class="nav-right">
                <?php
                $notification_count = 0;
                if ($is_logged_in && isset($_SESSION['user_id'])) {
                    try {
                        $conn = getDBConnection();
                        if ($conn) {
                            $notif_count_stmt = $conn->prepare("SELECT COUNT(*) as count FROM notifications WHERE user_id = ? AND is_read = 0");
                            if ($notif_count_stmt) {
                                $notif_count_stmt->bind_param("i", $_SESSION['user_id']);
                                $notif_count_stmt->execute();
                                $notif_count_result = $notif_count_stmt->get_result();
                                if ($notif_count_result) {
                                    $notification_count = $notif_count_result->fetch_assoc()['count'] ?? 0;
                                }
                                $notif_count_stmt->close();
                            }
                        }
                    } catch (Exception $e) {
                        // Silently fail - notification count will be 0
                        $notification_count = 0;
                    }
                }
                ?>
                <?php if ($is_logged_in): ?>
                <button id="notificationBtn" class="nav-icon-btn" title="Notifications" style="display: <?php echo $notification_count > 0 ? 'flex' : 'none'; ?>;">
                    <span class="icon">🔔</span>
                    <?php if ($notification_count > 0): ?>
                    <span id="notificationBadge" class="badge"><?php echo $notification_count; ?></span>
                    <?php endif; ?>
                </button>
                <?php endif; ?>
                <button id="themeToggle" class="nav-icon-btn" title="Toggle Theme">
                    <span class="icon" id="themeIcon">🌙</span>
                </button>
                <div class="user-menu">
                    <span id="userNameDisplay"><?php echo htmlspecialchars($username); ?></span>
                    <button id="logoutBtn" class="btn btn-small">Logout</button>
                </div>
            </div>
        </nav>

        <!-- Notification Dropdown -->
        <div id="notificationDropdown" class="notification-dropdown hidden">
            <div class="notification-header">
                <h3>Notifications</h3>
                <button id="clearNotifications" class="btn btn-small">Clear All</button>
            </div>
            <div id="notificationList" class="notification-list"></div>
        </div>

        <!-- Main Content -->
        <div class="main-container">
            <!-- Sidebar (Desktop) -->
            <aside class="sidebar">
                <nav class="sidebar-nav">
                    <?php if ($user_role === 'admin'): ?>
                        <!-- Admin navigation only for admin users -->
                        <div id="adminLinks" class="admin-section">
                            <a href="#" class="nav-link active" data-page="admin-dashboard">
                                <span class="icon">👑</span>
                                <span>Admin Dashboard</span>
                            </a>
                            <a href="#" class="nav-link" data-page="admin-tasks">
                                <span class="icon">➕</span>
                                <span>Manage Tasks</span>
                            </a>
                            <a href="#" class="nav-link" data-page="admin-users">
                                <span class="icon">👥</span>
                                <span>Manage Users</span>
                            </a>
                            <a href="#" class="nav-link" data-page="admin-reports">
                                <span class="icon">📋</span>
                                <span>Reports</span>
                            </a>
                        </div>
                    <?php else: ?>
                        <!-- User navigation only for normal users -->
                        <a href="#" class="nav-link active" data-page="dashboard">
                            <span class="icon">📊</span>
                            <span>Dashboard</span>
                        </a>
                        <a href="#" class="nav-link" data-page="tasks">
                            <span class="icon">✅</span>
                            <span>My Tasks</span>
                        </a>
                        <a href="#" class="nav-link" data-page="progress">
                            <span class="icon">📈</span>
                            <span>Progress</span>
                        </a>
                        <a href="#" class="nav-link" data-page="history">
                            <span class="icon">📜</span>
                            <span>History</span>
                        </a>
                    <?php endif; ?>
                </nav>
            </aside>

            <!-- Mobile Bottom Nav -->
            <nav class="bottom-nav">
                <?php if ($user_role === 'admin'): ?>
                    <!-- Admin bottom nav for admin users -->
                    <a href="#" class="bottom-nav-link active" data-page="admin-dashboard">
                        <span class="icon">👑</span>
                        <span>Dashboard</span>
                    </a>
                    <a href="#" class="bottom-nav-link" data-page="admin-tasks">
                        <span class="icon">➕</span>
                        <span>Tasks</span>
                    </a>
                    <a href="#" class="bottom-nav-link" data-page="admin-users">
                        <span class="icon">👥</span>
                        <span>Users</span>
                    </a>
                    <a href="#" class="bottom-nav-link" data-page="admin-reports">
                        <span class="icon">📋</span>
                        <span>Reports</span>
                    </a>
                <?php else: ?>
                    <!-- User bottom nav for normal users -->
                    <a href="#" class="bottom-nav-link active" data-page="dashboard">
                        <span class="icon">📊</span>
                        <span>Dashboard</span>
                    </a>
                    <a href="#" class="bottom-nav-link" data-page="tasks">
                        <span class="icon">✅</span>
                        <span>Tasks</span>
                    </a>
                    <a href="#" class="bottom-nav-link" data-page="progress">
                        <span class="icon">📈</span>
                        <span>Progress</span>
                    </a>
                    <a href="#" class="bottom-nav-link" data-page="history">
                        <span class="icon">📜</span>
                        <span>History</span>
                    </a>
                <?php endif; ?>
            </nav>

            <!-- Content Area -->
            <main class="content">
                <!-- Dynamic Content Container (SPA) -->
                <div id="pageContainer" class="page active">
                    <?php if ($is_logged_in): ?>
                        <!-- Load dashboard content directly if logged in -->
                        <div class="empty-state">
                            <div class="empty-state-icon">⏳</div>
                            <h3>Loading Dashboard...</h3>
                        </div>
                    <?php else: ?>
                        <!-- Pages will be loaded here dynamically -->
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>

    <!-- Add Task Modal -->
    <div id="addTaskModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="taskModalTitle">Add New Task</h2>
                <button class="modal-close" id="closeTaskModal">&times;</button>
            </div>
            <form id="addTaskForm" method="POST" action="add_task.php">
                <input type="hidden" name="task_id" id="hiddenTaskId" value="">
                <?php if (isset($_GET['task_error'])): ?>
                    <div class="alert alert-error" style="margin-bottom: 15px; padding: 10px; background: #fee; color: #c33; border-radius: 4px;">
                        <?php echo htmlspecialchars($_GET['task_error']); ?>
                    </div>
                <?php endif; ?>
                <div class="form-group">
                    <label for="taskName">Task Name *</label>
                    <input type="text" id="taskName" name="name" required>
                </div>
                <div class="form-group">
                    <label for="taskDescription">Description *</label>
                    <textarea id="taskDescription" name="description" rows="3" required></textarea>
                </div>
                <div class="form-group">
                    <label for="taskPriority">Priority *</label>
                    <select id="taskPriority" name="priority" required>
                        <option value="high">High</option>
                        <option value="medium">Medium</option>
                        <option value="low">Low</option>
                    </select>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="taskStartDate">Start Date *</label>
                        <input type="date" id="taskStartDate" name="start_date" required>
                    </div>
                    <div class="form-group">
                        <label for="taskEndDate">End Date (Optional)</label>
                        <input type="date" id="taskEndDate" name="end_date">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="taskStartTime">Start Time (Optional)</label>
                        <input type="time" id="taskStartTime" name="start_time" placeholder="e.g., 08:00">
                        <small class="form-hint">When the task should start (e.g., 8:00 AM)</small>
                    </div>
                    <div class="form-group">
                        <label for="taskEndTime">End Time (Optional)</label>
                        <input type="time" id="taskEndTime" name="end_time" placeholder="e.g., 18:00">
                        <small class="form-hint">When the task expires (e.g., 6:00 PM)</small>
                    </div>
                </div>
                <div class="form-group">
                    <label for="taskType">Task Type *</label>
                    <select id="taskType" name="task_type" required>
                        <option value="one-time">One-Time</option>
                        <option value="recurring">Recurring</option>
                    </select>
                </div>
                <div class="form-group" id="recurringDaysGroup" style="display: none;">
                    <label>Repeat Days *</label>
                    <div class="skip-days-container">
                        <label class="skip-day-checkbox">
                            <input type="checkbox" name="recurring_days[]" value="0" id="repeatSunday">
                            <span>Sunday</span>
                        </label>
                        <label class="skip-day-checkbox">
                            <input type="checkbox" name="recurring_days[]" value="1" id="repeatMonday">
                            <span>Monday</span>
                        </label>
                        <label class="skip-day-checkbox">
                            <input type="checkbox" name="recurring_days[]" value="2" id="repeatTuesday">
                            <span>Tuesday</span>
                        </label>
                        <label class="skip-day-checkbox">
                            <input type="checkbox" name="recurring_days[]" value="3" id="repeatWednesday">
                            <span>Wednesday</span>
                        </label>
                        <label class="skip-day-checkbox">
                            <input type="checkbox" name="recurring_days[]" value="4" id="repeatThursday">
                            <span>Thursday</span>
                        </label>
                        <label class="skip-day-checkbox">
                            <input type="checkbox" name="recurring_days[]" value="5" id="repeatFriday">
                            <span>Friday</span>
                        </label>
                        <label class="skip-day-checkbox">
                            <input type="checkbox" name="recurring_days[]" value="6" id="repeatSaturday">
                            <span>Saturday</span>
                        </label>
                    </div>
                    <small class="form-hint">Select the days of the week when this task should repeat (at least one day required)</small>
                </div>
                <div class="form-group">
                    <label for="taskAssignTo">Assign To *</label>
                    <select id="taskAssignTo" name="assigned_users[]" multiple required>
                        <?php
                        if ($is_logged_in && isAdmin()) {
                            try {
                                $conn = getDBConnection();
                                if ($conn) {
                                    $users_result = $conn->query("SELECT id, username, full_name FROM users WHERE role = 'user' ORDER BY username");
                                    if ($users_result) {
                                        while ($user = $users_result->fetch_assoc()): ?>
                                            <option value="<?php echo $user['id']; ?>">
                                                <?php echo escapeHtml($user['full_name'] ?: $user['username']); ?>
                                            </option>
                                        <?php endwhile;
                                    }
                                }
                            } catch (Exception $e) {
                                // Silently fail - dropdown will be empty
                            }
                        }
                        ?>
                    </select>
                    <small>Hold Ctrl/Cmd to select multiple users</small>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="taskPoints">Task Points *</label>
                        <input type="number" id="taskPoints" name="task_points" min="0" value="1" required>
                    </div>
                    <div class="form-group">
                        <label for="qualityPoints">Quality Points *</label>
                        <input type="number" id="qualityPoints" name="quality_points" min="0" value="10" required>
                    </div>
                    <div class="form-group">
                        <label for="lostTaskPoints">Lost Task Points</label>
                        <input type="number" id="lostTaskPoints" name="lost_task_points" value="-1">
                        <small class="form-hint">Negative value (e.g., -1)</small>
                    </div>
                </div>
                <div class="form-group">
                    <label for="taskNotes">Notes (Optional)</label>
                    <textarea id="taskNotes" name="notes" rows="2"></textarea>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn btn-secondary" id="cancelTaskBtn">Cancel</button>
                    <button type="submit" class="btn btn-primary" id="submitTaskBtn">Create Task</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Add User Modal -->
    <div id="addUserModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="addUserModalTitle">Add New User</h2>
                <button class="modal-close" id="closeUserModal">&times;</button>
            </div>
            <form id="addUserForm" method="POST" action="add_user.php">
                <input type="hidden" name="user_id" id="hiddenUserId">
                <?php if (isset($_GET['user_error'])): ?>
                    <div class="alert alert-error" style="margin-bottom: 15px; padding: 10px; background: #fee; color: #c33; border-radius: 4px;">
                        <?php echo htmlspecialchars($_GET['user_error']); ?>
                    </div>
                <?php endif; ?>
                <div class="form-group">
                    <label for="newUsername">Username *</label>
                    <input type="text" id="newUsername" name="username" required>
                </div>
                <div class="form-group">
                    <label for="newPassword" id="newPasswordLabel">Password *</label>
                    <input type="password" id="newPassword" name="password" required>
                    <small id="passwordHint" class="form-hint" style="display: none;">Leave blank when editing to keep current password</small>
                </div>
                <div class="form-group">
                    <label for="newUserRole">Role *</label>
                    <select id="newUserRole" name="role" required>
                        <option value="user">User</option>
                        <option value="admin">Admin</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="newUserEmail">Email (Optional)</label>
                    <input type="email" id="newUserEmail" name="email">
                </div>
                <div class="form-group">
                    <label for="newUserFullName">Full Name (Optional)</label>
                    <input type="text" id="newUserFullName" name="full_name">
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn btn-secondary" id="cancelUserBtn">Cancel</button>
                    <button type="submit" class="btn btn-primary" id="submitUserBtn">Create User</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Delete User Modal -->
    <div id="deleteUserModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Delete User</h2>
                <button class="modal-close" id="closeDeleteUserModal">&times;</button>
            </div>
            <p id="deleteUserMessage">Are you sure you want to delete this user? This action cannot be undone.</p>
            <form id="deleteUserForm" method="POST" action="delete_user.php">
                <input type="hidden" name="user_id" id="hiddenDeleteUserId">
                <div class="form-group">
                    <label for="adminPasswordDelete">Admin Password *</label>
                    <input type="password" id="adminPasswordDelete" name="admin_password" required>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn btn-secondary" id="cancelDeleteUserBtn">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete User</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Leave Management Modal -->
    <div id="leaveModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Manage Leave/Holiday</h2>
                <button class="modal-close" id="closeLeaveModal">&times;</button>
            </div>
            <div class="form-group">
                <label>User: <strong id="leaveUserName"></strong></label>
            </div>
            <form id="addLeaveForm">
                <input type="hidden" id="hiddenLeaveUserId" name="user_id">
                <div class="form-group">
                    <label for="leaveDate">Leave Date *</label>
                    <input type="date" id="leaveDate" required>
                    <small class="form-hint">Select a date when the user will be on leave. Tasks will be paused for this user on this date.</small>
                </div>
                <div class="form-group">
                    <button type="submit" class="btn btn-primary btn-small">Add Leave Date</button>
                </div>
            </form>
            <div class="form-group" style="margin-top: 20px;">
                <label><strong>Existing Leave Dates:</strong></label>
                <div id="leavesList" style="max-height: 200px; overflow-y: auto; margin-top: 10px;">
                    <p style="color: #999; padding: 10px;">Loading...</p>
                </div>
            </div>
            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" id="cancelLeaveBtn">Close</button>
            </div>
        </div>
    </div>

    <!-- Task Detail Modal (iOS Style) -->
    <div id="taskDetailModal" class="modal ios-modal">
        <div class="modal-content ios-modal-content">
            <div class="modal-header ios-modal-header">
                <h2 id="taskDetailTitle">Task Details</h2>
                <button class="modal-close" id="closeTaskDetailModal">&times;</button>
            </div>
            <div id="taskDetailContent" class="ios-modal-body">
                <div class="loading-spinner">
                    <div class="spinner"></div>
                    <p>Loading task details...</p>
                </div>
            </div>
            <div class="ios-modal-actions" id="taskDetailActions" style="display: none;">
                <button class="btn btn-secondary btn-small" id="taskDetailEditBtn">Edit</button>
                <button class="btn btn-danger btn-small" id="taskDetailDeleteBtn">Delete</button>
            </div>
        </div>
    </div>

    <!-- Delete Task Modal -->
    <div id="deleteTaskModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Delete Task</h2>
                <button class="modal-close" id="closeDeleteModal">&times;</button>
            </div>
            <p>Are you sure you want to delete this task? This action requires admin password.</p>
            <form id="deleteTaskForm" method="POST" action="delete_task.php">
                <input type="hidden" name="task_id" id="hiddenDeleteTaskId">
                <div class="form-group">
                    <label for="adminPassword">Admin Password *</label>
                    <input type="password" id="adminPassword" name="admin_password" required>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn btn-secondary" id="cancelDeleteBtn">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Task</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Disable Task Modal -->
    <div id="disableTaskModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Disable Task</h2>
                <button class="modal-close" id="closeDisableTaskModal">&times;</button>
            </div>
            <p id="disableTaskMessage">Are you sure you want to disable this task? Disabled tasks will be removed from all users' working tasks.</p>
            <form id="disableTaskForm">
                <input type="hidden" id="hiddenDisableTaskId" name="task_id">
                <div class="form-group">
                    <label for="disableReason">Reason for Disabling *</label>
                    <textarea id="disableReason" rows="3" placeholder="Please provide a reason for disabling this task..." required></textarea>
                    <small class="form-hint">This reason will be saved and displayed in the disabled tasks list.</small>
                </div>
                <div class="form-group">
                    <label for="adminPasswordDisable">Admin Password *</label>
                    <input type="password" id="adminPasswordDisable" required>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn btn-secondary" id="cancelDisableTaskBtn">Cancel</button>
                    <button type="submit" class="btn btn-warning">Disable Task</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Enable Task Modal -->
    <div id="enableTaskModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Enable Task</h2>
                <button class="modal-close" id="closeEnableTaskModal">&times;</button>
            </div>
            <p id="enableTaskMessage">Are you sure you want to enable this task? The task will be available to assigned users again.</p>
            <form id="enableTaskForm">
                <input type="hidden" id="hiddenEnableTaskId" name="task_id">
                <div class="form-group">
                    <label for="adminPasswordEnable">Admin Password *</label>
                    <input type="password" id="adminPasswordEnable" required>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn btn-secondary" id="cancelEnableTaskBtn">Cancel</button>
                    <button type="submit" class="btn btn-success">Enable Task</button>
                </div>
            </form>
        </div>
    </div>

    <!-- User Tasks Modal (iOS Style) -->
    <div id="userTasksModal" class="modal ios-modal">
        <div class="modal-content ios-modal-content">
            <div class="modal-header ios-modal-header">
                <h2 id="userTasksModalTitle">User Tasks</h2>
                <button class="modal-close" id="closeUserTasksModal">&times;</button>
            </div>
            <div class="ios-modal-toolbar">
                <div class="ios-date-picker-container">
                    <label for="taskDatePicker" class="ios-date-label">📅 Select Date:</label>
                    <input type="date" id="taskDatePicker" class="ios-date-input">
                    <button type="button" id="clearDateBtn" class="ios-clear-date-btn" title="Clear date filter">✕</button>
                </div>
            </div>
            <div class="ios-modal-body" id="userTasksModalBody">
                <div class="loading-spinner">
                    <div class="spinner"></div>
                    <p>Loading tasks...</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Mark Complete Modal -->
    <div id="markCompleteModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Mark Task as Complete</h2>
                <button class="modal-close" id="closeMarkCompleteModal">&times;</button>
            </div>
            <form id="markCompleteForm" method="POST" action="complete_task.php" enctype="multipart/form-data">
                <input type="hidden" name="task_id" id="hiddenCompleteTaskId" value="">
                <?php if (isset($_GET['complete_error'])): ?>
                    <div class="alert alert-error" style="margin-bottom: 15px; padding: 10px; background: #fee; color: #c33; border-radius: 4px;">
                        <?php echo htmlspecialchars($_GET['complete_error']); ?>
                    </div>
                <?php endif; ?>
                <div class="form-group">
                    <label for="completionUrl">URL (Optional)</label>
                    <input type="url" id="completionUrl" name="completion_url" placeholder="https://example.com">
                </div>
                <div class="form-group">
                    <label for="completionText">Text (Optional)</label>
                    <textarea id="completionText" name="completion_text" rows="4" placeholder="Enter any additional notes or text"></textarea>
                </div>
                <div class="form-group">
                    <label for="completionImage">Upload Image (Optional)</label>
                    <input type="file" id="completionImage" name="completion_image" accept="image/*">
                    <small>Accepted formats: JPG, PNG, GIF</small>
                    <div id="imagePreview" class="image-preview" style="display: none;">
                        <img id="previewImg" src="" alt="Preview">
                        <button type="button" id="removeImage" class="btn btn-small btn-secondary">Remove</button>
                    </div>
                </div>
                <div class="form-group">
                    <small class="form-hint">* At least one field (URL, Text, or Image) must be filled</small>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn btn-secondary" id="cancelMarkCompleteBtn">Cancel</button>
                    <button type="submit" class="btn btn-primary">Submit</button>
                </div>
            </form>
        </div>
    </div>

    <!-- iOS Style Confirmation Modal -->
    <div id="iosConfirmModal" class="modal ios-modal">
        <div class="modal-content ios-modal-content" style="max-width: 320px; padding: 0;">
            <div class="ios-confirm-header" style="padding: 20px; text-align: center; border-bottom: 0.5px solid var(--border);">
                <h3 id="iosConfirmTitle" style="margin: 0; font-size: 17px; font-weight: 600; color: var(--text-primary);">Confirm</h3>
            </div>
            <div class="ios-confirm-body" style="padding: 20px; text-align: center;">
                <p id="iosConfirmMessage" style="margin: 0; font-size: 15px; color: var(--text-secondary); line-height: 1.4;">Are you sure?</p>
            </div>
            <div class="ios-confirm-actions" style="display: flex; border-top: 0.5px solid var(--border);">
                <button id="iosConfirmCancel" class="ios-confirm-btn ios-confirm-cancel" style="flex: 1; padding: 16px; background: transparent; border: none; border-right: 0.5px solid var(--border); font-size: 17px; color: var(--accent); font-weight: 400; cursor: pointer;">Cancel</button>
                <button id="iosConfirmOk" class="ios-confirm-btn ios-confirm-ok" style="flex: 1; padding: 16px; background: transparent; border: none; font-size: 17px; color: var(--danger); font-weight: 600; cursor: pointer;">OK</button>
            </div>
        </div>
    </div>

    <script>
        // Set timezone to Asia/Kolkata for all JavaScript date operations
        // Store timezone offset for Kolkata (IST = UTC+5:30)
        const KOLKATA_TIMEZONE = 'Asia/Kolkata';
        const KOLKATA_OFFSET = 5.5 * 60 * 60 * 1000; // 5 hours 30 minutes in milliseconds
        
        // Helper function to get current time in Kolkata timezone (IST = UTC+5:30)
        function getKolkataTime() {
            // Get current UTC time
            const now = new Date();
            
            // Convert to IST using toLocaleString with Asia/Kolkata timezone
            // This ensures we always get IST time regardless of browser timezone
            const istString = now.toLocaleString('en-US', {
                timeZone: 'Asia/Kolkata',
                year: 'numeric',
                month: '2-digit',
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit',
                hour12: false
            });
            
            // Parse the IST string (format: "MM/DD/YYYY, HH:MM:SS")
            const parts = istString.match(/(\d+)\/(\d+)\/(\d+),\s+(\d+):(\d+):(\d+)/);
            if (parts) {
                const month = parts[1];
                const day = parts[2];
                const year = parts[3];
                const hour = parts[4];
                const minute = parts[5];
                const second = parts[6];
                
                // Create ISO string in IST timezone
                const isoString = `${year}-${month.padStart(2, '0')}-${day.padStart(2, '0')}T${hour.padStart(2, '0')}:${minute.padStart(2, '0')}:${second.padStart(2, '0')}+05:30`;
                return new Date(isoString);
            }
            
            // Fallback: just return current time
            return now;
        }
        
        // Helper function to create date in Kolkata timezone
        // Treats input date/time as Kolkata time (IST)
        // Uses timezone string +05:30 - JavaScript handles conversion automatically
        function createKolkataDate(dateString, timeString = null) {
            // Default to end of day if no time provided
            if (!timeString) {
                timeString = '23:59:59';
            }
            
            // Ensure time is in HH:MM:SS format
            const timeParts = timeString.split(':');
            if (timeParts.length === 2) {
                timeString = timeString + ':00';
            }
            
            // Create ISO string with timezone: YYYY-MM-DDTHH:MM:SS+05:30
            // JavaScript will automatically convert this to UTC internally
            const isoString = `${dateString}T${timeString}+05:30`;
            
            // Create Date object - JavaScript handles timezone conversion
            return new Date(isoString);
        }
        
        // Pass PHP variables to JavaScript
        const CURRENT_USER = {
            id: <?php echo getCurrentUserId() ?? 'null'; ?>,
            username: '<?php echo htmlspecialchars($username, ENT_QUOTES); ?>',
            role: '<?php echo htmlspecialchars($user_role, ENT_QUOTES); ?>'
        };
        const IS_LOGGED_IN = <?php echo $is_logged_in ? 'true' : 'false'; ?>;
        
        // Debug
        console.log('Page loaded. IS_LOGGED_IN:', IS_LOGGED_IN);
        console.log('CURRENT_USER:', CURRENT_USER);
        console.log('Session check:', {
            'session_status': '<?php echo session_status(); ?>',
            'user_id': '<?php echo $_SESSION['user_id'] ?? 'not set'; ?>',
            'username': '<?php echo $_SESSION['username'] ?? 'not set'; ?>'
        });
        
        // If temp user exists in sessionStorage but IS_LOGGED_IN is false, session might not have persisted
        if (!IS_LOGGED_IN && sessionStorage.getItem('temp_user')) {
            console.warn('Session may not have persisted. Temp user found:', sessionStorage.getItem('temp_user'));
            // Try to reload after a moment to allow session to be set
            setTimeout(() => {
                if (!IS_LOGGED_IN) {
                    console.log('Reloading to check session...');
                    window.location.reload();
                }
            }, 500);
        }
    </script>
    <script src="app-minimal.js"></script>
</body>
</html>
