<?php
// Include config to set timezone and get DB connection
require_once 'config.php';

// Use the configured DB connection (which sets MySQL timezone)
$conn = getDBConnection();

if (!$conn) {
    die("Database connection failed");
}

// Get MySQL time and timezone (after setting session timezone)
// Note: NOW() will return time in the session timezone (which should be +05:30)
$sql = "
    SELECT 
        NOW() AS mysql_time,
        @@global.time_zone AS global_tz,
        @@session.time_zone AS session_tz,
        UTC_TIMESTAMP() AS mysql_utc_time
";
$result = $conn->query($sql);
$row = $result->fetch_assoc();

// PHP server time & timezone (should be Asia/Kolkata after config.php)
$php_timezone = date_default_timezone_get();
$php_time = date('Y-m-d H:i:s');

// Also get time using DateTime with explicit timezone
$dt = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
$php_time_kolkata = $dt->format('Y-m-d H:i:s');

$conn->close();
?>

<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Time & Timezone Test</title>
    <style>
        body { font-family: Arial, sans-serif; padding: 20px; }
        h2 { color: #333; }
        .box { padding: 15px; border: 1px solid #ccc; margin-bottom: 20px; }
    </style>
</head>
<body>

<h1>Time & Timezone Test</h1>

<div class="box">
    <h2>MySQL Server</h2>
    <p><strong>MySQL Time (NOW() - in session timezone):</strong> <?php echo $row['mysql_time']; ?> (should be Kolkata time)</p>
    <p><strong>MySQL UTC Time:</strong> <?php echo $row['mysql_utc_time']; ?></p>
    <p><strong>Global Timezone:</strong> <?php echo $row['global_tz']; ?></p>
    <p><strong>Session Timezone:</strong> <?php echo $row['session_tz']; ?> (should be +05:30)</p>
    <p><strong>Status:</strong> 
        <?php 
        if ($row['session_tz'] === '+05:30' || $row['session_tz'] === '+05:30:00') {
            echo '<span style="color: green;">✓ Correct</span>';
        } else {
            echo '<span style="color: red;">✗ Incorrect - should be +05:30</span>';
        }
        ?>
    </p>
</div>

<div class="box">
    <h2>PHP Server</h2>
    <p><strong>PHP Timezone:</strong> <?php echo $php_timezone; ?> 
        <?php 
        if ($php_timezone === 'Asia/Kolkata' || $php_timezone === 'Asia/Calcutta') {
            echo '<span style="color: green;">✓ Correct</span>';
        } else {
            echo '<span style="color: red;">✗ Incorrect - should be Asia/Kolkata</span>';
        }
        ?>
    </p>
    <p><strong>PHP Current Time (date()):</strong> <?php echo $php_time; ?></p>
    <p><strong>PHP Current Time (DateTime Asia/Kolkata):</strong> <?php echo $php_time_kolkata; ?></p>
    <p><strong>Note:</strong> Both times should match and be in Kolkata timezone (UTC+5:30)</p>
</div>

<div class="box">
    <h2>Client (JavaScript / Browser)</h2>
    <p><strong>Browser Timezone:</strong> <span id="js-tz"></span></p>
    <p><strong>Browser Time:</strong> <span id="js-time"></span></p>
</div>

<script>
    function updateTime() {
        const now = new Date();
        document.getElementById('js-time').innerText = now.toLocaleString();
        document.getElementById('js-tz').innerText =
            Intl.DateTimeFormat().resolvedOptions().timeZone;
    }

    updateTime();
    setInterval(updateTime, 1000);
</script>

</body>
</html>
