<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1); // Temporarily display errors to debug
ini_set('log_errors', 1);

require_once '../config.php';

if (!isLoggedIn()) {
    header('Location: ../index.php');
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    die("Database connection failed. Please check your configuration.");
}

$user_id = getCurrentUserId();
$username = getCurrentUsername();
$full_name = $_SESSION['full_name'] ?? $username;
$user_role = getCurrentUserRole();

// Get notification count
$notification_count = 0;
try {
    $notif_count_stmt = $conn->prepare("SELECT COUNT(*) as count FROM notifications WHERE user_id = ? AND is_read = 0");
    if ($notif_count_stmt) {
        $notif_count_stmt->bind_param("i", $user_id);
        $notif_count_stmt->execute();
        $notif_count_result = $notif_count_stmt->get_result();
        if ($notif_count_result) {
            $notification_count = $notif_count_result->fetch_assoc()['count'] ?? 0;
        }
        $notif_count_stmt->close();
    }
} catch (Exception $e) {
    $notification_count = 0;
}

// Get date range for tasks (keep existing behavior)
$date_from = date('Y-m-d');
$date_to = date('Y-m-d', strtotime('+7 days'));

// Points date range (default to this week, allow custom selection)
try {
    $week_ref = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
    $week_start = (clone $week_ref)->modify('monday this week');
    $week_end = (clone $week_ref)->modify('sunday this week');
} catch (Exception $e) {
    $week_start = new DateTime('monday this week');
    $week_end = new DateTime('sunday this week');
}
$points_date_from = $_GET['date_from'] ?? $week_start->format('Y-m-d');
$points_date_to = $_GET['date_to'] ?? $week_end->format('Y-m-d');

// Get user points for the selected range
$points_stmt = $conn->prepare("
    SELECT
        COALESCE(SUM(tc.task_points_earned), 0) as task_points,
        COALESCE(SUM(tc.quality_points_earned), 0) as quality_points,
        COALESCE(SUM(tc.lost_task_points), 0) as lost_task_points,
        COALESCE(SUM(CASE WHEN tc.status = 'failed' THEN t.quality_points ELSE 0 END), 0) as lost_quality_points
    FROM task_completions tc
    LEFT JOIN tasks t ON tc.task_id = t.id
    WHERE tc.user_id = ?
      AND tc.completion_date BETWEEN ? AND ?
");
$points_stmt->bind_param("iss", $user_id, $points_date_from, $points_date_to);
$points_stmt->execute();
$points_result = $points_stmt->get_result();
$points = $points_result->fetch_assoc() ?: [
    'task_points' => 0,
    'quality_points' => 0,
    'lost_task_points' => 0,
    'lost_quality_points' => 0
];

// Today's stats (for today's date) - improved to handle failed tasks
// Get current time in Kolkata timezone for accurate failed task detection
try {
    $now_kolkata = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
    $current_date_kolkata = $now_kolkata->format('Y-m-d');
    $current_time_kolkata = $now_kolkata->format('H:i:s');
} catch (Exception $e) {
    $current_date_kolkata = date('Y-m-d');
    $current_time_kolkata = date('H:i:s');
}
// Use Kolkata date for all "today" calculations
$today = $current_date_kolkata;

// Get today's tasks - simplified query without leave days check first
try {
    // Get today's day of week (0 = Sunday, 1 = Monday, ..., 6 = Saturday)
    $today_date_obj = new DateTime($current_date_kolkata, new DateTimeZone('Asia/Kolkata'));
    $today_day_of_week = (int)$today_date_obj->format('w');
    
    $today_stmt = $conn->prepare("
        SELECT 
               t.id,
               t.task_type,
               t.skip_days,
               t.start_date,
               t.end_date,
               t.end_time,
               rti.instance_date as effective_date,
               rti.end_time as effective_end_time,
               rti.status as instance_status,
               CASE WHEN tc.id IS NOT NULL THEN tc.status ELSE 'pending' END as status
        FROM task_assignments ta
        JOIN tasks t ON ta.task_id = t.id
        LEFT JOIN recurring_task_instances rti ON t.id = rti.parent_task_id 
            AND rti.instance_date = ?
        LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
            AND ta.user_id = tc.user_id 
            AND tc.completion_date = ?
        WHERE ta.user_id = ? 
          AND t.start_date <= ? 
          AND (t.end_date IS NULL OR t.end_date >= ?) 
          AND t.is_enabled = 1
          AND t.task_type IN ('one-time', 'recurring')
    ");
    
    if (!$today_stmt) {
        throw new Exception("Failed to prepare query: " . $conn->error);
    }
    
    $bind_result = $today_stmt->bind_param("ssiss", $today, $today, $user_id, $today, $today);
    if (!$bind_result) {
        throw new Exception("Failed to bind parameters: " . $today_stmt->error);
    }
    
    $execute_result = $today_stmt->execute();
    if (!$execute_result) {
        throw new Exception("Failed to execute query: " . $today_stmt->error);
    }
    
    $today_result = $today_stmt->get_result();
    $total = 0;
    $completed = 0;
    $failed = 0;
    
    // Check for leave days separately
    $leave_check_stmt = $conn->prepare("SELECT COUNT(*) as count FROM user_leaves WHERE user_id = ? AND leave_date = ?");
    $leave_check_stmt->bind_param("is", $user_id, $today);
    $leave_check_stmt->execute();
    $leave_result = $leave_check_stmt->get_result();
    $leave_data = $leave_result->fetch_assoc();
    $on_leave = ($leave_data['count'] ?? 0) > 0;
    $leave_check_stmt->close();
    
    while ($row = $today_result->fetch_assoc()) {
        // Skip if user is on leave
        if ($on_leave) {
            continue;
        }
        
        // For recurring tasks, check if today is a recurring day
        if ($row['task_type'] === 'recurring') {
            $recurring_days = $row['skip_days'] ? explode(',', $row['skip_days']) : [];
            // Check if today's day of week is in the recurring days list
            if (empty($recurring_days) || !in_array((string)$today_day_of_week, $recurring_days)) {
                // Today is not a recurring day, skip this task
                continue;
            }
        }
        
        // For one-time tasks, only count from start date onward
        if ($row['task_type'] === 'one-time' && !empty($row['start_date'])) {
            if ($row['start_date'] > $current_date_kolkata) {
                continue;
            }
        }
        
        // Align status logic with API (use instance date/time for recurring)
        $computed_status = $row['status'];
        if ($computed_status === 'pending') {
            if (!empty($row['instance_status'])) {
                $computed_status = $row['instance_status'];
            } else {
                $effective_end_date = $row['effective_date'] ?? ($row['end_date'] ?? $row['start_date']);
                $effective_end_time = $row['effective_end_time'] ?? ($row['end_time'] ?? null);
                if ($effective_end_date < $current_date_kolkata ||
                    ($effective_end_date === $current_date_kolkata && $effective_end_time && $effective_end_time < $current_time_kolkata)) {
                    $computed_status = 'failed';
                }
            }
        }
        
        $total++;
        if ($computed_status === 'completed') {
            $completed++;
        } elseif ($computed_status === 'failed') {
            $failed++;
        }
    }
    $today_stmt->close();
} catch (Exception $e) {
    error_log("Error in today stats query: " . $e->getMessage());
    // Fallback to simple counts
    $total = 0;
    $completed = 0;
    $failed = 0;
}

// Calculate pending as total - completed - failed
$today_stats = [
    'total' => $total,
    'completed' => $completed,
    'failed' => $failed,
    'pending' => max(0, $total - $completed - $failed)
];

$progress_percentage = $today_stats['total'] > 0 
    ? round(($today_stats['completed'] / $today_stats['total']) * 100, 1)
    : 0;

// Get tasks within date range (showing all assigned tasks, not just today)
// For recurring tasks, join with instances table to get today's instance or next occurrence
// Exclude completed one-time tasks, but show next occurrence for completed recurring tasks
$tasks_query = "
    SELECT DISTINCT 
           t.*,
           rti.instance_date as effective_date,
           COALESCE(rti.start_time, t.start_time) as effective_start_time,
           COALESCE(rti.end_time, t.end_time) as effective_end_time,
           COALESCE(rti.status, 
               CASE WHEN tc.id IS NOT NULL AND tc.status = 'completed' THEN 'completed'
                    WHEN tc.id IS NOT NULL AND tc.status = 'failed' THEN 'failed'
                    WHEN (COALESCE(rti.instance_date, t.end_date) IS NOT NULL AND COALESCE(rti.instance_date, t.end_date) < ?) 
                         OR (COALESCE(rti.instance_date, t.end_date) = ? AND COALESCE(rti.end_time, t.end_time) IS NOT NULL AND COALESCE(rti.end_time, t.end_time) < ?)
                         THEN 'failed'
                    ELSE 'pending' END
           ) as status
    FROM task_assignments ta
    JOIN tasks t ON ta.task_id = t.id
    LEFT JOIN recurring_task_instances rti ON t.id = rti.parent_task_id 
        AND (
            (rti.instance_date = ? AND t.task_type = 'recurring')
            OR (rti.instance_date > ? AND t.task_type = 'recurring' AND rti.instance_date <= ?)
        )
    LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
        AND ta.user_id = tc.user_id 
        AND (t.task_type = 'one-time' OR (t.task_type = 'recurring' AND tc.completion_date = COALESCE(rti.instance_date, ?)))
        AND tc.completion_date = (
            SELECT MAX(completion_date) 
            FROM task_completions 
            WHERE task_id = t.id AND user_id = ta.user_id
                AND (t.task_type = 'one-time' OR completion_date = COALESCE(rti.instance_date, ?))
        )
    WHERE ta.user_id = ? 
      AND t.is_enabled = 1
      AND (
          (t.task_type = 'one-time' 
           AND t.start_date <= ? 
           AND (t.end_date IS NULL OR t.end_date >= ?)
           AND NOT EXISTS (
               SELECT 1 FROM task_completions tc2 
               WHERE tc2.task_id = t.id 
               AND tc2.user_id = ta.user_id 
               AND tc2.status = 'completed'
           ))
          OR (t.task_type = 'recurring' AND (
              (rti.id IS NOT NULL AND rti.instance_date = ? AND COALESCE(rti.status, 'pending') != 'completed')
              OR (rti.id IS NOT NULL AND rti.instance_date > ?)
              OR (rti.id IS NULL AND NOT EXISTS (
                  SELECT 1 FROM recurring_task_instances rti2 
                  WHERE rti2.parent_task_id = t.id 
                  AND rti2.instance_date = ?
              ))
          ))
      )
    ORDER BY 
        CASE WHEN (COALESCE(rti.instance_date, t.end_date) IS NOT NULL AND COALESCE(rti.instance_date, t.end_date) < ? AND COALESCE(rti.status, 'pending') != 'completed') 
                  OR (COALESCE(rti.instance_date, t.end_date) = ? AND COALESCE(rti.end_time, t.end_time) IS NOT NULL AND COALESCE(rti.end_time, t.end_time) < ? AND COALESCE(rti.status, 'pending') != 'completed')
             THEN 0 ELSE 1 END,
        t.priority DESC, 
        COALESCE(rti.instance_date, t.start_date) ASC
    LIMIT 50
";
$tasks_stmt = $conn->prepare($tasks_query);
if (!$tasks_stmt) {
    http_response_code(500);
    header('Content-Type: text/html');
    echo "<!DOCTYPE html><html><head><title>Error</title></head><body>";
    echo "<h1>Query Preparation Failed</h1>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($conn->error) . "</p>";
    echo "<p><strong>Query:</strong> <pre>" . htmlspecialchars($tasks_query) . "</pre></p>";
    exit;
}

// Count placeholders manually: 
// Line 93: ?, Line 94: ?, ?, Line 102: ?, Line 103: ?, ?, Line 107: ?, Line 112: ?, Line 114: ?, Line 118: ?, Line 119: ?, Line 127: ?, Line 128: ?, Line 132: ?, Line 137: ?, Line 138: ?, ?
// Total: 17 placeholders

// Parameters in order:
$bind_params = [
    $current_date_kolkata,  // 1: line 93
    $current_date_kolkata,  // 2: line 94
    $current_time_kolkata,  // 3: line 94
    $current_date_kolkata,  // 4: line 102
    $current_date_kolkata,  // 5: line 103
    $date_to,               // 6: line 103
    $current_date_kolkata,  // 7: line 107
    $current_date_kolkata,  // 8: line 112
    $user_id,               // 9: line 114 (integer)
    $date_to,               // 10: line 118
    $date_from,             // 11: line 119
    $current_date_kolkata,  // 12: line 127
    $current_date_kolkata,  // 13: line 128
    $current_date_kolkata,  // 14: line 132
    $current_date_kolkata,  // 15: line 137
    $current_date_kolkata,  // 16: line 138
    $current_time_kolkata   // 17: line 138
];

$types = "sssssssssisssssss"; // 17 types: 16 strings + 1 integer

if (!$tasks_stmt->bind_param($types, ...$bind_params)) {
    http_response_code(500);
    header('Content-Type: text/html');
    echo "<!DOCTYPE html><html><head><title>Error</title></head><body>";
    echo "<h1>Bind Parameter Failed</h1>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($tasks_stmt->error) . "</p>";
    echo "<p><strong>Expected:</strong> 17 parameters</p>";
    echo "<p><strong>Types:</strong> " . htmlspecialchars($types) . "</p>";
    exit;
}
if (!$tasks_stmt->execute()) {
    error_log("Query execution failed: " . $tasks_stmt->error);
    error_log("Query: " . $tasks_query);
    die("Error executing query. Please check the logs.");
}
$tasks_result = $tasks_stmt->get_result();

$tasks = [];
$pending_tasks = [];
// Get user's leave dates
$leave_dates = getUserLeaveDates($user_id, $date_from, $date_to);
$is_on_leave_today = in_array($current_date_kolkata, $leave_dates, true);

// Helper function to get next occurrence date for recurring tasks
function getNextRecurringOccurrence($conn, $task_id, $current_date) {
    $task_stmt = $conn->prepare("SELECT start_time, end_time, skip_days FROM tasks WHERE id = ? AND task_type = 'recurring'");
    $task_stmt->bind_param("i", $task_id);
    $task_stmt->execute();
    $task_result = $task_stmt->get_result();
    $task = $task_result->fetch_assoc();
    
    if (!$task) return null;
    
    // Get recurring days (stored in skip_days field, repurposed)
    $recurring_days = $task['skip_days'] ? explode(',', $task['skip_days']) : [];
    if (empty($recurring_days)) {
        return null; // No recurring days defined
    }
    
    // Find next occurrence date that matches one of the recurring days
    $instance_date = new DateTime($current_date, new DateTimeZone('Asia/Kolkata'));
    $next_date = clone $instance_date;
    $max_date = date('Y-m-d', strtotime('+30 days'));
    $max_attempts = 14; // Check up to 2 weeks ahead
    $found_next = false;
    
    for ($i = 0; $i < $max_attempts; $i++) {
        $next_date->modify("+1 day");
        $next_date_str = $next_date->format('Y-m-d');
        $next_day_of_week = (int)$next_date->format('w');
        
        // Check if this day is in the recurring days list
        if (in_array((string)$next_day_of_week, $recurring_days) && $next_date_str <= $max_date) {
            $found_next = true;
            break;
        }
    }
    
    if (!$found_next) {
        return null; // Couldn't find next occurrence within 2 weeks
    }
    
    // Check if instance already exists
    $check_stmt = $conn->prepare("SELECT id, instance_date, start_time, end_time, status FROM recurring_task_instances WHERE parent_task_id = ? AND instance_date = ?");
    $check_stmt->bind_param("is", $task_id, $next_date_str);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if ($check_result->num_rows > 0) {
        return $check_result->fetch_assoc();
    } else if ($next_date_str <= $max_date) {
        // Create the instance
        $insert_stmt = $conn->prepare("
            INSERT INTO recurring_task_instances 
            (parent_task_id, instance_date, start_time, end_time, status)
            VALUES (?, ?, ?, ?, 'pending')
        ");
        $insert_stmt->bind_param("isss", 
            $task_id,
            $next_date_str,
            $task['start_time'],
            $task['end_time']
        );
        
        if ($insert_stmt->execute()) {
            return [
                'id' => $conn->insert_id,
                'instance_date' => $next_date_str,
                'start_time' => $task['start_time'],
                'end_time' => $task['end_time'],
                'status' => 'pending'
            ];
        }
    }
    
    return null;
}

// Get today's day of week for filtering recurring tasks (0 = Sunday, 1 = Monday, ..., 6 = Saturday)
$today_date_obj_for_filter = new DateTime($current_date_kolkata, new DateTimeZone('Asia/Kolkata'));
$today_day_of_week_for_filter = (int)$today_date_obj_for_filter->format('w');

while ($row = $tasks_result->fetch_assoc()) {
    // For recurring tasks, check if today is a recurring day
    if ($row['task_type'] === 'recurring') {
        $recurring_days = $row['skip_days'] ? explode(',', $row['skip_days']) : [];
        // If there's an instance for today, check if today is a recurring day
        if (!empty($row['effective_date']) && $row['effective_date'] === $current_date_kolkata) {
            // Check if today is a recurring day
            if (empty($recurring_days) || !in_array((string)$today_day_of_week_for_filter, $recurring_days)) {
                // Today is not a recurring day, skip this task
                continue;
            }
        } else {
            // No instance for today, check if today is a recurring day
            if (empty($recurring_days) || !in_array((string)$today_day_of_week_for_filter, $recurring_days)) {
                // Today is not a recurring day, skip this task
                continue;
            }
        }
    }
    
    // If recurring task has no instance for today but today is a valid recurring day,
    // treat it as today's task for filtering.
    if ($row['task_type'] === 'recurring') {
        $recurring_days = $row['skip_days'] ? explode(',', $row['skip_days']) : [];
        if ((empty($row['effective_date']) || $row['effective_date'] > $current_date_kolkata)
            && !empty($recurring_days)
            && in_array((string)$today_day_of_week_for_filter, $recurring_days)) {
            $row['effective_date'] = $current_date_kolkata;
            $row['effective_start_time'] = $row['start_time'];
            $row['effective_end_time'] = $row['end_time'];
        }
    }
    
    // For one-time tasks, only show in pending list from start date onward
    if ($row['task_type'] === 'one-time') {
        if (!empty($row['start_date']) && $row['start_date'] > $current_date_kolkata) {
            continue;
        }
    }
    
    // For recurring tasks that are completed today, get next occurrence
    if ($row['task_type'] === 'recurring' && $row['status'] === 'completed' && 
        !empty($row['effective_date']) && $row['effective_date'] === $current_date_kolkata) {
        $next_occurrence = getNextRecurringOccurrence($conn, $row['id'], $row['effective_date']);
        if ($next_occurrence) {
            $row['effective_date'] = $next_occurrence['instance_date'];
            $row['effective_start_time'] = $next_occurrence['start_time'];
            $row['effective_end_time'] = $next_occurrence['end_time'];
            $row['status'] = 'pending';
        } else {
            // No next occurrence found, skip this task
            continue;
        }
    }
    
    // Check if task falls on a leave date
    $task_start = $row['effective_date'] ?? $row['start_date'];
    $task_end = $row['effective_date'] ?? $row['end_date'] ?? $task_start;
    
    // Check if any date in the task range is a leave date (for date-range list)
    $is_on_leave = false;
    foreach ($leave_dates as $leave_date) {
        if ($leave_date >= $task_start && $leave_date <= $task_end) {
            $is_on_leave = true;
            break;
        }
    }
    
    // Only add task if:
    // - user is not on leave during task period
    // - task is pending
    // - task date is within the selected date range (date_from to date_to)
    $task_date_for_filter = $row['effective_date'] ?? $row['start_date'];
    if (
        !$is_on_leave &&
        $task_date_for_filter >= $date_from &&
        $task_date_for_filter <= $date_to
    ) {
        $tasks[] = $row;
    }
}

// Build pending list for today using dedicated query (matches popup behavior)
$pending_tasks_today = [];
if (!$is_on_leave_today) {
    $pending_stmt = $conn->prepare("
        SELECT DISTINCT t.*,
               rti.instance_date as effective_date,
               rti.start_time as effective_start_time,
               rti.end_time as effective_end_time,
               rti.status as instance_status
        FROM tasks t
        JOIN task_assignments ta ON t.id = ta.task_id
        LEFT JOIN recurring_task_instances rti ON t.id = rti.parent_task_id 
            AND rti.instance_date = ?
        WHERE ta.user_id = ?
          AND t.is_enabled = 1
          AND t.start_date <= ?
          AND (t.end_date IS NULL OR t.end_date >= ?)
    ");
    if ($pending_stmt) {
        $pending_stmt->bind_param("siss", $current_date_kolkata, $user_id, $current_date_kolkata, $current_date_kolkata);
        $pending_stmt->execute();
        $pending_result = $pending_stmt->get_result();
        
        $completion_today_stmt = $conn->prepare("
            SELECT status
            FROM task_completions
            WHERE task_id = ? AND user_id = ? AND completion_date = ?
            ORDER BY completed_at DESC
            LIMIT 1
        ");
        
        while ($row = $pending_result->fetch_assoc()) {
            // Recurring day check
            if ($row['task_type'] === 'recurring') {
                $recurring_days = $row['skip_days'] ? explode(',', $row['skip_days']) : [];
                if (empty($recurring_days) || !in_array((string)$today_day_of_week_for_filter, $recurring_days)) {
                    continue;
                }
            }
            
            // Determine today's status
            $status_today = 'pending';
            if (!empty($row['instance_status'])) {
                $status_today = $row['instance_status'];
            } elseif ($completion_today_stmt) {
                $completion_today_stmt->bind_param("iis", $row['id'], $user_id, $current_date_kolkata);
                $completion_today_stmt->execute();
                $completion_today_result = $completion_today_stmt->get_result();
                $completion_today = $completion_today_result->fetch_assoc();
                if ($completion_today) {
                    $status_today = $completion_today['status'];
                }
            }
            
            // Check failed for today
            $effective_end_date = $row['effective_date'] ?? ($row['end_date'] ?? $row['start_date']);
            $effective_end_time = $row['effective_end_time'] ?? ($row['end_time'] ?? null);
            if ($status_today === 'pending') {
                if ($effective_end_date < $current_date_kolkata || 
                    ($effective_end_date === $current_date_kolkata && $effective_end_time && $effective_end_time < $current_time_kolkata)) {
                    $status_today = 'failed';
                }
            }
            
            if ($status_today === 'pending') {
                $row['status'] = $status_today;
                $pending_tasks_today[] = $row;
            }
        }
        
        if ($completion_today_stmt) {
            $completion_today_stmt->close();
        }
        $pending_stmt->close();
    }
}

$pending_tasks = $pending_tasks_today;

function formatDate($dateString) {
    return date('M d, Y', strtotime($dateString));
}

function escapeHtml($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}

$nav_titles = [
    'user-dashboard.php' => 'Dashboard',
    'user-tasks.php' => 'My Tasks',
    'user-progress.php' => 'Progress',
    'user-history.php' => 'History'
];
$current_page = basename($_SERVER['PHP_SELF'] ?? '');
$nav_title = $nav_titles[$current_page] ?? 'Task Manager';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Task Management System</title>
    <link rel="stylesheet" href="../styles.css">
</head>
<body class="dashboard-page">
    <!-- Navbar -->
    <nav class="navbar">
        <div class="nav-left">
            <h2 class="nav-title"><?php echo htmlspecialchars($nav_title, ENT_QUOTES, 'UTF-8'); ?></h2>
        </div>
        <div class="nav-right">
            <button id="notificationBtn" class="nav-icon-btn" title="Notifications" style="display: <?php echo $notification_count > 0 ? 'flex' : 'none'; ?>;">
                <span class="icon">🔔</span>
                <?php if ($notification_count > 0): ?>
                <span id="notificationBadge" class="badge"><?php echo $notification_count; ?></span>
                <?php endif; ?>
            </button>
            <button id="themeToggle" class="nav-icon-btn" title="Toggle Theme">
                <span class="icon" id="themeIcon">🌙</span>
            </button>
            <div class="user-menu">
                <span id="userNameDisplay"><?php echo htmlspecialchars($username); ?></span>
                <a href="../logout.php" class="btn btn-small">Logout</a>
            </div>
        </div>
    </nav>

    <!-- Notification Dropdown -->
    <div id="notificationDropdown" class="notification-dropdown hidden">
        <div class="notification-header">
            <h3>Notifications</h3>
            <button id="clearNotifications" class="btn btn-small">Clear All</button>
        </div>
        <div id="notificationList" class="notification-list"></div>
    </div>

    <!-- Main Content -->
    <div class="main-container">
        <!-- Sidebar (Desktop) -->
        <aside class="sidebar">
            <nav class="sidebar-nav">
                <a href="user-dashboard.php" class="nav-link active">
                    <img class="nav-icon" src="../icon/dashboard_icon.png" alt="Dashboard">
                    <span>Dashboard</span>
                </a>
                <a href="user-tasks.php" class="nav-link">
                    <img class="nav-icon" src="../icon/mytask.png" alt="My Tasks">
                    <span>My Tasks</span>
                </a>
                <a href="user-progress.php" class="nav-link">
                    <img class="nav-icon" src="../icon/progress.png" alt="Progress">
                    <span>Progress</span>
                </a>
                <a href="user-history.php" class="nav-link">
                    <img class="nav-icon" src="../icon/history-icon.png" alt="History">
                    <span>History</span>
                </a>
            </nav>
        </aside>

        <!-- Mobile Bottom Nav -->
        <nav class="bottom-nav">
            <a href="user-dashboard.php" class="bottom-nav-link active">
                <img class="nav-icon" src="../icon/dashboard_icon.png" alt="Dashboard">
                <span>Dashboard</span>
            </a>
            <a href="user-tasks.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/mytask.png" alt="Tasks">
                <span>Tasks</span>
            </a>
            <a href="user-progress.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/progress.png" alt="Progress">
                <span>Progress</span>
            </a>
            <a href="user-history.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/history-icon.png" alt="History">
                <span>History</span>
            </a>
        </nav>

        <!-- Content Area -->
        <main class="content">
            <div class="page active">
                <div class="page-header">
                    <h1>Dashboard</h1>
                    <div class="page-header-bottom">
                        <p class="page-subtitle">Welcome back, <?php echo escapeHtml($full_name); ?>!</p>
                        <form method="GET" action="" class="date-range-inline">
                            <input type="date" name="date_from" value="<?php echo $points_date_from; ?>" class="date-input-inline">
                            <span class="date-separator">to</span>
                            <input type="date" name="date_to" value="<?php echo $points_date_to; ?>" class="date-input-inline">
                            <button type="submit" class="btn btn-small">Filter</button>
                        </form>
                    </div>
                </div>

                <!-- Points Metrics Grid -->
                <div class="points-metrics-grid">
                    <div class="points-metric-card task-points">
                        <div class="metric-info">
                            <h3><?php echo $points['task_points']; ?></h3>
                            <p>Task Points</p>
                        </div>
                    </div>
                    <div class="points-metric-card lost-task-points">
                        <div class="metric-info">
                            <h3 class="negative-value"><?php echo $points['lost_task_points']; ?></h3>
                            <p>Lost Task Points</p>
                        </div>
                    </div>
                    <div class="points-metric-card quality-points">
                        <div class="metric-info">
                            <h3><?php echo $points['quality_points']; ?></h3>
                            <p>Quality Points</p>
                        </div>
                    </div>
                    <div class="points-metric-card lost-quality-points">
                        <div class="metric-info">
                            <h3 class="negative-value"><?php echo $points['lost_quality_points']; ?></h3>
                            <p>Lost Quality Points</p>
                        </div>
                    </div>
                </div>

                <!-- Today's Stats -->
                <div class="today-stats-section">
                    <div class="today-stats-header">
                        <h2>Today's Overview</h2>
                        <div class="progress-section-inline">
                            <div class="progress-bar-container">
                                <div class="progress-label">
                                    <span><?php echo $progress_percentage; ?>%</span>
                                </div>
                                <div class="progress-bar-wrapper">
                                    <div class="progress-bar-fill" style="width: <?php echo $progress_percentage; ?>%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="today-stats-grid">
                        <div class="today-stat-card" style="cursor: pointer;" data-status="all" onclick="if(typeof window.showTodayTasks === 'function') { window.showTodayTasks('all'); } else { console.error('showTodayTasks function not found'); }">
                            <div class="today-stat-icon">📝</div>
                            <div class="today-stat-info">
                                <h3><?php echo $today_stats['total']; ?></h3>
                                <p>Today Total Tasks</p>
                            </div>
                        </div>
                        <div class="today-stat-card" style="cursor: pointer;" data-status="pending" onclick="if(typeof window.showTodayTasks === 'function') { window.showTodayTasks('pending'); } else { console.error('showTodayTasks function not found'); }">
                            <div class="today-stat-icon">⏱️</div>
                            <div class="today-stat-info">
                                <h3><?php echo $today_stats['pending']; ?></h3>
                                <p>Today Pending Tasks</p>
                            </div>
                        </div>
                        <div class="today-stat-card" style="cursor: pointer;" data-status="completed" onclick="if(typeof window.showTodayTasks === 'function') { window.showTodayTasks('completed'); } else { console.error('showTodayTasks function not found'); }">
                            <div class="today-stat-icon">✅</div>
                            <div class="today-stat-info">
                                <h3><?php echo $today_stats['completed']; ?></h3>
                                <p>Today Completed Tasks</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Pending Tasks List -->
                <div class="pending-tasks-section">
                    <div class="tasks-list">
                        <?php if (empty($pending_tasks)): ?>
                            <?php $show_completed_empty = ($progress_percentage >= 100 && $today_stats['pending'] === 0); ?>
                            <div class="empty-state">
                                <div class="empty-state-icon">
                                    <?php if ($show_completed_empty): ?>
                                        <img src="../icon/completed.png" alt="Completed" />
                                    <?php else: ?>
                                        📝
                                    <?php endif; ?>
                                </div>
                                <h3><?php echo $show_completed_empty ? 'Completed 100%' : 'No tasks found'; ?></h3>
                                <p><?php echo $show_completed_empty ? 'All today\'s tasks are completed.' : 'You have no tasks for today.'; ?></p>
                            </div>
                        <?php else: ?>
                            <?php
                            $completion_today_stmt = $conn->prepare("
                                SELECT status
                                FROM task_completions
                                WHERE task_id = ? AND user_id = ? AND completion_date = ?
                                ORDER BY completed_at DESC
                                LIMIT 1
                            ");
                            ?>
                            <?php foreach ($pending_tasks as $task): ?>
                                <?php
                                // For recurring tasks, use instance date; for one-time, use task dates
                                $effectiveDate = !empty($task['effective_date']) ? $task['effective_date'] : $task['start_date'];
                                $effectiveEndDate = !empty($task['effective_date']) ? $task['effective_date'] : $task['end_date'];
                                $effectiveStartTime = !empty($task['effective_start_time']) ? $task['effective_start_time'] : $task['start_time'];
                                $effectiveEndTime = !empty($task['effective_end_time']) ? $task['effective_end_time'] : $task['end_time'];
                                
                                // If recurring task has no instance for today but today is a valid recurring day,
                                // show it as today's task using today's date/time.
                                if ($task['task_type'] === 'recurring') {
                                    $recurring_days = $task['skip_days'] ? explode(',', $task['skip_days']) : [];
                                    $today_date_obj = new DateTime($current_date_kolkata, new DateTimeZone('Asia/Kolkata'));
                                    $today_day_of_week = (int)$today_date_obj->format('w');
                                    if ((empty($effectiveDate) || $effectiveDate > $current_date_kolkata) 
                                        && !empty($recurring_days) 
                                        && in_array((string)$today_day_of_week, $recurring_days)) {
                                        $effectiveDate = $current_date_kolkata;
                                        $effectiveEndDate = $current_date_kolkata;
                                        $effectiveStartTime = $task['start_time'];
                                        $effectiveEndTime = $task['end_time'];
                                    }
                                }
                                
                                $startDate = new DateTime($effectiveDate, new DateTimeZone('Asia/Kolkata'));
                                $endDate = $effectiveEndDate ? new DateTime($effectiveEndDate, new DateTimeZone('Asia/Kolkata')) : null;
                                $dueTime = $endDate ? $endDate->getTimestamp() * 1000 : $startDate->getTimestamp() * 1000;
                                
                                // Determine today's status (matches modal logic)
                                $status_today = $task['status'];
                                if ($task['task_type'] === 'recurring' && $completion_today_stmt) {
                                    $completion_today_stmt->bind_param("iis", $task['id'], $user_id, $current_date_kolkata);
                                    $completion_today_stmt->execute();
                                    $completion_today_result = $completion_today_stmt->get_result();
                                    $completion_today = $completion_today_result->fetch_assoc();
                                    if ($completion_today) {
                                        $status_today = $completion_today['status'];
                                    } else {
                                        $status_today = 'pending';
                                    }
                                }
                                
                                // Check if task is failed (past end date/time) - using Kolkata timezone
                                $isFailed = false;
                                if ($status_today === 'pending' && $endDate) {
                                    $now = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
                                    $endDateObj = new DateTime($effectiveEndDate, new DateTimeZone('Asia/Kolkata'));
                                    
                                    if ($endDateObj < $now) {
                                        $isFailed = true;
                                    } elseif ($endDateObj->format('Y-m-d') === $now->format('Y-m-d') && !empty($effectiveEndTime)) {
                                        // Same day, check time in Kolkata timezone
                                        $endDateTime = new DateTime($effectiveEndDate . ' ' . $effectiveEndTime, new DateTimeZone('Asia/Kolkata'));
                                        if ($endDateTime < $now) {
                                            $isFailed = true;
                                        }
                                    }
                                }
                                
                                // Only show tasks that are due today in the pending list
                                if ($task['task_type'] === 'recurring') {
                                    if ($effectiveDate !== $current_date_kolkata) {
                                        continue;
                                    }
                                } else {
                                    if (!empty($task['start_date']) && $task['start_date'] > $current_date_kolkata) {
                                        continue;
                                    }
                                    if (!empty($effectiveEndDate) && $effectiveEndDate < $current_date_kolkata) {
                                        continue;
                                    }
                                }
                                
                                // Skip failed tasks from Pending Tasks list
                                if ($status_today === 'failed' || $isFailed) {
                                    continue;
                                }
                                
                                // Only show pending tasks (not completed, not failed)
                                if ($status_today !== 'pending') {
                                    continue;
                                }
                                
                                $typeLabel = $task['task_type'] === 'recurring' ? 'Recurring' : 'One-Time';
                                $lostTaskPointsDisplay = $task['lost_task_points'] < 0 ? $task['lost_task_points'] : '';
                                ?>
                                <div class="task-card <?php echo $task['priority']; ?>-priority <?php echo $task['status'] === 'completed' ? 'completed' : ''; ?> <?php echo $isFailed ? 'failed-overdue failed' : ''; ?>" 
                                     data-task-id="<?php echo $task['id']; ?>"
                                     data-task-type="<?php echo $task['task_type']; ?>"
                                     data-end-date="<?php echo $effectiveEndDate; ?>"
                                     data-end-time="<?php echo $effectiveEndTime ?? ''; ?>"
                                     data-task-end-date="<?php echo $task['end_date'] ?? ''; ?>"
                                     data-skip-days="<?php echo $task['skip_days'] ?? ''; ?>"
                                     data-effective-date="<?php echo $effectiveDate; ?>">
                                    <div class="task-header">
                                        <div class="task-title-section">
                                            <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 4px;">
                                                <h3 class="task-title"><?php echo escapeHtml($task['name']); ?></h3>
                                                <?php if (isset($task['task_number'])): ?>
                                                    <span class="task-number-badge" style="font-size: 11px; color: var(--text-secondary); font-weight: 500; background: var(--bg-tertiary); padding: 2px 8px; border-radius: 6px;">
                                                        <?php echo escapeHtml($task['task_number']); ?>
                                                    </span>
                                                <?php endif; ?>
                                            </div>
                                            <div class="task-meta">
                                                <span class="task-meta-item">
                                                    <span class="icon"><?php echo $task['task_type'] === 'recurring' ? '🔄' : '📌'; ?></span>
                                                    <span><?php echo $typeLabel; ?></span>
                                                </span>
                                                <span class="task-meta-item">
                                                    <span class="icon">📅</span>
                                                    <span>Date: <?php echo formatDate($effectiveDate); ?></span>
                                                </span>
                                                <?php if ($effectiveEndDate): ?>
                                                    <span class="task-meta-item">
                                                        <span class="icon">⏰</span>
                                                        <span>Due: <?php echo formatDate($effectiveEndDate); ?><?php if (!empty($effectiveEndTime)): ?> at <?php echo date('g:i A', strtotime($effectiveEndTime)); ?><?php endif; ?></span>
                                                    </span>
                                                <?php endif; ?>
                                                <?php if (!empty($effectiveStartTime)): ?>
                                                    <span class="task-meta-item">
                                                        <span class="icon">🕐</span>
                                                        <span>Start: <?php echo date('g:i A', strtotime($effectiveStartTime)); ?></span>
                                                    </span>
                                                <?php endif; ?>
                                                <span class="task-meta-item">
                                                    <span class="icon">🏆</span>
                                                    <span>QP <?php echo $task['quality_points']; ?></span>
                                                </span>
                                                <span class="task-meta-item">
                                                    <span class="icon">📊</span>
                                                    <span>TP <?php echo $task['task_points']; ?></span>
                                                </span>
                                                <?php if ($lostTaskPointsDisplay): ?>
                                                    <span class="task-meta-item negative-value">
                                                        <span class="icon">📉</span>
                                                        <span>Lost TP <?php echo $lostTaskPointsDisplay; ?></span>
                                                    </span>
                                                <?php endif; ?>
                                                <span class="task-priority <?php echo $task['priority']; ?>"><?php echo ucfirst($task['priority']); ?> Priority</span>
                                            </div>
                                        </div>
                                        <?php if ($task['status'] === 'completed'): ?>
                                            <span class="task-status completed">Completed</span>
                                        <?php elseif ($task['status'] === 'failed'): ?>
                                            <span class="task-status failed">Failed</span>
                                        <?php elseif ($isFailed): ?>
                                            <span class="task-status failed">Failed</span>
                                        <?php elseif ($task['status'] === 'pending' && !empty($effectiveEndDate)): ?>
                                            <div class="countdown-timer" id="countdown-<?php echo $task['id']; ?>" data-end-date="<?php echo $effectiveEndDate; ?>" data-end-time="<?php echo $effectiveEndTime ?? ''; ?>">
                                                <span class="countdown-label">Remaining time:</span>
                                                <span class="countdown-value">Calculating...</span>
                                                <?php if (!empty($effectiveEndTime)): ?>
                                                    <span class="countdown-end-time">
                                                        <span class="countdown-end-icon">⏰</span>
                                                        <span class="countdown-end-text"><?php echo date('g:i A', strtotime($effectiveEndTime)); ?></span>
                                                    </span>
                                                <?php endif; ?>
                                            </div>
                                        <?php else: ?>
                                            <span class="task-status pending">Pending</span>
                                        <?php endif; ?>
                                    </div>
                                    <p class="task-description"><?php echo escapeHtml($task['description']); ?></p>
                                    <div class="task-footer">
                                        <div class="task-actions">
                                            <?php if ($task['status'] === 'pending' && !$isFailed): ?>
                                                <button class="btn btn-complete btn-small mark-complete-btn" data-task-id="<?php echo $task['id']; ?>" onclick="openMarkCompleteModal(<?php echo $task['id']; ?>)">Mark Complete</button>
                                            <?php endif; ?>
                                            <button class="btn btn-primary btn-small view-task-btn" data-task-id="<?php echo $task['id']; ?>" onclick="viewTaskDetails(<?php echo $task['id']; ?>)">View Details</button>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                            <?php if ($completion_today_stmt) { $completion_today_stmt->close(); } ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Task Detail Modal (iOS Style) -->
    <div id="taskDetailModal" class="modal ios-modal">
        <div class="modal-content ios-modal-content">
            <div class="modal-header ios-modal-header">
                <h2 id="taskDetailTitle">Task Details</h2>
                <button class="modal-close" id="closeTaskDetailModal">&times;</button>
            </div>
            <div id="taskDetailContent" class="ios-modal-body">
                <div class="loading-spinner">
                    <div class="spinner"></div>
                    <p>Loading task details...</p>
                </div>
            </div>
            <div class="ios-modal-actions" id="taskDetailActions" style="display: none;">
                <button class="btn btn-secondary btn-small" id="taskDetailEditBtn">Edit</button>
                <button class="btn btn-danger btn-small" id="taskDetailDeleteBtn">Delete</button>
            </div>
        </div>
    </div>

    <!-- User Tasks Modal (iOS Style) -->
    <div id="userTasksModal" class="modal ios-modal">
        <div class="modal-content ios-modal-content">
            <div class="modal-header ios-modal-header">
                <h2 id="userTasksModalTitle">User Tasks</h2>
                <button class="modal-close" id="closeUserTasksModal">&times;</button>
            </div>
            <div class="ios-modal-toolbar">
                <div class="ios-date-picker-container">
                    <label for="taskDatePicker" class="ios-date-label">📅 Select Date:</label>
                    <input type="date" id="taskDatePicker" class="ios-date-input">
                    <button type="button" id="clearDateBtn" class="ios-clear-date-btn" title="Clear date filter">✕</button>
                </div>
            </div>
            <div class="ios-modal-body" id="userTasksModalBody">
                <div class="loading-spinner">
                    <div class="spinner"></div>
                    <p>Loading tasks...</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Mark Complete Modal -->
    <div id="markCompleteModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Mark Task as Complete</h2>
                <button class="modal-close" id="closeMarkCompleteModal">&times;</button>
            </div>
            <form id="markCompleteForm" method="POST" action="../complete_task.php" enctype="multipart/form-data">
                <input type="hidden" name="task_id" id="hiddenCompleteTaskId" value="">
                <?php if (isset($_GET['complete_error'])): ?>
                    <div class="alert alert-error" style="margin-bottom: 15px; padding: 10px; background: #fee; color: #c33; border-radius: 4px;">
                        <?php echo htmlspecialchars($_GET['complete_error']); ?>
                    </div>
                <?php endif; ?>
                <div class="form-group">
                    <label for="completionUrl">URL (Optional)</label>
                    <input type="url" id="completionUrl" name="completion_url" placeholder="https://example.com">
                </div>
                <div class="form-group">
                    <label for="completionText">Text (Optional)</label>
                    <textarea id="completionText" name="completion_text" rows="4" placeholder="Enter any additional notes or text"></textarea>
                </div>
                <div class="form-group">
                    <label for="completionImage">Upload Image (Optional)</label>
                    <input type="file" id="completionImage" name="completion_image" accept="image/*">
                    <small>Accepted formats: JPG, PNG, GIF</small>
                    <div id="imagePreview" class="image-preview" style="display: none;">
                        <img id="previewImg" src="" alt="Preview">
                        <button type="button" id="removeImage" class="btn btn-small btn-secondary">Remove</button>
                    </div>
                </div>
                <div class="form-group">
                    <small class="form-hint">* At least one field (URL, Text, or Image) must be filled</small>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn btn-secondary" id="cancelMarkCompleteBtn">Cancel</button>
                    <button type="submit" class="btn btn-primary">Submit</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Set timezone to Asia/Kolkata for all JavaScript date operations
        const KOLKATA_TIMEZONE = 'Asia/Kolkata';
        const KOLKATA_OFFSET = 5.5 * 60 * 60 * 1000; // 5 hours 30 minutes in milliseconds
        
        // Helper function to get current time in Kolkata timezone (IST = UTC+5:30)
        function getKolkataTime() {
            const now = new Date();
            const istString = now.toLocaleString('en-US', {
                timeZone: 'Asia/Kolkata',
                year: 'numeric',
                month: '2-digit',
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit',
                hour12: false
            });
            const parts = istString.match(/(\d+)\/(\d+)\/(\d+),\s+(\d+):(\d+):(\d+)/);
            if (parts) {
                const month = parts[1];
                const day = parts[2];
                const year = parts[3];
                const hour = parts[4];
                const minute = parts[5];
                const second = parts[6];
                const isoString = `${year}-${month.padStart(2, '0')}-${day.padStart(2, '0')}T${hour.padStart(2, '0')}:${minute.padStart(2, '0')}:${second.padStart(2, '0')}+05:30`;
                return new Date(isoString);
            }
            return now;
        }
        
        // Helper function to create date in Kolkata timezone
        function createKolkataDate(dateString, timeString = null) {
            if (!timeString) {
                timeString = '23:59:59';
            }
            const timeParts = timeString.split(':');
            if (timeParts.length === 2) {
                timeString = timeString + ':00';
            }
            const isoString = `${dateString}T${timeString}+05:30`;
            return new Date(isoString);
        }
        
        // Helper function to get today's date string in Kolkata timezone (YYYY-MM-DD format)
        function getTodayKolkataDateString() {
            const kolkataTime = getKolkataTime();
            const year = kolkataTime.getFullYear();
            const month = String(kolkataTime.getMonth() + 1).padStart(2, '0');
            const day = String(kolkataTime.getDate()).padStart(2, '0');
            return `${year}-${month}-${day}`;
        }
        
        // Pass PHP variables to JavaScript
        const CURRENT_USER = {
            id: <?php echo $user_id; ?>,
            username: '<?php echo htmlspecialchars($username, ENT_QUOTES); ?>',
            role: '<?php echo htmlspecialchars($user_role, ENT_QUOTES); ?>'
        };
        const IS_LOGGED_IN = true;
    </script>
    <script src="../app-minimal.js"></script>
    <script>
        // Initialize countdown timers after page loads
        if (typeof initCountdownTimers === 'function') {
            initCountdownTimers();
        }

        // Setup modal close button
        document.addEventListener('DOMContentLoaded', function() {
            const closeBtn = document.getElementById('closeUserTasksModal');
            const modal = document.getElementById('userTasksModal');
            if (closeBtn && modal) {
                closeBtn.addEventListener('click', function() {
                    if (typeof closeModal === 'function') {
                        closeModal('userTasksModal');
                    } else {
                        modal.classList.remove('active');
                        document.body.style.overflow = '';
                    }
                });
            }
            
            // Close modal when clicking outside
            if (modal) {
                modal.addEventListener('click', function(e) {
                    if (e.target === modal) {
                        if (typeof closeModal === 'function') {
                            closeModal('userTasksModal');
                        } else {
                            modal.classList.remove('active');
                            document.body.style.overflow = '';
                        }
                    }
                });
            }
            
            // Close mark complete modal before form submission to prevent double modals
            const markCompleteForm = document.getElementById('markCompleteForm');
            if (markCompleteForm) {
                markCompleteForm.addEventListener('submit', function() {
                    // Close the modal before submitting
                    const markCompleteModal = document.getElementById('markCompleteModal');
                    if (markCompleteModal) {
                        if (typeof closeModal === 'function') {
                            closeModal('markCompleteModal');
                        } else {
                            markCompleteModal.classList.remove('active');
                            document.body.style.overflow = '';
                        }
                    }
                });
            }
            
            // Setup close button for mark complete modal
            const closeMarkCompleteBtn = document.getElementById('closeMarkCompleteModal');
            if (closeMarkCompleteBtn) {
                closeMarkCompleteBtn.addEventListener('click', function() {
                    const markCompleteModal = document.getElementById('markCompleteModal');
                    if (markCompleteModal) {
                        if (typeof closeModal === 'function') {
                            closeModal('markCompleteModal');
                        } else {
                            markCompleteModal.classList.remove('active');
                            document.body.style.overflow = '';
                        }
                    }
                });
            }
        });

        // Show today's tasks in modal (make globally available)
        window.showTodayTasks = async function showTodayTasks(status) {
            console.log('showTodayTasks called with status:', status);
            const modal = document.getElementById('userTasksModal');
            const modalTitle = document.getElementById('userTasksModalTitle');
            const modalBody = document.getElementById('userTasksModalBody');
            const datePicker = document.getElementById('taskDatePicker');
            const clearDateBtn = document.getElementById('clearDateBtn');
            
            if (!modal || !modalTitle || !modalBody) {
                const errorMsg = 'Tasks modal not found. Please refresh the page.';
                console.error('Modal elements not found:', { modal, modalTitle, modalBody });
                if (typeof showToast === 'function') {
                    showToast(errorMsg, 'error');
                } else {
                    alert(errorMsg);
                }
                return;
            }
            
            console.log('Modal elements found, proceeding...');
            
            // Store current status for date filter changes
            window.currentTasksModalStatus = status;
            
            // Set title based on status
            const titles = {
                'all': 'Tasks',
                'pending': 'Pending Tasks',
                'completed': 'Completed Tasks',
                'failed': 'Failed Tasks'
            };
            modalTitle.textContent = titles[status] || 'Tasks';
            
            // Set today's date in date picker (using Kolkata timezone)
            if (datePicker) {
                const today = getTodayKolkataDateString();
                datePicker.value = today;
                if (clearDateBtn) {
                    clearDateBtn.style.display = 'inline-block';
                }
            }
            
            // Wire date filter change once
            if (datePicker && !datePicker.dataset.bound) {
                datePicker.addEventListener('change', function () {
                    const selectedDate = datePicker.value || getTodayKolkataDateString();
                    window.loadTodayTasksForModal(window.currentTasksModalStatus || 'all', selectedDate);
                });
                datePicker.dataset.bound = 'true';
            }
            
            if (clearDateBtn && !clearDateBtn.dataset.bound) {
                clearDateBtn.addEventListener('click', function () {
                    if (datePicker) {
                        datePicker.value = getTodayKolkataDateString();
                    }
                    window.loadTodayTasksForModal(window.currentTasksModalStatus || 'all', getTodayKolkataDateString());
                });
                clearDateBtn.dataset.bound = 'true';
            }
            
            // Show loading state
            modalBody.innerHTML = `
                <div class="loading-spinner">
                    <div class="spinner"></div>
                    <p>Loading tasks...</p>
                </div>
            `;
            
            // Open modal using the openModal function if available, otherwise use classList
            if (typeof openModal === 'function') {
                openModal('userTasksModal');
            } else {
                modal.classList.add('active');
                document.body.style.overflow = 'hidden';
            }
            
            // Wait a bit for modal to be visible, then load tasks
            setTimeout(async () => {
                await window.loadTodayTasksForModal(status, datePicker ? datePicker.value : null);
            }, 100);
        }

        // Load today's tasks for the modal (make globally available)
        window.loadTodayTasksForModal = async function loadTodayTasksForModal(status, selectedDate = null) {
            const modalBody = document.getElementById('userTasksModalBody');
            const modalTitle = document.getElementById('userTasksModalTitle');
            
            if (!modalBody) return;
            
            try {
                // Use selected date or today's date in Kolkata timezone
                const dateValue = selectedDate || getTodayKolkataDateString();
                let url = `../api/tasks.php?date=${dateValue}`;
                
                if (status !== 'all') {
                    url += `&status=${status}`;
                }
                
                console.log('Fetching tasks from:', url);
                const response = await fetch(url);
                
                if (!response.ok) {
                    const errorText = await response.text();
                    console.error('API error response:', errorText);
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                const result = await response.json();
                console.log('API response:', result);
                
                if (!result.success) {
                    throw new Error(result.message || 'Failed to load tasks');
                }
                
                if (result.success && result.tasks) {
                    const tasks = result.tasks;
                    
                if (tasks.length === 0) {
                        modalBody.innerHTML = `
                            <div class="ios-empty-state">
                                <div class="ios-empty-icon">📝</div>
                                <h3>No tasks found</h3>
                            <p>No ${status === 'all' ? '' : status} tasks for this date.</p>
                            </div>
                        `;
                    } else {
                        modalBody.innerHTML = `
                            <div class="ios-task-stats">
                                <div class="ios-stat-item">
                                    <span class="ios-stat-label">Total:</span>
                                    <span class="ios-stat-value">${tasks.length}</span>
                                </div>
                            </div>
                        ` + tasks.map(task => {
                            const startDate = new Date(task.start_date);
                            const endDate = task.end_date ? new Date(task.end_date) : null;
                            const effectiveDate = task.effective_date || task.start_date;
                            const effectiveEndDate = task.effective_end_date || task.end_date || effectiveDate;
                            const effectiveStartTime = task.effective_start_time || task.start_time;
                            const effectiveEndTime = task.effective_end_time || task.end_time;
                            
                            const priorityColors = {
                                high: '#0f8a1f',
                                medium: '#28a745',
                                low: '#6fa86f'
                            };
                            const pColor = priorityColors[task.priority] || '#86868b';
                            
                            const statusColors = {
                                pending: '#ff9500',
                                completed: '#34c759',
                                failed: '#ff3b30'
                            };
                            const sColor = statusColors[task.status] || '#86868b';
                            const statusIcons = {
                                pending: '⏳',
                                completed: '✅',
                                failed: '❌'
                            };
                            const statusIcon = statusIcons[task.status] || '';
                            const isCompleted = task.status === 'completed';
                            
                            return `
                                <div class="ios-task-item ${isCompleted ? 'ios-task-item-completed' : ''}">
                                    <div class="ios-task-header">
                                        <div class="ios-task-badges">
                                            <span class="ios-task-status" style="background: ${sColor}20; color: ${sColor}; border-color: ${sColor}40;">
                                                ${statusIcon ? `<span class="status-icon">${statusIcon}</span>` : ''}
                                                ${task.status || 'pending'}
                                            </span>
                                            <span class="ios-task-priority" style="background: ${pColor}20; color: ${pColor}; border-color: ${pColor}40;">${task.priority}</span>
                                        </div>
                                    </div>
                                    <h3 class="ios-task-title">
                                        ${isCompleted ? '<span class="ios-task-complete-icon">✅</span>' : ''}
                                        ${escapeHtml(task.name)}
                                    </h3>
                                    <div class="ios-task-meta">
                                        <span class="ios-task-meta-item">
                                            <span class="ios-meta-icon">📅</span>
                                            <span>${new Date(effectiveDate).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })}</span>
                                        </span>
                                        ${effectiveEndTime ? `
                                        <span class="ios-task-meta-item">
                                            <span class="ios-meta-icon">⏰</span>
                                            <span>Due: ${new Date(effectiveEndDate + 'T' + effectiveEndTime).toLocaleTimeString('en-US', { hour: 'numeric', minute: '2-digit' })}</span>
                                        </span>
                                        ` : ''}
                                        <span class="ios-task-meta-item">
                                            <span class="ios-meta-icon">${task.task_type === 'recurring' ? '🔄' : '📌'}</span>
                                            <span>${task.task_type === 'recurring' ? 'Recurring' : 'One-Time'}</span>
                                        </span>
                                    </div>
                                    <div class="ios-task-points">
                                        <span class="ios-point-badge">TP: ${task.task_points}</span>
                                        <span class="ios-point-badge">QP: ${task.quality_points}</span>
                                    </div>
                                    ${task.description ? `<p class="ios-task-description">${escapeHtml(task.description)}</p>` : ''}
                                </div>
                            `;
                        }).join('');
                    }
                } else {
                    modalBody.innerHTML = `
                        <div class="ios-empty-state">
                            <div class="ios-empty-icon">❌</div>
                            <h3>Error loading tasks</h3>
                            <p>${result.message || 'Failed to load tasks'}</p>
                        </div>
                    `;
                }
            } catch (error) {
                console.error('Error loading today tasks:', error);
                modalBody.innerHTML = `
                    <div class="ios-empty-state">
                        <div class="ios-empty-icon">❌</div>
                        <h3>Error</h3>
                        <p>${error.message}</p>
                    </div>
                `;
            }
        }

        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    </script>
</body>
</html>
