<?php
// Start output buffering to capture any errors
ob_start();

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Register shutdown function to catch fatal errors
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        ob_clean(); // Clear any previous output
        http_response_code(500);
        header('Content-Type: text/html; charset=utf-8');
        echo "<!DOCTYPE html><html><head><title>Fatal Error</title><meta charset='utf-8'></head><body>";
        echo "<h1>Fatal Error</h1>";
        echo "<p><strong>Error:</strong> " . htmlspecialchars($error['message']) . "</p>";
        echo "<p><strong>File:</strong> " . htmlspecialchars($error['file']) . "</p>";
        echo "<p><strong>Line:</strong> " . $error['line'] . "</p>";
        echo "<p><strong>Type:</strong> " . $error['type'] . "</p>";
        ob_end_flush();
        exit;
    }
});

require_once '../config.php';

if (!isLoggedIn()) {
    header('Location: ../index.php');
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    http_response_code(500);
    header('Content-Type: text/html');
    echo "<!DOCTYPE html><html><head><title>Error</title></head><body>";
    echo "<h1>Database Connection Failed</h1>";
    echo "<p>Database connection failed. Please check your configuration.</p>";
    exit;
}

$user_id = getCurrentUserId();
$username = getCurrentUsername();
$full_name = $_SESSION['full_name'] ?? $username;
$user_role = getCurrentUserRole();

// Get notification count
$notification_count = 0;
try {
    $notif_count_stmt = $conn->prepare("SELECT COUNT(*) as count FROM notifications WHERE user_id = ? AND is_read = 0");
    if ($notif_count_stmt) {
        $notif_count_stmt->bind_param("i", $user_id);
        $notif_count_stmt->execute();
        $notif_count_result = $notif_count_stmt->get_result();
        if ($notif_count_result) {
            $notification_count = $notif_count_result->fetch_assoc()['count'] ?? 0;
        }
        $notif_count_stmt->close();
    }
} catch (Exception $e) {
    $notification_count = 0;
}

// Get current date and time in Kolkata timezone for accurate comparisons
try {
    $now_kolkata = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
    $current_date_kolkata = $now_kolkata->format('Y-m-d');
    $current_time_kolkata = $now_kolkata->format('H:i:s');
} catch (Exception $e) {
    error_log("Error creating DateTime: " . $e->getMessage());
    // Fallback to current date/time
    $current_date_kolkata = date('Y-m-d');
    $current_time_kolkata = date('H:i:s');
}

// Get filter parameters
$date_from = $_GET['date_from'] ?? null;
$date_to = $_GET['date_to'] ?? null;

// Build query to get all tasks (completed, failed, and pending)
// Use UNION to get completed/failed tasks from task_completions and pending tasks from assignments
$query = "
    SELECT * FROM (
    (
        -- Get completed and failed tasks from task_completions
        SELECT DISTINCT
               t.id as task_id,
               t.name as task_name,
               t.description as task_description,
               t.priority,
               t.task_number,
               t.task_type,
               COALESCE(rti.instance_date, tc.completion_date, t.start_date) as effective_date,
               COALESCE(rti.end_time, t.end_time) as effective_end_time,
               tc.status,
               tc.completion_date,
               tc.completion_url,
               tc.completion_text,
               tc.completion_image,
               tc.task_points_earned,
               tc.quality_points_earned,
               tc.completed_at,
               COALESCE(rti.instance_date, tc.completion_date, t.start_date) as task_date
        FROM task_completions tc
        JOIN tasks t ON tc.task_id = t.id
        JOIN task_assignments ta ON ta.task_id = t.id AND ta.user_id = tc.user_id
        LEFT JOIN recurring_task_instances rti ON t.id = rti.parent_task_id 
            AND t.task_type = 'recurring'
            AND rti.instance_date = tc.completion_date
        WHERE tc.user_id = ? 
          AND t.is_enabled = 1
          AND tc.completion_date >= ?
          AND tc.completion_date <= ?
    )
    UNION
    (
        -- Get pending tasks from task_assignments (not in task_completions or not completed)
        SELECT DISTINCT
               t.id as task_id,
               t.name as task_name,
               t.description as task_description,
               t.priority,
               t.task_number,
               t.task_type,
               COALESCE(rti.instance_date, t.start_date) as effective_date,
               COALESCE(rti.end_time, t.end_time) as effective_end_time,
               CASE 
                   WHEN (COALESCE(rti.instance_date, t.end_date) IS NOT NULL AND COALESCE(rti.instance_date, t.end_date) < ?) 
                        OR (COALESCE(rti.instance_date, t.end_date) = ? AND COALESCE(rti.end_time, t.end_time) IS NOT NULL AND COALESCE(rti.end_time, t.end_time) < ?)
                        THEN 'failed'
                   ELSE 'pending' 
               END as status,
               NULL as completion_date,
               NULL as completion_url,
               NULL as completion_text,
               NULL as completion_image,
               NULL as task_points_earned,
               NULL as quality_points_earned,
               NULL as completed_at,
               COALESCE(rti.instance_date, t.start_date) as task_date
        FROM task_assignments ta
        JOIN tasks t ON ta.task_id = t.id
        LEFT JOIN recurring_task_instances rti ON t.id = rti.parent_task_id 
            AND t.task_type = 'recurring'
            AND rti.instance_date >= ?
            AND rti.instance_date <= ?
        WHERE ta.user_id = ? 
          AND t.is_enabled = 1
          AND (
              (t.task_type = 'one-time' 
               AND t.start_date >= ? 
               AND t.start_date <= ?
               AND NOT EXISTS (
                   SELECT 1 FROM task_completions tc2 
                   WHERE tc2.task_id = t.id 
                   AND tc2.user_id = ta.user_id 
                   AND tc2.status = 'completed'
               ))
              OR (t.task_type = 'recurring' 
                  AND rti.instance_date >= ?
                  AND rti.instance_date <= ?
                  AND (rti.status IS NULL OR rti.status = 'pending')
                  AND NOT EXISTS (
                      SELECT 1 FROM task_completions tc3 
                      WHERE tc3.task_id = t.id 
                      AND tc3.user_id = ta.user_id 
                      AND tc3.completion_date = rti.instance_date
                      AND tc3.status = 'completed'
                  ))
          )
    )
    ) AS combined
    ORDER BY combined.task_date DESC, 
             CASE WHEN combined.status = 'completed' THEN 1 
                  WHEN combined.status = 'failed' THEN 2 
                  ELSE 3 END
    LIMIT 100
";

$params = [];
$types = '';

// Set default date range if not provided (last 2 days + today)
if (!$date_from) {
    $date_from = date('Y-m-d', strtotime('-2 days', strtotime($current_date_kolkata)));
}
if (!$date_to) {
    $date_to = $current_date_kolkata;
}

// Bind parameters for the query
$params = [
    $user_id,              // 1: line 113 (i)
    $date_from,            // 2: line 115 (s)
    $date_to,              // 3: line 116 (s)
    $current_date_kolkata, // 4: line 131 (s)
    $current_date_kolkata, // 5: line 132 (s)
    $current_time_kolkata, // 6: line 132 (s)
    $date_from,            // 7: line 148 (s)
    $date_to,              // 8: line 149 (s)
    $user_id,              // 9: line 150 (i)
    $date_from,            // 10: line 154 (s)
    $date_to,              // 11: line 155 (s)
    $date_from,            // 12: line 163 (s)
    $date_to               // 13: line 164 (s)
];

$types = 'issssssssisss'; // 13 parameters: i(1)-s(2)-s(3)-s(4)-s(5)-s(6)-s(7)-s(8)-s(9)-i(10)-s(11)-s(12)-s(13)

$stmt = $conn->prepare($query);
if (!$stmt) {
    http_response_code(500);
    header('Content-Type: text/html');
    echo "<!DOCTYPE html><html><head><title>Error</title></head><body>";
    echo "<h1>Query Preparation Failed</h1>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($conn->error) . "</p>";
    exit;
}

if (!$stmt->bind_param($types, ...$params)) {
    http_response_code(500);
    header('Content-Type: text/html');
    echo "<!DOCTYPE html><html><head><title>Error</title></head><body>";
    echo "<h1>Bind Parameter Failed</h1>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($stmt->error) . "</p>";
    echo "<p><strong>Expected:</strong> " . strlen($types) . " parameters</p>";
    echo "<p><strong>Types:</strong> " . htmlspecialchars($types) . "</p>";
    echo "<p><strong>Params count:</strong> " . count($params) . "</p>";
    exit;
}

if (!$stmt->execute()) {
    http_response_code(500);
    header('Content-Type: text/html');
    echo "<!DOCTYPE html><html><head><title>Error</title></head><body>";
    echo "<h1>Query Execution Failed</h1>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($stmt->error) . "</p>";
    exit;
}

$result = $stmt->get_result();

$history = [];
while ($row = $result->fetch_assoc()) {
    // Status is already determined in the query, but ensure it's set
    if (empty($row['status'])) {
        // Check if task is actually failed (past due date)
        $task_date = $row['effective_date'] ?? $row['task_date'];
        $end_time = $row['effective_end_time'];
        
        if ($task_date && $task_date < date('Y-m-d')) {
            $row['status'] = 'failed';
        } elseif ($task_date === date('Y-m-d') && $end_time) {
            $now = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
            $end_datetime = new DateTime($task_date . ' ' . $end_time, new DateTimeZone('Asia/Kolkata'));
            if ($end_datetime < $now) {
                $row['status'] = 'failed';
            } else {
                $row['status'] = 'pending';
            }
        } else {
            $row['status'] = 'pending';
        }
    }
    
    $history[] = $row;
}

// Sort by date (most recent first) and then by status
usort($history, function($a, $b) {
    $date_a = $a['task_date'] ?? $a['effective_date'] ?? $a['completion_date'] ?? '';
    $date_b = $b['task_date'] ?? $b['effective_date'] ?? $b['completion_date'] ?? '';
    
    if ($date_a === $date_b) {
        // If same date, sort by status: completed first, then failed, then pending
        $status_order = ['completed' => 1, 'failed' => 2, 'pending' => 3];
        return ($status_order[$a['status']] ?? 4) - ($status_order[$b['status']] ?? 4);
    }
    
    return strcmp($date_b, $date_a); // Most recent first
});

function formatDate($dateString) {
    return date('M d, Y', strtotime($dateString));
}

function formatDateTime($dateString) {
    return date('M d, Y H:i', strtotime($dateString));
}

function escapeHtml($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}

$nav_titles = [
    'user-dashboard.php' => 'Dashboard',
    'user-tasks.php' => 'My Tasks',
    'user-progress.php' => 'Progress',
    'user-history.php' => 'History'
];
$current_page = basename($_SERVER['PHP_SELF'] ?? '');
$nav_title = $nav_titles[$current_page] ?? 'Task Manager';

function isLateCompletion($taskDate, $endTime, $completedAt) {
    if (!$taskDate || !$endTime || !$completedAt) {
        return null;
    }
    try {
        $tz = new DateTimeZone('Asia/Kolkata');
        $deadline = new DateTime($taskDate . ' ' . $endTime, $tz);
        $completed = new DateTime($completedAt, $tz);
        return $completed > $deadline;
    } catch (Exception $e) {
        return null;
    }
}

$summary = [
    'total' => count($history),
    'completed' => 0,
    'failed' => 0,
    'pending' => 0,
    'points' => 0,
    'quality' => 0
];

foreach ($history as $item) {
    $status = $item['status'] ?? 'pending';
    if (isset($summary[$status])) {
        $summary[$status]++;
    }
    if ($item['task_points_earned'] !== null) {
        $summary['points'] += (int)$item['task_points_earned'];
    }
    if ($item['quality_points_earned'] !== null) {
        $summary['quality'] += (int)$item['quality_points_earned'];
    }
}

$range_label = formatDate($date_from) . ' - ' . formatDate($date_to);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>History - Task Management System</title>
    <link rel="stylesheet" href="../styles.css">
    <style>
        :root {
            --ios-accent-blue: #2f80ed;
            --ios-accent-purple: #9b51e0;
            --ios-accent-pink: #eb5757;
            --ios-accent-teal: #00b894;
            --ios-accent-orange: #f2994a;
        }
        .ios-page {
            padding: 8px 6px 80px;
        }
        .ios-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 16px;
        }
        .ios-large-title {
            font-size: 28px;
            font-weight: 700;
            letter-spacing: -0.3px;
            margin: 0;
        }
        .ios-subtitle {
            color: var(--text-secondary);
            font-size: 13px;
            margin-top: 4px;
        }
        .ios-card {
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 18px;
            padding: 14px;
            box-shadow: 0 6px 18px rgba(0, 0, 0, 0.06);
        }
        .ios-card.color-pop {
            border: none;
            color: #fff;
        }
        .ios-card.color-pop.total {
            background: linear-gradient(135deg, var(--ios-accent-blue), #56ccf2);
        }
        .ios-card.color-pop.completed {
            background: linear-gradient(135deg, #27ae60, var(--ios-accent-teal));
        }
        .ios-card.color-pop.failed {
            background: linear-gradient(135deg, #eb5757, #f2994a);
        }
        .ios-card.color-pop.pending {
            background: linear-gradient(135deg, #f2c94c, #f2994a);
        }
        .ios-card.color-pop.points {
            background: linear-gradient(135deg, var(--ios-accent-purple), #bb6bd9);
        }
        .ios-card.color-pop.quality {
            background: linear-gradient(135deg, #6a11cb, #2575fc);
        }
        .ios-summary-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
            gap: 10px;
            margin-bottom: 16px;
        }
        .ios-summary-item {
            display: flex;
            align-items: center;
            justify-content: space-between;
            font-size: 13px;
        }
        .ios-summary-value {
            font-weight: 700;
            font-size: 18px;
        }
        .ios-segmented {
            display: inline-flex;
            background: var(--bg-tertiary);
            border-radius: 14px;
            padding: 4px;
            gap: 4px;
        }
        .ios-segment {
            background: transparent;
            border: none;
            padding: 6px 10px;
            border-radius: 10px;
            font-size: 12px;
            color: var(--text-secondary);
        }
        .ios-segment.active {
            background: var(--bg-primary);
            color: var(--text-primary);
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.06);
        }
        .ios-filter-bar {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 14px;
        }
        .ios-filter-group {
            display: flex;
            gap: 8px;
            align-items: center;
        }
        .ios-filter-group label {
            font-size: 12px;
            color: var(--text-secondary);
        }
        .ios-filter-input {
            background: var(--bg-tertiary);
            border: 1px solid var(--border-color);
            border-radius: 12px;
            padding: 6px 10px;
            font-size: 12px;
            color: var(--text-primary);
        }
        .ios-history-list {
            display: grid;
            gap: 12px;
        }
        .ios-history-item {
            position: relative;
            overflow: hidden;
        }
        .ios-history-item.status-failed {
            border-color: rgba(235, 87, 87, 0.35);
            background: linear-gradient(180deg, rgba(235, 87, 87, 0.08), rgba(235, 87, 87, 0.02));
        }
        .ios-history-item.status-pending {
            border-color: rgba(242, 153, 74, 0.35);
            background: linear-gradient(180deg, rgba(242, 153, 74, 0.08), rgba(242, 153, 74, 0.02));
        }
        .ios-status-pill {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            padding: 4px 10px;
            border-radius: 999px;
            font-size: 11px;
            font-weight: 600;
        }
        .ios-status-completed {
            background: rgba(46, 204, 113, 0.12);
            color: #1e7f4a;
        }
        .ios-status-failed {
            background: rgba(231, 76, 60, 0.12);
            color: #b03a2e;
        }
        .ios-status-pending {
            background: rgba(241, 196, 15, 0.12);
            color: #8f6b00;
        }
        .ios-status-completed::before,
        .ios-status-failed::before,
        .ios-status-pending::before {
            content: '';
            width: 8px;
            height: 8px;
            border-radius: 999px;
            display: inline-block;
        }
        .ios-status-completed::before {
            background: #27ae60;
        }
        .ios-status-failed::before {
            background: #eb5757;
        }
        .ios-status-pending::before {
            background: #f2c94c;
        }
        .ios-meta {
            display: flex;
            flex-wrap: wrap;
            gap: 8px 14px;
            margin-top: 8px;
            font-size: 12px;
            color: var(--text-secondary);
        }
        .ios-meta strong {
            color: var(--text-primary);
            font-weight: 600;
        }
        .ios-badge {
            background: var(--bg-tertiary);
            border-radius: 10px;
            padding: 2px 8px;
            font-size: 11px;
        }
        .ios-badge.accent {
            background: rgba(47, 128, 237, 0.12);
            color: var(--ios-accent-blue);
        }
        .ios-badge.warn {
            background: rgba(242, 153, 74, 0.14);
            color: #b86b2f;
        }
        .ios-badge.success {
            background: rgba(39, 174, 96, 0.14);
            color: #1e7f4a;
        }
        .ios-divider {
            height: 1px;
            background: var(--border-color);
            margin: 10px 0;
        }
        .ios-empty {
            text-align: center;
            padding: 30px 12px;
        }
        @media (max-width: 768px) {
            .ios-page {
                padding-bottom: 120px;
            }
            .ios-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 8px;
            }
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar">
        <div class="nav-left">
            <h2 class="nav-title"><?php echo htmlspecialchars($nav_title, ENT_QUOTES, 'UTF-8'); ?></h2>
        </div>
        <div class="nav-right">
            <button id="notificationBtn" class="nav-icon-btn" title="Notifications" style="display: <?php echo $notification_count > 0 ? 'flex' : 'none'; ?>;">
                <span class="icon">🔔</span>
                <?php if ($notification_count > 0): ?>
                <span id="notificationBadge" class="badge"><?php echo $notification_count; ?></span>
                <?php endif; ?>
            </button>
            <button id="themeToggle" class="nav-icon-btn" title="Toggle Theme">
                <span class="icon" id="themeIcon">🌙</span>
            </button>
            <div class="user-menu">
                <span id="userNameDisplay"><?php echo htmlspecialchars($username); ?></span>
                <a href="../logout.php" class="btn btn-small">Logout</a>
            </div>
        </div>
    </nav>

    <!-- Notification Dropdown -->
    <div id="notificationDropdown" class="notification-dropdown hidden">
        <div class="notification-header">
            <h3>Notifications</h3>
            <button id="clearNotifications" class="btn btn-small">Clear All</button>
        </div>
        <div id="notificationList" class="notification-list"></div>
    </div>

    <!-- Main Content -->
    <div class="main-container">
        <!-- Sidebar (Desktop) -->
        <aside class="sidebar">
            <nav class="sidebar-nav">
                <a href="user-dashboard.php" class="nav-link">
                    <img class="nav-icon" src="../icon/dashboard_icon.png" alt="Dashboard">
                    <span>Dashboard</span>
                </a>
                <a href="user-tasks.php" class="nav-link">
                    <img class="nav-icon" src="../icon/mytask.png" alt="My Tasks">
                    <span>My Tasks</span>
                </a>
                <a href="user-progress.php" class="nav-link">
                    <img class="nav-icon" src="../icon/progress.png" alt="Progress">
                    <span>Progress</span>
                </a>
                <a href="user-history.php" class="nav-link active">
                    <img class="nav-icon" src="../icon/history-icon.png" alt="History">
                    <span>History</span>
                </a>
            </nav>
        </aside>

        <!-- Mobile Bottom Nav -->
        <nav class="bottom-nav">
            <a href="user-dashboard.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/dashboard_icon.png" alt="Dashboard">
                <span>Dashboard</span>
            </a>
            <a href="user-tasks.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/mytask.png" alt="Tasks">
                <span>Tasks</span>
            </a>
            <a href="user-progress.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/progress.png" alt="Progress">
                <span>Progress</span>
            </a>
            <a href="user-history.php" class="bottom-nav-link active">
                <img class="nav-icon" src="../icon/history-icon.png" alt="History">
                <span>History</span>
            </a>
        </nav>

        <!-- Content Area -->
        <main class="content">
            <div class="page active ios-page">
                <div class="ios-header">
                    <div>
                    </div>
                </div>

                <div class="ios-summary-grid">
                    <div class="ios-card ios-summary-item color-pop total">
                        <span>Total</span>
                        <span class="ios-summary-value"><?php echo $summary['total']; ?></span>
                    </div>
                    <div class="ios-card ios-summary-item color-pop completed">
                        <span>Completed</span>
                        <span class="ios-summary-value"><?php echo $summary['completed']; ?></span>
                    </div>
                    <div class="ios-card ios-summary-item color-pop failed">
                        <span>Failed</span>
                        <span class="ios-summary-value"><?php echo $summary['failed']; ?></span>
                    </div>
                    <div class="ios-card ios-summary-item color-pop pending">
                        <span>Pending</span>
                        <span class="ios-summary-value"><?php echo $summary['pending']; ?></span>
                    </div>
                    <div class="ios-card ios-summary-item color-pop points">
                        <span>Task Points</span>
                        <span class="ios-summary-value"><?php echo $summary['points']; ?></span>
                    </div>
                    <div class="ios-card ios-summary-item color-pop quality">
                        <span>Quality Points</span>
                        <span class="ios-summary-value"><?php echo $summary['quality']; ?></span>
                    </div>
                </div>

                <div class="ios-card" style="margin-bottom: 14px;">
                    <form id="historyFilterForm" method="GET" action="" class="ios-filter-bar">
                        <div class="ios-filter-group">
                            <label for="historyDateFrom">From</label>
                            <input type="date" name="date_from" id="historyDateFrom" value="<?php echo $date_from ?? ''; ?>" class="ios-filter-input">
                        </div>
                        <div class="ios-filter-group">
                            <label for="historyDateTo">To</label>
                            <input type="date" name="date_to" id="historyDateTo" value="<?php echo $date_to ?? ''; ?>" class="ios-filter-input">
                        </div>
                        <button type="submit" class="btn btn-small">Apply</button>
                    </form>
                </div>

                <div class="ios-history-list">
                    <?php if (empty($history)): ?>
                        <div class="ios-card ios-empty">
                            <div class="empty-state-icon">📜</div>
                            <h3>No history yet</h3>
                            <p>Complete tasks to build your activity history.</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($history as $item): ?>
                            <?php
                            $task_date = $item['task_date'] ?? $item['effective_date'] ?? $item['completion_date'] ?? '';
                            $display_date = $task_date ? formatDate($task_date) : 'N/A';
                            $completion_time = $item['completed_at'] ? formatDateTime($item['completed_at']) : null;
                            $display_time = $completion_time ?? ($task_date ? formatDate($task_date) : 'N/A');
                            $priority = $item['priority'] ?? 'normal';
                            $task_type = $item['task_type'] ?? 'one-time';
                            $end_time = $item['effective_end_time'] ?? null;
                            $late_flag = isLateCompletion($task_date, $end_time, $item['completed_at']);
                            ?>
                            <div class="ios-card ios-history-item status-<?php echo $item['status']; ?>">
                                <div style="display: flex; justify-content: space-between; gap: 10px; align-items: flex-start;">
                                    <div>
                                        <div style="display: flex; align-items: center; gap: 8px; flex-wrap: wrap;">
                                            <h3 class="history-title" style="margin: 0;"><?php echo escapeHtml($item['task_name']); ?></h3>
                                            <?php if (!empty($item['task_number'])): ?>
                                                <span class="ios-badge accent">#<?php echo escapeHtml($item['task_number']); ?></span>
                                            <?php endif; ?>
                                            <span class="ios-badge"><?php echo ucfirst($task_type); ?></span>
                                        </div>
                                        <div class="ios-subtitle"><?php echo escapeHtml($item['task_description'] ?? ''); ?></div>
                                    </div>
                                    <div class="ios-status-pill ios-status-<?php echo $item['status']; ?>">
                                        <?php echo ucfirst($item['status']); ?>
                                    </div>
                                </div>
                                <div class="ios-meta">
                                    <span><strong>Date:</strong> <?php echo $display_date; ?></span>
                                    <span><strong>Activity:</strong> <?php echo $display_time; ?></span>
                                    <?php if ($end_time): ?>
                                        <span><strong>Due:</strong> <?php echo escapeHtml($end_time); ?></span>
                                    <?php endif; ?>
                                    <span><strong>Priority:</strong> <?php echo ucfirst($priority); ?></span>
                                    <?php if ($late_flag === true): ?>
                                        <span class="ios-badge warn">Late completion</span>
                                    <?php elseif ($late_flag === false && $item['status'] === 'completed'): ?>
                                        <span class="ios-badge success">On time</span>
                                    <?php endif; ?>
                                </div>

                                <?php if ($item['status'] === 'completed' || $item['status'] === 'failed'): ?>
                                    <div class="ios-divider"></div>
                                    <div class="ios-meta">
                                        <?php if ($item['task_points_earned'] !== null): ?>
                                            <span><strong>Points:</strong> <?php echo $item['task_points_earned']; ?> TP · <?php echo $item['quality_points_earned']; ?> QP</span>
                                        <?php endif; ?>
                                        <?php if ($completion_time): ?>
                                            <span><strong>Completed:</strong> <?php echo $completion_time; ?></span>
                                        <?php endif; ?>
                                        <?php if ($item['completion_text']): ?>
                                            <span><strong>Notes:</strong> <?php echo escapeHtml($item['completion_text']); ?></span>
                                        <?php endif; ?>
                                        <?php if ($item['completion_url']): ?>
                                            <span><strong>Link:</strong> <a href="<?php echo escapeHtml($item['completion_url']); ?>" target="_blank">Open</a></span>
                                        <?php endif; ?>
                                        <?php if ($item['completion_image']): ?>
                                            <span><strong>Image:</strong> <a href="<?php echo escapeHtml($item['completion_image']); ?>" target="_blank">View</a></span>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Set timezone to Asia/Kolkata for all JavaScript date operations
        const KOLKATA_TIMEZONE = 'Asia/Kolkata';
        const KOLKATA_OFFSET = 5.5 * 60 * 60 * 1000;
        
        function getKolkataTime() {
            const now = new Date();
            const istString = now.toLocaleString('en-US', {
                timeZone: 'Asia/Kolkata',
                year: 'numeric',
                month: '2-digit',
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit',
                hour12: false
            });
            const parts = istString.match(/(\d+)\/(\d+)\/(\d+),\s+(\d+):(\d+):(\d+)/);
            if (parts) {
                const month = parts[1];
                const day = parts[2];
                const year = parts[3];
                const hour = parts[4];
                const minute = parts[5];
                const second = parts[6];
                const isoString = `${year}-${month.padStart(2, '0')}-${day.padStart(2, '0')}T${hour.padStart(2, '0')}:${minute.padStart(2, '0')}:${second.padStart(2, '0')}+05:30`;
                return new Date(isoString);
            }
            return now;
        }
        
        function createKolkataDate(dateString, timeString = null) {
            if (!timeString) {
                timeString = '23:59:59';
            }
            const timeParts = timeString.split(':');
            if (timeParts.length === 2) {
                timeString = timeString + ':00';
            }
            const isoString = `${dateString}T${timeString}+05:30`;
            return new Date(isoString);
        }
        
        const CURRENT_USER = {
            id: <?php echo $user_id; ?>,
            username: '<?php echo htmlspecialchars($username, ENT_QUOTES); ?>',
            role: '<?php echo htmlspecialchars($user_role, ENT_QUOTES); ?>'
        };
        const IS_LOGGED_IN = true;
    </script>
    <script src="../app-minimal.js"></script>
</body>
</html>
