<?php
require_once '../config.php';

if (!isLoggedIn()) {
    header('Location: ../index.php');
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    die("Database connection failed. Please check your configuration.");
}

$user_id = getCurrentUserId();
$username = getCurrentUsername();
$full_name = $_SESSION['full_name'] ?? $username;
$user_role = getCurrentUserRole();

// Get notification count
$notification_count = 0;
try {
    $notif_count_stmt = $conn->prepare("SELECT COUNT(*) as count FROM notifications WHERE user_id = ? AND is_read = 0");
    if ($notif_count_stmt) {
        $notif_count_stmt->bind_param("i", $user_id);
        $notif_count_stmt->execute();
        $notif_count_result = $notif_count_stmt->get_result();
        if ($notif_count_result) {
            $notification_count = $notif_count_result->fetch_assoc()['count'] ?? 0;
        }
        $notif_count_stmt->close();
    }
} catch (Exception $e) {
    $notification_count = 0;
}

// Get period parameter
$period = $_GET['period'] ?? 'last7';
$date_from = null;
$date_to = date('Y-m-d');

// Calculate date range based on period
switch ($period) {
    case 'today':
        $date_from = date('Y-m-d');
        break;
    case 'yesterday':
        $date_from = date('Y-m-d', strtotime('-1 day'));
        $date_to = $date_from;
        break;
    case 'yesterdaybefore':
        $date_from = date('Y-m-d', strtotime('-2 days'));
        $date_to = $date_from;
        break;
    case 'last3':
        $date_from = date('Y-m-d', strtotime('-3 days'));
        break;
    case 'week':
    case 'last7':
        $date_from = date('Y-m-d', strtotime('-7 days'));
        break;
    case 'month':
    case 'last30':
        $date_from = date('Y-m-d', strtotime('-30 days'));
        break;
    case 'thismonth':
        $date_from = date('Y-m-01');
        $date_to = date('Y-m-t');
        break;
    case 'lastmonth':
        $date_from = date('Y-m-01', strtotime('first day of last month'));
        $date_to = date('Y-m-t', strtotime('last day of last month'));
        break;
    default:
        $date_from = date('Y-m-d', strtotime('-7 days'));
}

// Get completion stats - count all assigned tasks in the period
$stats_stmt = $conn->prepare("
    SELECT 
        COUNT(DISTINCT ta.task_id) as total,
        COUNT(DISTINCT CASE 
            WHEN tc.status = 'completed' 
             AND tc.completion_date >= ? 
             AND tc.completion_date <= ? 
            THEN ta.task_id END) as completed,
        COUNT(DISTINCT CASE 
            WHEN tc.status = 'failed' 
             AND tc.completion_date >= ? 
             AND tc.completion_date <= ? 
            THEN ta.task_id END) as failed,
        COUNT(DISTINCT CASE 
            WHEN (tc.id IS NULL 
              OR tc.status NOT IN ('completed', 'failed') 
              OR tc.completion_date < ? 
              OR tc.completion_date > ?) 
            THEN ta.task_id END) as pending
    FROM task_assignments ta
    JOIN tasks t ON ta.task_id = t.id
    LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
        AND ta.user_id = tc.user_id
    WHERE ta.user_id = ? 
      AND t.start_date <= ?
      AND (t.end_date IS NULL OR t.end_date >= ?)
      AND t.is_enabled = 1
");
$stats_stmt->bind_param("ssssssiss", $date_from, $date_to, $date_from, $date_to, $date_from, $date_to, $user_id, $date_to, $date_from);
$stats_stmt->execute();
$stats_result = $stats_stmt->get_result();
$stats = $stats_result->fetch_assoc() ?: ['total' => 0, 'completed' => 0, 'failed' => 0, 'pending' => 0];

$completion_percentage = $stats['total'] > 0 ? round(($stats['completed'] / $stats['total']) * 100, 1) : 0;
$failed_percentage = $stats['total'] > 0 ? round(($stats['failed'] / $stats['total']) * 100, 1) : 0;

// Get points - from all completions in the period
$points_stmt = $conn->prepare("
    SELECT 
        COALESCE(SUM(CASE WHEN tc.completion_date >= ? AND tc.completion_date <= ? THEN tc.task_points_earned ELSE 0 END), 0) as task_points,
        COALESCE(SUM(CASE WHEN tc.completion_date >= ? AND tc.completion_date <= ? THEN tc.quality_points_earned ELSE 0 END), 0) as quality_points,
        COALESCE(SUM(CASE WHEN tc.completion_date >= ? AND tc.completion_date <= ? THEN tc.lost_task_points ELSE 0 END), 0) as lost_task_points
    FROM task_completions tc
    WHERE tc.user_id = ?
");
$points_stmt->bind_param("ssssssi", $date_from, $date_to, $date_from, $date_to, $date_from, $date_to, $user_id);
$points_stmt->execute();
$points_result = $points_stmt->get_result();
$points = $points_result->fetch_assoc() ?: ['task_points' => 0, 'quality_points' => 0, 'lost_task_points' => 0];

$net_points = $points['task_points'] + $points['quality_points'] + $points['lost_task_points'];

// Total possible points for the period (assigned tasks)
$totals_stmt = $conn->prepare("
    SELECT 
        COALESCE(SUM(t.task_points), 0) as task_points_total,
        COALESCE(SUM(t.quality_points), 0) as quality_points_total
    FROM task_assignments ta
    JOIN tasks t ON ta.task_id = t.id
    WHERE ta.user_id = ?
      AND t.start_date <= ?
      AND (t.end_date IS NULL OR t.end_date >= ?)
      AND t.is_enabled = 1
");
$totals_stmt->bind_param("iss", $user_id, $date_to, $date_from);
$totals_stmt->execute();
$totals_result = $totals_stmt->get_result();
$totals = $totals_result->fetch_assoc() ?: ['task_points_total' => 0, 'quality_points_total' => 0];

$quality_progress = $totals['quality_points_total'] > 0 
    ? round(($points['quality_points'] / $totals['quality_points_total']) * 100, 1) 
    : 0;

$ring_radius = 18;
$ring_circumference = 2 * M_PI * $ring_radius;
$completion_dashoffset = $ring_circumference - (($completion_percentage / 100) * $ring_circumference);
$failed_dashoffset = $ring_circumference - (($failed_percentage / 100) * $ring_circumference);

// Points history
$history_stmt = $conn->prepare("
    SELECT 
        t.name as task_name,
        tc.status,
        tc.completion_date,
        tc.task_points_earned,
        tc.quality_points_earned,
        tc.lost_task_points
    FROM task_completions tc
    JOIN tasks t ON tc.task_id = t.id
    WHERE tc.user_id = ?
      AND tc.completion_date >= ?
      AND tc.completion_date <= ?
    ORDER BY tc.completion_date DESC, tc.completed_at DESC
    LIMIT 20
");
$history_stmt->bind_param("iss", $user_id, $date_from, $date_to);
$history_stmt->execute();
$history_result = $history_stmt->get_result();
$points_history = $history_result ? $history_result->fetch_all(MYSQLI_ASSOC) : [];

$nav_titles = [
    'user-dashboard.php' => 'Dashboard',
    'user-tasks.php' => 'My Tasks',
    'user-progress.php' => 'Progress',
    'user-history.php' => 'History'
];
$current_page = basename($_SERVER['PHP_SELF'] ?? '');
$nav_title = $nav_titles[$current_page] ?? 'Task Manager';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Progress - Task Management System</title>
    <link rel="stylesheet" href="../styles.css">
    <style>
        :root {
            color-scheme: light;
        }
        * {
            box-sizing: border-box;
        }
        body {
            margin: 0;
            font-family: -apple-system, BlinkMacSystemFont, "SF Pro Text", "Segoe UI", sans-serif;
            background: linear-gradient(180deg, #f2f6ff 0%, #f9fbff 55%, #ffffff 100%);
            color: #0f172a;
            min-height: 100vh;
        }
        .ios-shell {
            max-width: 1080px;
            margin: 0 auto;
            padding: 12px 0 96px;
        }
        .top-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 16px;
            margin-bottom: 10px;
        }
        .title-block h1 {
            margin: 0;
            font-size: 24px;
            font-weight: 700;
        }
        .title-block p {
            margin: 2px 0 0;
            color: #64748b;
            font-size: 13px;
        }
        .pill-row {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
            background: #2131B0;
            padding: 6px;
            border-radius: 20px;
            box-shadow: 0 6px 24px rgba(15, 23, 42, 0.08);
        }
        .pill-row a {
            text-decoration: none;
            color: #e2e8f0;
            font-size: 12px;
            padding: 6px 12px;
            border-radius: 999px;
            transition: 0.2s ease;
        }
        .pill-row a.active {
            background: #ffffff;
            color: #2131B0;
            box-shadow: 0 6px 16px rgba(255, 255, 255, 0.35);
        }
        .ios-grid {
            display: grid;
            gap: 16px;
        }
        .ios-card {
            background: rgba(255, 255, 255, 0.92);
            border-radius: 22px;
            padding: 18px;
            box-shadow: 0 12px 40px rgba(15, 23, 42, 0.1);
            border: 1px solid rgba(148, 163, 184, 0.16);
        }
        .metric-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(170px, 1fr));
            gap: 12px;
        }
        .metric-card {
            background: linear-gradient(135deg, #ffffff, #f4f8ff);
            border-radius: 18px;
            padding: 14px;
            min-height: 88px;
            display: flex;
            flex-direction: column;
            gap: 6px;
        }
        .metric-label {
            font-size: 12px;
            color: #64748b;
        }
        .metric-value {
            font-size: 22px;
            font-weight: 700;
            color: #0f172a;
        }
        .metric-sub {
            font-size: 11px;
            color: #94a3b8;
        }
        @media (max-width: 640px) {
            .metric-row {
                grid-template-columns: repeat(2, minmax(0, 1fr));
                gap: 10px;
            }
            .metric-card {
                padding: 10px 12px;
                min-height: 72px;
            }
            .metric-label {
                font-size: 11px;
            }
            .metric-value {
                font-size: 18px;
            }
            .metric-sub {
                font-size: 10px;
            }
        }
        .progress-mini {
            display: flex;
            align-items: center;
            gap: 14px;
        }
        .ring {
            width: 52px;
            height: 52px;
            position: relative;
        }
        .ring svg {
            transform: rotate(-90deg);
        }
        .ring text {
            font-size: 10px;
            fill: #0f172a;
            font-weight: 600;
        }
        .progress-row {
            display: grid;
            gap: 14px;
        }
        .progress-label {
            display: flex;
            justify-content: space-between;
            font-size: 12px;
            color: #475569;
            margin-bottom: 6px;
        }
        .progress-track {
            height: 8px;
            border-radius: 999px;
            background: #e2e8f0;
            overflow: hidden;
        }
        .progress-fill {
            height: 100%;
            border-radius: 999px;
        }
        .fill-complete {
            background: linear-gradient(90deg, #22c55e, #86efac);
        }
        .fill-failed {
            background: linear-gradient(90deg, #ef4444, #fca5a5);
        }
        .fill-quality {
            background: linear-gradient(90deg, #6366f1, #a5b4fc);
        }
        .summary-list {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
            overflow-x: visible;
        }
        .summary-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 6px 10px;
            border-radius: 10px;
            background: #eef2ff;
            flex: 1 1 120px;
        }
        .summary-item span:first-child {
            color: #475569;
            font-size: 11px;
        }
        .summary-item span:last-child {
            font-weight: 700;
            font-size: 12px;
        }
        .history-list {
            display: grid;
            gap: 10px;
            max-height: 420px;
            overflow-y: auto;
            padding-right: 4px;
        }
        .history-item {
            display: grid;
            grid-template-columns: 1fr auto;
            gap: 8px;
            padding: 12px 14px;
            border-radius: 16px;
            background: #f1f5f9;
        }
        .history-title {
            font-weight: 600;
            font-size: 13px;
            margin-bottom: 4px;
        }
        .history-meta {
            font-size: 11px;
            color: #64748b;
        }
        .history-points {
            text-align: right;
            font-size: 12px;
            color: #0f172a;
            white-space: nowrap;
        }
        .history-badge {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 999px;
            font-size: 10px;
            margin-top: 6px;
        }
        .history-badge.completed {
            background: rgba(34, 197, 94, 0.14);
            color: #15803d;
        }
        .history-badge.failed {
            background: rgba(239, 68, 68, 0.14);
            color: #b91c1c;
        }
        .negative {
            color: #dc2626;
        }
        @media (min-width: 900px) {
            .ios-grid.dual {
                grid-template-columns: 1.4fr 1fr;
            }
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar">
        <div class="nav-left">
            <h2 class="nav-title"><?php echo htmlspecialchars($nav_title, ENT_QUOTES, 'UTF-8'); ?></h2>
        </div>
        <div class="nav-right">
            <button id="notificationBtn" class="nav-icon-btn" title="Notifications" style="display: <?php echo $notification_count > 0 ? 'flex' : 'none'; ?>;">
                <span class="icon">🔔</span>
                <?php if ($notification_count > 0): ?>
                <span id="notificationBadge" class="badge"><?php echo $notification_count; ?></span>
                <?php endif; ?>
            </button>
            <button id="themeToggle" class="nav-icon-btn" title="Toggle Theme">
                <span class="icon" id="themeIcon">🌙</span>
            </button>
            <div class="user-menu">
                <span id="userNameDisplay"><?php echo htmlspecialchars($username); ?></span>
                <a href="../logout.php" class="btn btn-small">Logout</a>
            </div>
        </div>
    </nav>

    <!-- Notification Dropdown -->
    <div id="notificationDropdown" class="notification-dropdown hidden">
        <div class="notification-header">
            <h3>Notifications</h3>
            <button id="clearNotifications" class="btn btn-small">Clear All</button>
        </div>
        <div id="notificationList" class="notification-list"></div>
    </div>

    <!-- Main Content -->
    <div class="main-container">
        <!-- Sidebar (Desktop) -->
        <aside class="sidebar">
            <nav class="sidebar-nav">
                <a href="user-dashboard.php" class="nav-link">
                    <img class="nav-icon" src="../icon/dashboard_icon.png" alt="Dashboard">
                    <span>Dashboard</span>
                </a>
                <a href="user-tasks.php" class="nav-link">
                    <img class="nav-icon" src="../icon/mytask.png" alt="My Tasks">
                    <span>My Tasks</span>
                </a>
                <a href="user-progress.php" class="nav-link active">
                    <img class="nav-icon" src="../icon/progress.png" alt="Progress">
                    <span>Progress</span>
                </a>
                <a href="user-history.php" class="nav-link">
                    <img class="nav-icon" src="../icon/history-icon.png" alt="History">
                    <span>History</span>
                </a>
            </nav>
        </aside>

        <!-- Mobile Bottom Nav -->
        <nav class="bottom-nav">
            <a href="user-dashboard.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/dashboard_icon.png" alt="Dashboard">
                <span>Dashboard</span>
            </a>
            <a href="user-tasks.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/mytask.png" alt="Tasks">
                <span>Tasks</span>
            </a>
            <a href="user-progress.php" class="bottom-nav-link active">
                <img class="nav-icon" src="../icon/progress.png" alt="Progress">
                <span>Progress</span>
            </a>
            <a href="user-history.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/history-icon.png" alt="History">
                <span>History</span>
            </a>
        </nav>

        <!-- Content Area -->
        <main class="content">
            <div class="page active">
                <div class="ios-shell">
                    <div class="top-bar">
                        <div class="title-block">
                            <p><?php echo htmlspecialchars($full_name); ?> · <?php echo date('M d', strtotime($date_from)); ?> - <?php echo date('M d, Y', strtotime($date_to)); ?></p>
                        </div>
                    </div>

                    <div class="pill-row">
                        <a href="?period=yesterday" class="<?php echo $period === 'yesterday' ? 'active' : ''; ?>">Yesterday</a>
                        <a href="?period=yesterdaybefore" class="<?php echo $period === 'yesterdaybefore' ? 'active' : ''; ?>">Yesterday Before</a>
                        <a href="?period=last3" class="<?php echo $period === 'last3' ? 'active' : ''; ?>">Last 3 Days</a>
                        <a href="?period=last7" class="<?php echo $period === 'last7' || $period === 'week' ? 'active' : ''; ?>">Last 7 Days</a>
                        <a href="?period=last30" class="<?php echo $period === 'last30' || $period === 'month' ? 'active' : ''; ?>">Last 30 Days</a>
                        <a href="?period=thismonth" class="<?php echo $period === 'thismonth' ? 'active' : ''; ?>">This Month</a>
                        <a href="?period=lastmonth" class="<?php echo $period === 'lastmonth' ? 'active' : ''; ?>">Last Month</a>
                    </div>

                    <div class="ios-grid" style="margin-top: 18px;">
                        <div class="ios-card">
                            <div class="metric-row">
                                <div class="metric-card">
                                    <span class="metric-label">Total Task Points</span>
                                    <span class="metric-value"><?php echo $points['task_points']; ?></span>
                                    <span class="metric-sub">out of <?php echo $totals['task_points_total']; ?></span>
                                </div>
                                <div class="metric-card">
                                    <span class="metric-label">Total Quality Points</span>
                                    <span class="metric-value"><?php echo $points['quality_points']; ?></span>
                                    <span class="metric-sub">out of <?php echo $totals['quality_points_total']; ?></span>
                                </div>
                                <div class="metric-card">
                                    <span class="metric-label">Negative Task Points</span>
                                    <span class="metric-value negative"><?php echo $points['lost_task_points']; ?></span>
                                    <span class="metric-sub">net: <?php echo $net_points; ?></span>
                                </div>
                                <div class="metric-card">
                                    <span class="metric-label">Total Tasks</span>
                                    <span class="metric-value"><?php echo $stats['total']; ?></span>
                                    <span class="metric-sub">completed + failed + pending</span>
                                </div>
                            </div>
                        </div>

                        <div class="ios-card ios-grid dual">
                        <div class="ios-card" style="box-shadow:none; border:none; background:transparent; padding:0;">
                            <div class="progress-mini">
                                <div class="ring">
                                    <svg width="52" height="52" viewBox="0 0 52 52">
                                        <circle cx="26" cy="26" r="<?php echo $ring_radius; ?>" stroke="#e2e8f0" stroke-width="6" fill="none"></circle>
                                        <circle cx="26" cy="26" r="<?php echo $ring_radius; ?>" stroke="#22c55e" stroke-width="6" fill="none"
                                                stroke-dasharray="<?php echo $ring_circumference; ?>" stroke-dashoffset="<?php echo $completion_dashoffset; ?>" stroke-linecap="round"></circle>
                                    </svg>
                                </div>
                                <div>
                                    <div class="metric-label">Complete %</div>
                                    <div class="metric-value"><?php echo $completion_percentage; ?>%</div>
                                </div>
                            </div>
                            <div class="progress-mini" style="margin-top: 16px;">
                                <div class="ring">
                                    <svg width="52" height="52" viewBox="0 0 52 52">
                                        <circle cx="26" cy="26" r="<?php echo $ring_radius; ?>" stroke="#e2e8f0" stroke-width="6" fill="none"></circle>
                                        <circle cx="26" cy="26" r="<?php echo $ring_radius; ?>" stroke="#ef4444" stroke-width="6" fill="none"
                                                stroke-dasharray="<?php echo $ring_circumference; ?>" stroke-dashoffset="<?php echo $failed_dashoffset; ?>" stroke-linecap="round"></circle>
                                    </svg>
                                </div>
                                <div>
                                    <div class="metric-label">Failed %</div>
                                    <div class="metric-value"><?php echo $failed_percentage; ?>%</div>
                                </div>
                            </div>
                        </div>
                        <div class="ios-card" style="box-shadow:none; border:none; background:transparent; padding:0;">
                            <div class="progress-row">
                                <div>
                                    <div class="progress-label">
                                        <span>Quality Progress</span>
                                        <span><?php echo $points['quality_points']; ?> / <?php echo $totals['quality_points_total']; ?></span>
                                    </div>
                                    <div class="progress-track">
                                        <div class="progress-fill fill-quality" style="width: <?php echo $quality_progress; ?>%"></div>
                                    </div>
                                </div>
                                <div>
                                    <div class="progress-label">
                                        <span>Failed Progress</span>
                                        <span><?php echo $stats['failed']; ?> / <?php echo $stats['total']; ?></span>
                                    </div>
                                    <div class="progress-track">
                                        <div class="progress-fill fill-failed" style="width: <?php echo $failed_percentage; ?>%"></div>
                                    </div>
                                </div>
                                <div>
                                    <div class="progress-label">
                                        <span>Completed Progress</span>
                                        <span><?php echo $stats['completed']; ?> / <?php echo $stats['total']; ?></span>
                                    </div>
                                    <div class="progress-track">
                                        <div class="progress-fill fill-complete" style="width: <?php echo $completion_percentage; ?>%"></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                        <div class="ios-card ios-grid dual">
                        <div class="ios-card" style="box-shadow:none; border:none; background:transparent; padding:0;">
                            <h3 style="margin:0 0 12px; font-size:15px;">Task Summary</h3>
                            <div class="summary-list">
                                <div class="summary-item">
                                    <span>Total Completed</span>
                                    <span><?php echo $stats['completed']; ?></span>
                                </div>
                                <div class="summary-item">
                                    <span>Total Failed</span>
                                    <span><?php echo $stats['failed']; ?></span>
                                </div>
                                <div class="summary-item">
                                    <span>Total Pending</span>
                                    <span><?php echo $stats['pending']; ?></span>
                                </div>
                                <div class="summary-item">
                                    <span>Total Tasks</span>
                                    <span><?php echo $stats['total']; ?></span>
                                </div>
                            </div>
                        </div>
                        <div class="ios-card" style="box-shadow:none; border:none; background:transparent; padding:0;">
                            <h3 style="margin:0 0 12px; font-size:15px;">Points History</h3>
                            <div class="history-list">
                                <?php if (count($points_history) === 0): ?>
                                    <div class="summary-item">
                                        <span>No history found</span>
                                        <span>—</span>
                                    </div>
                                <?php else: ?>
                                    <?php foreach ($points_history as $history): ?>
                                        <div class="history-item">
                                            <div>
                                                <div class="history-title"><?php echo htmlspecialchars($history['task_name']); ?></div>
                                                <div class="history-meta"><?php echo date('M d, Y', strtotime($history['completion_date'])); ?></div>
                                                <span class="history-badge <?php echo $history['status'] === 'failed' ? 'failed' : 'completed'; ?>">
                                                    <?php echo ucfirst($history['status']); ?>
                                                </span>
                                            </div>
                                            <div class="history-points">
                                                <div>Task: <?php echo $history['task_points_earned']; ?></div>
                                                <div>Quality: <?php echo $history['quality_points_earned']; ?></div>
                                                <?php if (!empty($history['lost_task_points'])): ?>
                                                    <div class="negative">Lost: <?php echo $history['lost_task_points']; ?></div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    </div>
                </div>
            </div>
        </main>
    </div>


    <script>
        // Set timezone to Asia/Kolkata for all JavaScript date operations
        const KOLKATA_TIMEZONE = 'Asia/Kolkata';
        const KOLKATA_OFFSET = 5.5 * 60 * 60 * 1000;
        
        function getKolkataTime() {
            const now = new Date();
            const istString = now.toLocaleString('en-US', {
                timeZone: 'Asia/Kolkata',
                year: 'numeric',
                month: '2-digit',
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit',
                hour12: false
            });
            const parts = istString.match(/(\d+)\/(\d+)\/(\d+),\s+(\d+):(\d+):(\d+)/);
            if (parts) {
                const month = parts[1];
                const day = parts[2];
                const year = parts[3];
                const hour = parts[4];
                const minute = parts[5];
                const second = parts[6];
                const isoString = `${year}-${month.padStart(2, '0')}-${day.padStart(2, '0')}T${hour.padStart(2, '0')}:${minute.padStart(2, '0')}:${second.padStart(2, '0')}+05:30`;
                return new Date(isoString);
            }
            return now;
        }
        
        function createKolkataDate(dateString, timeString = null) {
            if (!timeString) {
                timeString = '23:59:59';
            }
            const timeParts = timeString.split(':');
            if (timeParts.length === 2) {
                timeString = timeString + ':00';
            }
            const isoString = `${dateString}T${timeString}+05:30`;
            return new Date(isoString);
        }
        
        const CURRENT_USER = {
            id: <?php echo $user_id; ?>,
            username: '<?php echo htmlspecialchars($username, ENT_QUOTES); ?>',
            role: '<?php echo htmlspecialchars($user_role, ENT_QUOTES); ?>'
        };
        const IS_LOGGED_IN = true;
    </script>
    <script src="../app-minimal.js"></script>
</body>
</html>
