<?php
require_once '../config.php';

if (!isLoggedIn()) {
    header('Location: ../index.php');
    exit;
}

$conn = getDBConnection();
if (!$conn) {
    die("Database connection failed. Please check your configuration.");
}

$user_id = getCurrentUserId();
$username = getCurrentUsername();
$full_name = $_SESSION['full_name'] ?? $username;
$user_role = getCurrentUserRole();

// Get notification count
$notification_count = 0;
try {
    $notif_count_stmt = $conn->prepare("SELECT COUNT(*) as count FROM notifications WHERE user_id = ? AND is_read = 0");
    if ($notif_count_stmt) {
        $notif_count_stmt->bind_param("i", $user_id);
        $notif_count_stmt->execute();
        $notif_count_result = $notif_count_stmt->get_result();
        if ($notif_count_result) {
            $notification_count = $notif_count_result->fetch_assoc()['count'] ?? 0;
        }
        $notif_count_stmt->close();
    }
} catch (Exception $e) {
    $notification_count = 0;
}

$today = date('Y-m-d');

// Get current time in Kolkata timezone for accurate time comparisons
$now_kolkata = new DateTime('now', new DateTimeZone('Asia/Kolkata'));
$current_time = $now_kolkata->format('H:i:s');
$current_date = $now_kolkata->format('Y-m-d');

// Get filter parameters - default to today's all tasks
$status = $_GET['status'] ?? 'all';
$date = $_GET['date'] ?? $today; // Default to today
$priority = $_GET['priority'] ?? 'all';

// Build query - Show selected date tasks
// For recurring tasks, join with instances table for the selected date
$query = "
    SELECT DISTINCT 
           t.*,
           rti.instance_date as effective_date,
           COALESCE(rti.start_time, t.start_time) as effective_start_time,
           COALESCE(rti.end_time, t.end_time) as effective_end_time,
           rti.status as instance_status,
           tc.status as completion_status,
           tc.completion_date as completion_date,
           tc.completed_at as completed_at
    FROM task_assignments ta
    JOIN tasks t ON ta.task_id = t.id
    LEFT JOIN recurring_task_instances rti ON t.id = rti.parent_task_id 
        AND rti.instance_date = ?
        AND t.task_type = 'recurring'
    LEFT JOIN task_completions tc ON ta.task_id = tc.task_id 
        AND ta.user_id = tc.user_id 
        AND tc.completion_date = ?
    WHERE ta.user_id = ? 
      AND t.is_enabled = 1
      AND t.start_date <= ? 
      AND (t.end_date IS NULL OR t.end_date >= ?)
";

// Parameters: date (rti), date (completion), user_id, date (start range), date (end range)
$params = [$date, $date, $user_id, $date, $date];
$types = 'ssiss';

// Priority filter
if ($priority !== 'all') {
    $query .= " AND t.priority = ?";
    $params[] = $priority;
    $types .= 's';
}

$query .= " ORDER BY t.priority DESC, COALESCE(rti.instance_date, t.start_date) ASC";

$stmt = $conn->prepare($query);
if (!$stmt) {
    error_log("Query preparation failed: " . $conn->error);
    error_log("Query: " . $query);
    die("Error preparing query: " . htmlspecialchars($conn->error));
}
if (!empty($params)) {
    if (!$stmt->bind_param($types, ...$params)) {
        error_log("Bind param failed: " . $stmt->error);
        die("Error binding parameters: " . htmlspecialchars($stmt->error));
    }
}
if (!$stmt->execute()) {
    error_log("Query execution failed: " . $stmt->error);
    error_log("Query: " . $query);
    die("Error executing query: " . htmlspecialchars($stmt->error));
}
$result = $stmt->get_result();

$tasks = [];
$all_tasks = []; // Store all tasks before filtering
// Get user's leave dates for the selected day only
$leave_dates = getUserLeaveDates($user_id, $date, $date);
$is_on_leave_selected_date = in_array($date, $leave_dates, true);

// Get selected date's day of week for recurring filtering (0 = Sunday, 1 = Monday, ..., 6 = Saturday)
$selected_date_obj = new DateTime($date, new DateTimeZone('Asia/Kolkata'));
$selected_day_of_week = (int)$selected_date_obj->format('w');

while ($row = $result->fetch_assoc()) {
    if ($is_on_leave_selected_date) {
        continue;
    }

    // For recurring tasks, check if selected date is a recurring day
    if ($row['task_type'] === 'recurring') {
        $recurring_days = $row['skip_days'] ? explode(',', $row['skip_days']) : [];
        if (empty($recurring_days) || !in_array((string)$selected_day_of_week, $recurring_days, true)) {
            continue;
        }

        // If no instance for selected date, use selected date/time for display
        if (empty($row['effective_date']) || $row['effective_date'] !== $date) {
            $row['effective_date'] = $date;
            $row['effective_start_time'] = $row['start_time'];
            $row['effective_end_time'] = $row['end_time'];
        }
    }

    // For one-time tasks, verify the selected date is within the task's date range
    if ($row['task_type'] === 'one-time') {
        $task_start = $row['start_date'];
        $task_end = $row['end_date'] ?? $task_start;
        if ($date < $task_start || ($task_end && $date > $task_end)) {
            continue;
        }
        // For one-time tasks, use the selected date as effective_date for display
        $row['effective_date'] = $date;
    }

    // Compute task status aligned with dashboard/api logic
    if (!empty($row['completion_status'])) {
        $row['status'] = $row['completion_status'];
    } elseif (!empty($row['instance_status'])) {
        $row['status'] = $row['instance_status'];
    } else {
        $row['status'] = 'pending';
    }

    // Store all tasks for counts (after date + recurring checks)
    $all_tasks[] = $row;

    // Filter by status if needed (if status is 'all', show all statuses)
    if ($status === 'all' || $row['status'] === $status) {
        $tasks[] = $row;
    }
}

// Sort tasks: pending first, then completed, then failed
if (!empty($tasks)) {
    usort($tasks, function($a, $b) {
        $order = ['pending' => 1, 'completed' => 2, 'failed' => 3];
        $a_status = $a['status'] ?? 'pending';
        $b_status = $b['status'] ?? 'pending';
        $a_order = $order[$a_status] ?? 4;
        $b_order = $order[$b_status] ?? 4;
        
        if ($a_order !== $b_order) {
            return $a_order - $b_order;
        }
        
        // If same status, sort by priority (high to low)
        $priority_order = ['high' => 3, 'medium' => 2, 'low' => 1];
        $a_priority = $priority_order[$a['priority'] ?? 'low'] ?? 0;
        $b_priority = $priority_order[$b['priority'] ?? 'low'] ?? 0;
        
        return $b_priority - $a_priority;
    });
}

// Calculate counts for display (excluding leave dates, only for selected date)
$status_counts = [
    'pending' => 0,
    'completed' => 0,
    'failed' => 0
];

foreach ($all_tasks as $task) {
    $task_status = $task['status'] ?? 'pending';
    if (isset($status_counts[$task_status])) {
        $status_counts[$task_status]++;
    }
}

function formatDate($dateString) {
    return date('M d, Y', strtotime($dateString));
}

function escapeHtml($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}

$nav_titles = [
    'user-dashboard.php' => 'Dashboard',
    'user-tasks.php' => 'My Tasks',
    'user-progress.php' => 'Progress',
    'user-history.php' => 'History'
];
$current_page = basename($_SERVER['PHP_SELF'] ?? '');
$nav_title = $nav_titles[$current_page] ?? 'Task Manager';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Tasks - Task Management System</title>
    <link rel="stylesheet" href="../styles.css">
</head>
<body class="ios-app">
    <style>
        :root {
            color-scheme: light;
        }
        .ios-app {
            font-family: -apple-system, BlinkMacSystemFont, "SF Pro Text", "SF Pro Display", "Segoe UI", Roboto, Helvetica, Arial, sans-serif;
            background: radial-gradient(circle at top left, #f1f7ff 0%, #f7f7fb 45%, #ffffff 100%);
            color: #0b0b0f;
        }
        .ios-app .navbar {
            background: rgba(255, 255, 255, 0.9);
            backdrop-filter: blur(16px);
            border-bottom: 1px solid rgba(15, 23, 42, 0.08);
            box-shadow: 0 8px 24px rgba(15, 23, 42, 0.08);
        }
        .ios-app .nav-title {
            font-weight: 700;
            letter-spacing: 0.2px;
        }
        .ios-app .nav-icon-btn {
            border-radius: 12px;
            background: linear-gradient(135deg, #eef4ff, #ffffff);
            box-shadow: 0 6px 16px rgba(59, 130, 246, 0.15);
        }
        .ios-app .badge {
            background: linear-gradient(135deg, #ff5e5e, #ff8b5e);
            color: #fff;
        }
        .ios-app .main-container {
            background: transparent;
        }
        .ios-app .sidebar {
            background: rgba(255, 255, 255, 0.88);
            backdrop-filter: blur(12px);
            border-right: 1px solid rgba(15, 23, 42, 0.08);
        }
        .ios-app .sidebar .nav-link {
            border-radius: 14px;
            padding: 10px 12px;
            font-weight: 600;
            color: #1f2937;
        }
        .ios-app .sidebar .nav-link.active {
            background: linear-gradient(135deg, #d6e4ff, #f0f5ff);
            color: #1d4ed8;
            box-shadow: 0 8px 18px rgba(29, 78, 216, 0.15);
        }
        .ios-app .bottom-nav {
            background: rgba(255, 255, 255, 0.92);
            backdrop-filter: blur(18px);
            border-top: 1px solid rgba(15, 23, 42, 0.08);
            box-shadow: 0 -10px 24px rgba(15, 23, 42, 0.08);
        }
        .ios-app .bottom-nav-link {
            font-weight: 600;
            color: #334155;
        }
        .ios-app .bottom-nav-link.active {
            color: #2563eb;
        }
        .ios-app .content {
            padding: 16px 16px 128px;
        }
        .ios-app .page-header h1 {
            font-size: 28px;
            font-weight: 800;
            letter-spacing: -0.4px;
        }
        .ios-app .task-counts {
            color: #64748b !important;
        }
        .ios-app .task-filters {
            background: #2131B0;
            border-radius: 18px;
            padding: 12px;
            box-shadow: 0 10px 30px rgba(15, 23, 42, 0.08);
            border: 1px solid rgba(15, 23, 42, 0.2);
            gap: 10px;
        }
        .ios-app .filter-group label {
            color: #dbeafe;
            font-weight: 600;
            letter-spacing: 0.2px;
        }
        .ios-app .filter-select,
        .ios-app .filter-date {
            border-radius: 12px;
            border: 1px solid rgba(148, 163, 184, 0.4);
            padding: 8px 10px;
            background: #f8fafc;
            box-shadow: inset 0 0 0 1px rgba(226, 232, 240, 0.3);
        }
        @media (max-width: 768px) {
            .ios-app .task-filters {
                padding: 10px;
                gap: 8px;
                display: grid;
                grid-template-columns: repeat(2, minmax(0, 1fr));
            }
            .ios-app .task-filters .filter-group {
                min-width: 0;
            }
            .ios-app .task-filters label {
                font-size: 12px;
            }
            .ios-app .filter-select,
            .ios-app .filter-date {
                font-size: 12px;
                padding: 6px 8px;
            }
            .ios-app .task-filters .btn {
                width: 100%;
                font-size: 12px;
                padding: 6px 10px;
            }
        }
        .ios-app .btn,
        .ios-app .btn.btn-small {
            border-radius: 12px;
            padding: 8px 14px;
            font-weight: 600;
        }
        .ios-app .btn-primary {
            background: linear-gradient(135deg, #3b82f6, #2563eb);
            border: none;
            box-shadow: 0 8px 16px rgba(37, 99, 235, 0.25);
        }
        .ios-app .btn-secondary {
            background: #f1f5f9;
            color: #1f2937;
            border: none;
        }
        .ios-app .btn-danger {
            background: linear-gradient(135deg, #f87171, #ef4444);
            border: none;
        }
        .ios-app .tasks-list {
            margin-top: 12px;
        }
        .ios-app .task-card {
            border-radius: 22px;
            border: 1px solid rgba(148, 163, 184, 0.2);
            background: rgba(255, 255, 255, 0.95);
            box-shadow: 0 14px 32px rgba(15, 23, 42, 0.08);
            padding: 14px 14px 12px;
        }
        .ios-app .task-card.completed {
            background: #d1fae5;
            border-color: rgba(34, 197, 94, 0.45);
        }
        .ios-app .task-title .completed-icon {
            margin-right: 6px;
        }
        .ios-app .task-card.high-priority {
            border-color: rgba(34, 197, 94, 0.25);
            box-shadow: 0 14px 32px rgba(15, 23, 42, 0.08);
        }
        .ios-app .task-card.medium-priority {
            border-color: rgba(34, 197, 94, 0.22);
        }
        .ios-app .task-card.low-priority {
            border-color: rgba(34, 197, 94, 0.18);
        }
        .ios-app .task-title {
            font-size: 18px;
            font-weight: 700;
            letter-spacing: -0.2px;
        }
        .ios-app .task-meta {
            color: #64748b;
            font-size: 13px;
            display: flex;
            flex-wrap: wrap;
            gap: 8px 12px;
        }
        .ios-app .task-meta-item {
            background: #f8fafc;
            border-radius: 10px;
            padding: 4px 8px;
        }
        .ios-app .task-priority {
            border-radius: 999px;
            padding: 6px 12px;
            font-weight: 700;
        }
        .ios-app .task-priority.high {
            background: rgba(34, 197, 94, 0.12);
            color: #15803d;
        }
        .ios-app .task-priority.medium {
            background: rgba(34, 197, 94, 0.12);
            color: #15803d;
        }
        .ios-app .task-priority.low {
            background: rgba(34, 197, 94, 0.12);
            color: #15803d;
        }
        .ios-app .task-status {
            border-radius: 999px;
            padding: 4px 10px;
            font-weight: 700;
            font-size: 12px;
        }
        .ios-app .task-status.completed {
            background: rgba(34, 197, 94, 0.14);
            color: #15803d;
        }
        .ios-app .task-status.failed {
            background: rgba(239, 68, 68, 0.14);
            color: #dc2626;
        }
        .ios-app .task-status.pending {
            background: rgba(59, 130, 246, 0.12);
            color: #1d4ed8;
        }
        .ios-app .countdown-timer {
            border-radius: 14px;
            background: #f1f5ff;
            padding: 8px 10px;
        }
        .ios-app .task-description {
            color: #475569;
            margin-top: 8px;
        }
        .ios-app .task-actions .btn {
            box-shadow: 0 8px 16px rgba(15, 23, 42, 0.08);
        }
        .ios-app .empty-state {
            background: rgba(255, 255, 255, 0.9);
            border-radius: 22px;
            padding: 18px;
            border: 1px dashed rgba(148, 163, 184, 0.3);
        }
        .ios-app .ios-modal-content {
            border-radius: 22px;
            border: 1px solid rgba(148, 163, 184, 0.2);
            box-shadow: 0 20px 40px rgba(15, 23, 42, 0.2);
        }
        .ios-app .modal-header {
            background: #f8fafc;
            border-top-left-radius: 22px;
            border-top-right-radius: 22px;
        }
        .ios-app .modal-content {
            border-radius: 22px;
            border: 1px solid rgba(148, 163, 184, 0.2);
        }
    </style>
    <!-- Navbar -->
    <nav class="navbar">
        <div class="nav-left">
            <h2 class="nav-title"><?php echo htmlspecialchars($nav_title, ENT_QUOTES, 'UTF-8'); ?></h2>
        </div>
        <div class="nav-right">
            <button id="notificationBtn" class="nav-icon-btn" title="Notifications" style="display: <?php echo $notification_count > 0 ? 'flex' : 'none'; ?>;">
                <span class="icon">🔔</span>
                <?php if ($notification_count > 0): ?>
                <span id="notificationBadge" class="badge"><?php echo $notification_count; ?></span>
                <?php endif; ?>
            </button>
            <button id="themeToggle" class="nav-icon-btn" title="Toggle Theme">
                <span class="icon" id="themeIcon">🌙</span>
            </button>
            <div class="user-menu">
                <span id="userNameDisplay"><?php echo htmlspecialchars($username); ?></span>
                <a href="../logout.php" class="btn btn-small">Logout</a>
            </div>
        </div>
    </nav>

    <!-- Notification Dropdown -->
    <div id="notificationDropdown" class="notification-dropdown hidden">
        <div class="notification-header">
            <h3>Notifications</h3>
            <button id="clearNotifications" class="btn btn-small">Clear All</button>
        </div>
        <div id="notificationList" class="notification-list"></div>
    </div>

    <!-- Main Content -->
    <div class="main-container">
        <!-- Sidebar (Desktop) -->
        <aside class="sidebar">
            <nav class="sidebar-nav">
                <a href="user-dashboard.php" class="nav-link">
                    <img class="nav-icon" src="../icon/dashboard_icon.png" alt="Dashboard">
                    <span>Dashboard</span>
                </a>
                <a href="user-tasks.php" class="nav-link active">
                    <img class="nav-icon" src="../icon/mytask.png" alt="My Tasks">
                    <span>My Tasks</span>
                </a>
                <a href="user-progress.php" class="nav-link">
                    <img class="nav-icon" src="../icon/progress.png" alt="Progress">
                    <span>Progress</span>
                </a>
                <a href="user-history.php" class="nav-link">
                    <img class="nav-icon" src="../icon/history-icon.png" alt="History">
                    <span>History</span>
                </a>
            </nav>
        </aside>

        <!-- Mobile Bottom Nav -->
        <nav class="bottom-nav">
            <a href="user-dashboard.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/dashboard_icon.png" alt="Dashboard">
                <span>Dashboard</span>
            </a>
            <a href="user-tasks.php" class="bottom-nav-link active">
                <img class="nav-icon" src="../icon/mytask.png" alt="Tasks">
                <span>Tasks</span>
            </a>
            <a href="user-progress.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/progress.png" alt="Progress">
                <span>Progress</span>
            </a>
            <a href="user-history.php" class="bottom-nav-link">
                <img class="nav-icon" src="../icon/history-icon.png" alt="History">
                <span>History</span>
            </a>
        </nav>

        <!-- Content Area -->
        <main class="content">
            <div class="page active">
                <div class="page-header">
                    <div class="task-counts" style="margin-top: 10px; font-size: 14px; color: #666;">
                        <span>Pending: <?php echo $status_counts['pending']; ?></span> | 
                        <span>Completed: <?php echo $status_counts['completed']; ?></span> | 
                        <span>Failed: <?php echo $status_counts['failed']; ?></span>
                    </div>
                </div>

                <!-- Tasks List -->
                <div class="pending-tasks-section">
                    <form id="tasksFilterForm" method="GET" action="" class="task-filters">
                        <div class="filter-group">
                            <label for="statusFilter">Status</label>
                            <select name="status" id="statusFilter" class="filter-select">
                                <option value="all" <?php echo ($status === 'all' || $status === '') ? 'selected' : ''; ?>>All</option>
                                <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="completed" <?php echo $status === 'completed' ? 'selected' : ''; ?>>Completed</option>
                                <option value="failed" <?php echo $status === 'failed' ? 'selected' : ''; ?>>Failed</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="dateFilter">Date</label>
                            <input type="date" name="date" id="dateFilter" value="<?php echo htmlspecialchars($date); ?>" class="filter-date">
                        </div>
                        <div class="filter-group">
                            <label for="priorityFilter">Priority</label>
                            <select name="priority" id="priorityFilter" class="filter-select">
                                <option value="all" <?php echo ($priority === 'all' || $priority === '') ? 'selected' : ''; ?>>All</option>
                                <option value="high" <?php echo $priority === 'high' ? 'selected' : ''; ?>>High</option>
                                <option value="medium" <?php echo $priority === 'medium' ? 'selected' : ''; ?>>Medium</option>
                                <option value="low" <?php echo $priority === 'low' ? 'selected' : ''; ?>>Low</option>
                            </select>
                        </div>
                        <div class="filter-group" style="align-self: flex-end;">
                            <button type="submit" class="btn btn-primary btn-small" style="margin-top: 0;">Apply</button>
                        </div>
                    </form>
                    <div class="tasks-list">
                        <?php if (empty($tasks)): ?>
                            <div class="empty-state">
                                <div class="empty-state-icon">📝</div>
                                <h3>No tasks found</h3>
                                <p>No tasks match your filters.</p>
                            </div>
                        <?php else: ?>
                            <?php foreach ($tasks as $task): ?>
                                <?php
                                // For recurring tasks, use instance date; for one-time, use task dates
                                $effectiveDate = !empty($task['effective_date']) ? $task['effective_date'] : $task['start_date'];
                                $effectiveEndDate = !empty($task['effective_date']) ? $task['effective_date'] : $task['end_date'];
                                $effectiveStartTime = !empty($task['effective_start_time']) ? $task['effective_start_time'] : $task['start_time'];
                                $effectiveEndTime = !empty($task['effective_end_time']) ? $task['effective_end_time'] : $task['end_time'];
                                
                                $startDate = new DateTime($effectiveDate, new DateTimeZone('Asia/Kolkata'));
                                $endDate = $effectiveEndDate ? new DateTime($effectiveEndDate, new DateTimeZone('Asia/Kolkata')) : null;
                                $dueTime = $endDate ? $endDate->getTimestamp() * 1000 : $startDate->getTimestamp() * 1000;
                                
                                $typeLabel = $task['task_type'] === 'recurring' ? 'Recurring' : 'One-Time';
                                $lostTaskPointsDisplay = $task['lost_task_points'] < 0 ? $task['lost_task_points'] : '';
                                $status_value = $task['status'] ?? 'pending';
                                $status_class = in_array($status_value, ['pending', 'completed', 'failed'], true)
                                    ? 'status-' . $status_value
                                    : 'status-pending';
                                $completedAtDisplay = '';
                                if (!empty($task['completed_at'])) {
                                    $completedAtDisplay = date('g:i A', strtotime($task['completed_at']));
                                }
                                ?>
                                <div class="task-card <?php echo $task['priority']; ?>-priority <?php echo $task['status'] === 'completed' ? 'completed' : ''; ?> <?php echo $task['status'] === 'failed' ? 'failed' : ''; ?> <?php echo $status_class; ?>" 
                                     data-task-id="<?php echo $task['id']; ?>"
                                     data-task-type="<?php echo $task['task_type']; ?>"
                                     data-end-date="<?php echo $effectiveEndDate; ?>"
                                     data-end-time="<?php echo $effectiveEndTime ?? ''; ?>"
                                     data-task-end-date="<?php echo $task['end_date'] ?? ''; ?>"
                                     data-skip-days="<?php echo $task['skip_days'] ?? ''; ?>"
                                     data-effective-date="<?php echo $effectiveDate; ?>">
                                    <div class="task-header">
                                        <div class="task-title-section">
                                            <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 4px;">
                                                <h3 class="task-title">
                                                    <?php if ($task['status'] === 'completed'): ?>
                                                        <img class="completed-icon" src="../icon/completed.png" alt="Completed">
                                                    <?php endif; ?>
                                                    <?php echo escapeHtml($task['name']); ?>
                                                </h3>
                                                <?php if (isset($task['task_number'])): ?>
                                                    <span class="task-number-badge" style="font-size: 11px; color: var(--text-secondary); font-weight: 500; background: var(--bg-tertiary); padding: 2px 8px; border-radius: 6px;">
                                                        <?php echo escapeHtml($task['task_number']); ?>
                                                    </span>
                                                <?php endif; ?>
                                            </div>
                                            <div class="task-meta">
                                                <?php if ($task['status'] === 'completed'): ?>
                                                    <span class="task-meta-item">
                                                        <span class="icon"><?php echo $task['task_type'] === 'recurring' ? '🔄' : '📌'; ?></span>
                                                        <span><?php echo $typeLabel; ?></span>
                                                    </span>
                                                    <?php if (!empty($task['completion_date'])): ?>
                                                        <span class="task-meta-item">
                                                            <span class="icon">✅</span>
                                                            <span>Completed: <?php echo formatDate($task['completion_date']); ?><?php if ($completedAtDisplay): ?> at <?php echo $completedAtDisplay; ?><?php endif; ?></span>
                                                        </span>
                                                    <?php endif; ?>
                                                    <span class="task-meta-item">
                                                        <span class="icon">📅</span>
                                                        <span>Start Date: <?php echo formatDate($effectiveDate); ?><?php if (!empty($effectiveStartTime)): ?> at <?php echo date('g:i A', strtotime($effectiveStartTime)); ?><?php endif; ?></span>
                                                    </span>
                                                    <?php if ($effectiveEndDate): ?>
                                                        <span class="task-meta-item">
                                                            <span class="icon">⏰</span>
                                                            <span>End Date: <?php echo formatDate($effectiveEndDate); ?><?php if (!empty($effectiveEndTime)): ?> at <?php echo date('g:i A', strtotime($effectiveEndTime)); ?><?php endif; ?></span>
                                                        </span>
                                                    <?php endif; ?>
                                                <?php else: ?>
                                                    <span class="task-meta-item">
                                                        <span class="icon"><?php echo $task['task_type'] === 'recurring' ? '🔄' : '📌'; ?></span>
                                                        <span><?php echo $typeLabel; ?></span>
                                                    </span>
                                                    <span class="task-meta-item">
                                                        <span class="icon">📅</span>
                                                        <span>Date: <?php echo formatDate($effectiveDate); ?></span>
                                                    </span>
                                                    <?php if ($effectiveEndDate): ?>
                                                        <span class="task-meta-item">
                                                            <span class="icon">⏰</span>
                                                            <span>Due: <?php echo formatDate($effectiveEndDate); ?><?php if (!empty($effectiveEndTime)): ?> at <?php echo date('g:i A', strtotime($effectiveEndTime)); ?><?php endif; ?></span>
                                                        </span>
                                                    <?php endif; ?>
                                                    <?php if (!empty($effectiveStartTime)): ?>
                                                        <span class="task-meta-item">
                                                            <span class="icon">🕐</span>
                                                            <span>Start: <?php echo date('g:i A', strtotime($effectiveStartTime)); ?></span>
                                                        </span>
                                                    <?php endif; ?>
                                                    <span class="task-meta-item">
                                                        <span class="icon">🏆</span>
                                                        <span>QP <?php echo $task['quality_points']; ?></span>
                                                    </span>
                                                    <span class="task-meta-item">
                                                        <span class="icon">📊</span>
                                                        <span>TP <?php echo $task['task_points']; ?></span>
                                                    </span>
                                                    <?php if ($lostTaskPointsDisplay): ?>
                                                        <span class="task-meta-item negative-value">
                                                            <span class="icon">📉</span>
                                                            <span>Lost TP <?php echo $lostTaskPointsDisplay; ?></span>
                                                        </span>
                                                    <?php endif; ?>
                                                    <span class="task-priority <?php echo $task['priority']; ?>"><?php echo ucfirst($task['priority']); ?> Priority</span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        <?php if ($task['status'] === 'completed'): ?>
                                            <span class="task-status completed">Completed</span>
                                        <?php elseif ($task['status'] === 'failed'): ?>
                                            <span class="task-status failed">Failed</span>
                                        <?php elseif ($task['status'] === 'pending' && !empty($effectiveEndDate)): ?>
                                            <div class="countdown-timer" id="countdown-<?php echo $task['id']; ?>" data-end-date="<?php echo $effectiveEndDate; ?>" data-end-time="<?php echo $effectiveEndTime ?? ''; ?>">
                                                <span class="countdown-label">Remaining time:</span>
                                                <span class="countdown-value">Calculating...</span>
                                                <?php if (!empty($effectiveEndTime)): ?>
                                                    <span class="countdown-end-time">
                                                        <span class="countdown-end-icon">⏰</span>
                                                        <span class="countdown-end-text"><?php echo date('g:i A', strtotime($effectiveEndTime)); ?></span>
                                                    </span>
                                                <?php endif; ?>
                                            </div>
                                        <?php else: ?>
                                            <span class="task-status pending">Pending</span>
                                        <?php endif; ?>
                                    </div>
                                    <p class="task-description"><?php echo escapeHtml($task['description']); ?></p>
                                    <?php if ($task['status'] !== 'completed'): ?>
                                        <div class="task-footer">
                                            <div class="task-actions">
                                                <?php if ($task['status'] === 'pending'): ?>
                                                    <button class="btn btn-complete btn-small mark-complete-btn" data-task-id="<?php echo $task['id']; ?>" onclick="openMarkCompleteModal(<?php echo $task['id']; ?>)">Mark Complete</button>
                                                <?php endif; ?>
                                                <button class="btn btn-primary btn-small view-task-btn" data-task-id="<?php echo $task['id']; ?>" onclick="viewTaskDetails(<?php echo $task['id']; ?>)">View Details</button>
                                            </div>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Task Detail Modal (iOS Style) -->
    <div id="taskDetailModal" class="modal ios-modal">
        <div class="modal-content ios-modal-content">
            <div class="modal-header ios-modal-header">
                <h2 id="taskDetailTitle">Task Details</h2>
                <button class="modal-close" id="closeTaskDetailModal">&times;</button>
            </div>
            <div id="taskDetailContent" class="ios-modal-body">
                <div class="loading-spinner">
                    <div class="spinner"></div>
                    <p>Loading task details...</p>
                </div>
            </div>
            <div class="ios-modal-actions" id="taskDetailActions" style="display: none;">
                <button class="btn btn-secondary btn-small" id="taskDetailEditBtn">Edit</button>
                <button class="btn btn-danger btn-small" id="taskDetailDeleteBtn">Delete</button>
            </div>
        </div>
    </div>

    <!-- Mark Complete Modal -->
    <div id="markCompleteModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Mark Task as Complete</h2>
                <button class="modal-close" id="closeMarkCompleteModal">&times;</button>
            </div>
            <form id="markCompleteForm" method="POST" action="../complete_task.php" enctype="multipart/form-data">
                <input type="hidden" name="task_id" id="hiddenCompleteTaskId" value="">
                <?php if (isset($_GET['complete_error'])): ?>
                    <div class="alert alert-error" style="margin-bottom: 15px; padding: 10px; background: #fee; color: #c33; border-radius: 4px;">
                        <?php echo htmlspecialchars($_GET['complete_error']); ?>
                    </div>
                <?php endif; ?>
                <div class="form-group">
                    <label for="completionUrl">URL (Optional)</label>
                    <input type="url" id="completionUrl" name="completion_url" placeholder="https://example.com">
                </div>
                <div class="form-group">
                    <label for="completionText">Text (Optional)</label>
                    <textarea id="completionText" name="completion_text" rows="4" placeholder="Enter any additional notes or text"></textarea>
                </div>
                <div class="form-group">
                    <label for="completionImage">Upload Image (Optional)</label>
                    <input type="file" id="completionImage" name="completion_image" accept="image/*">
                    <small>Accepted formats: JPG, PNG, GIF</small>
                    <div id="imagePreview" class="image-preview" style="display: none;">
                        <img id="previewImg" src="" alt="Preview">
                        <button type="button" id="removeImage" class="btn btn-small btn-secondary">Remove</button>
                    </div>
                </div>
                <div class="form-group">
                    <small class="form-hint">* At least one field (URL, Text, or Image) must be filled</small>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn btn-secondary" id="cancelMarkCompleteBtn">Cancel</button>
                    <button type="submit" class="btn btn-primary">Submit</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Set timezone to Asia/Kolkata for all JavaScript date operations
        const KOLKATA_TIMEZONE = 'Asia/Kolkata';
        const KOLKATA_OFFSET = 5.5 * 60 * 60 * 1000;
        
        function getKolkataTime() {
            const now = new Date();
            const istString = now.toLocaleString('en-US', {
                timeZone: 'Asia/Kolkata',
                year: 'numeric',
                month: '2-digit',
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit',
                hour12: false
            });
            const parts = istString.match(/(\d+)\/(\d+)\/(\d+),\s+(\d+):(\d+):(\d+)/);
            if (parts) {
                const month = parts[1];
                const day = parts[2];
                const year = parts[3];
                const hour = parts[4];
                const minute = parts[5];
                const second = parts[6];
                const isoString = `${year}-${month.padStart(2, '0')}-${day.padStart(2, '0')}T${hour.padStart(2, '0')}:${minute.padStart(2, '0')}:${second.padStart(2, '0')}+05:30`;
                return new Date(isoString);
            }
            return now;
        }
        
        function createKolkataDate(dateString, timeString = null) {
            if (!timeString) {
                timeString = '23:59:59';
            }
            const timeParts = timeString.split(':');
            if (timeParts.length === 2) {
                timeString = timeString + ':00';
            }
            const isoString = `${dateString}T${timeString}+05:30`;
            return new Date(isoString);
        }
        
        const CURRENT_USER = {
            id: <?php echo $user_id; ?>,
            username: '<?php echo htmlspecialchars($username, ENT_QUOTES); ?>',
            role: '<?php echo htmlspecialchars($user_role, ENT_QUOTES); ?>'
        };
        const IS_LOGGED_IN = true;
    </script>
    <script src="../app-minimal.js"></script>
    <script>
        // Initialize countdown timers after page loads
        if (typeof initCountdownTimers === 'function') {
            initCountdownTimers();
        }
    </script>
</body>
</html>
